<?php
// File: wp-content/plugins/dynamic-seo-pages/includes/class-dpg-admin.php

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class DPG_Admin {

    /* -------------------------------------------------------------------------
     * CONSTANTS
     * ---------------------------------------------------------------------- */
    const OPTION_PAGES_ACTIVE      = 'dpg_pages_active';
    const OPTION_DELETED_TEMPLATES = 'dpg_deleted_templates';
    const OPTION_GLOBAL_SCORE      = 'dpg_global_score';

    /** @var DPG_Form_Base[] */
    private static $form_instances = [];

    /* -------------------------------------------------------------------------
     * BOOTSTRAP
     * ---------------------------------------------------------------------- */
    public static function init() {
        // Ensure WP_Filesystem is ready
        if ( ! function_exists( 'WP_Filesystem' ) ) {
            require_once ABSPATH . 'wp-admin/includes/file.php';
        }
        WP_Filesystem();

        self::load_form_classes();

        // Admin menu & assets
        add_action( 'admin_menu',            [ __CLASS__, 'add_menu' ] );
        add_action( 'admin_enqueue_scripts', [ __CLASS__, 'enqueue_admin_assets' ] );

        // Form handlers
        add_action( 'admin_post_dpg_create_template',   [ __CLASS__, 'handle_create_template' ] );
        add_action( 'admin_post_dpg_update_template',   [ __CLASS__, 'handle_update_template' ] );
        add_action( 'admin_post_dpg_delete_template',   [ __CLASS__, 'handle_delete_template' ] );
        add_action( 'admin_post_dpg_restore_template',  [ __CLASS__, 'handle_restore_template' ] );
        add_action( 'admin_post_dpg_permanent_delete',  [ __CLASS__, 'handle_permanent_delete' ] );
        add_action( 'admin_post_dpg_save_sitemap',      [ __CLASS__, 'handle_save_sitemap' ] );
        add_action( 'admin_post_dpg_update_page_score', [ __CLASS__, 'handle_update_page_score' ] );
        add_action( 'admin_post_dpg_export_scores',     [ __CLASS__, 'handle_export_page_scores' ] );
        add_action( 'admin_post_dpg_import_scores',     [ __CLASS__, 'handle_import_page_scores' ] );
        add_action( 'admin_post_dpg_clear_deleted_templates', [ __CLASS__, 'clear_all_deleted_templates' ] );
     

        // AJAX hooks
        add_action( 'wp_ajax_dpg_seo_analysis',        [ __CLASS__, 'handle_seo_analysis' ] );
        add_action( 'wp_ajax_nopriv_dpg_seo_analysis', [ __CLASS__, 'handle_seo_analysis' ] );
        add_action( 'wp_ajax_dpg_auto_save_template',  [ __CLASS__, 'handle_auto_save' ] );
        add_action( 'wp_ajax_dpg_quick_save_template', [ __CLASS__, 'handle_quick_save' ] );
        add_action( 'wp_ajax_dpg_clear_cache',         [ __CLASS__, 'handle_clear_cache' ] );

        // Front-end redirect for deleted templates
        add_action( 'template_redirect', [ __CLASS__, 'handle_deleted_template_redirects' ], 1 );
        add_action( 'admin_init',        [ __CLASS__, 'handle_refresh_rules' ] );

        self::init_forms();
    }

    /* -------------------------------------------------------------------------
     * LOAD FORM CLASSES
     * ---------------------------------------------------------------------- */
    private static function load_form_classes() {
        global $wp_filesystem;
        $form_files = [
            'class-dpg-form-base.php',
            'class-dpg-form-create.php',
            'class-dpg-form-edit.php',
            'class-dpg-form-settings.php',
        ];
        foreach ( $form_files as $file ) {
            $path = DPG_PLUGIN_DIR . 'includes/forms/' . $file;
            if ( $wp_filesystem->exists( $path ) ) {
                require_once $path;
            }
        }
    }

    /* -------------------------------------------------------------------------
     * INITIALIZE FORMS
     * ---------------------------------------------------------------------- */
    private static function init_forms() {
        $form_config = [
            'seo_analyzer' => [
                'enabled' => true,
                'auto_run' => true,
                'debounce_delay' => 1500
            ],
            'validation' => [
                'min_content_length' => 10,
                'max_content_length' => 10000,
                'title_length' => [ 'min' => 30, 'max' => 60 ],
                'description_length' => [ 'min' => 120, 'max' => 160 ]
            ],
            'debug_mode' => get_option( 'dpg_debug_mode', false )
        ];

        if ( class_exists( 'DPG_Form_Create' ) ) {
            self::$form_instances['create'] = new DPG_Form_Create( $form_config );
        }
        if ( class_exists( 'DPG_Form_Edit' ) ) {
            self::$form_instances['edit_class']  = 'DPG_Form_Edit';
            self::$form_instances['edit_config'] = $form_config;
        }
        if ( class_exists( 'DPG_Form_Settings' ) ) {
            self::$form_instances['settings'] = new DPG_Form_Settings( $form_config );
        }
    }

    /* -------------------------------------------------------------------------
     * ADMIN ASSETS
     * ---------------------------------------------------------------------- */
    public static function enqueue_admin_assets( $hook_suffix ) {
        if ( $hook_suffix !== 'toplevel_page_dpg-dashboard' ) {
            return;
        }

        $base_url = plugin_dir_url( DPG_PLUGIN_FILE ) . 'includes/forms/assets/';
        $ver      = DPG_VERSION;

        // Styles
        wp_enqueue_style( 'dpg-admin-forms', $base_url . 'css/admin-forms.css', [], $ver );
        wp_enqueue_style( 'dpg-components', $base_url . 'css/components.css', [ 'dpg-admin-forms' ], $ver );
        wp_enqueue_style( 'dpg-alerts',     $base_url . 'css/alerts.css',     [ 'dpg-admin-forms' ], $ver );

        // Scripts
        wp_enqueue_script( 'dpg-form-validation',  $base_url . 'js/form-validation.js',  [ 'jquery' ], $ver, true );
        wp_enqueue_script( 'dpg-unified-seo',      $base_url . 'js/seo-analyzer.js',     [ 'jquery', 'wp-i18n' ], $ver, true );
        wp_enqueue_script( 'dpg-components-js',    $base_url . 'js/components.js',       [ 'jquery', 'dpg-form-validation' ], $ver, true );

        wp_localize_script( 'dpg-unified-seo', 'dpgForm', [
            'ajaxUrl'   => admin_url( 'admin-ajax.php' ),
            'nonce'     => wp_create_nonce( 'dpg_form_nonce' ),
            'homeUrl'   => trailingslashit( home_url() ),
            'debug'     => defined( 'WP_DEBUG' ) && WP_DEBUG,
            'strings'   => [ /* ... validation & UI strings ... */ ],
            'seoAnalyzer' => [
                'options'  => [ 'enabled' => true, 'autoRun' => true, 'debounceDelay' => 1500 ],
                'settings' => [ 'minContentLength' => 10, 'maxContentLength' => 10000, 'titleLength' => [30,60], 'descriptionLength'=>[120,160] ],
                'restApi'  => [ 'available' => function_exists('rest_get_url_prefix'), 'endpoints'=>[] ]
            ],
            'freeVersion' => [
                'maxTemplates'     => self::get_max_code(),
                'maxItems'         => self::listvars(),
                'remainingTemplates'=> self::get_remaining_code_slots(),
            ]
        ] );

        // CodeMirror support
        wp_enqueue_code_editor( [ 'type' => 'text/html' ] );
        wp_enqueue_script( 'wp-theme-plugin-editor' );
        wp_enqueue_style( 'wp-codemirror' );
    }

    /* -------------------------------------------------------------------------
     * ADMIN MENU
     * ---------------------------------------------------------------------- */
    public static function add_menu() {
        add_menu_page(
            __( 'Dynamic SEO', 'dpg' ),
            __( 'Dynamic SEO', 'dpg' ),
            'manage_options',
            'dpg-dashboard',
            [ __CLASS__, 'render_dashboard' ],
            'dashicons-chart-line',
            60
        );
        // Remove auto-submenu
        remove_submenu_page( 'dpg-dashboard', 'dpg-dashboard' );
    }

    /* -------------------------------------------------------------------------
     * SHOW ADMIN NOTICES (REPLACED)
     * ---------------------------------------------------------------------- */
private static function show_admin_notices() {
    // 1) item‐limit exceeded
    if ( isset( $_GET['dpg_items_error'] ) ) {
        printf(
            '<div class="notice notice-error is-dismissible"><p>%s</p></div>',
            esc_html__( 'Sorry, this plugin only allows you to create 50 items.', 'dpg' )
        );
    }

    // 2) all your existing notices, plus new restore/delete errors
    $notices = [
        'created'                  => [ 'success', __( 'Template created successfully.',                'dpg' ) ],
        'updated'                  => [ 'success', __( 'Template updated successfully.',                'dpg' ) ],
        'deleted'                  => [ 'warning', __( 'Template deleted. Restore from Deleted tab.',   'dpg' ) ],
        'permanent_deleted'        => [ 'warning', __( 'Template permanently deleted.',                'dpg' ) ],
        'deleted_cleared'          => [ 'success', __( 'All deleted templates cleared.',               'dpg' ) ],
        'license_saved'            => [ 'success', __( 'Settings saved successfully.',                 'dpg' ) ],
        'rules_refreshed'          => [ 'success', __( 'URL rules refreshed successfully.',           'dpg' ) ],
        'sitemap_saved'            => [ 'success', __( 'Sitemap settings saved successfully.',        'dpg' ) ],
        'scores_imported'          => [ 'success', __( 'Scores imported successfully.',               'dpg' ) ],
        'cleanup_legacy'           => [ 'success', __( 'Legacy data cleaned up.',                     'dpg' ) ],
        'restored'                 => [ 'success', __( 'Template restored successfully.',             'dpg' ) ],

        // new errors:
        'restore_error'            => [ 'error',   __( 'You already have the maximum number of active templates. Delete one before restoring.', 'dpg' ) ],
        'restore_error_notfound'   => [ 'error',   __( 'Could not find that deleted template to restore.',             'dpg' ) ],
        'delete_error_notfound'    => [ 'error',   __( 'Could not find that template to delete.',                       'dpg' ) ],
    ];

    foreach ( $notices as $key => list( $type, $message ) ) {
        if ( isset( $_GET[ $key ] ) ) {
            printf(
                '<div class="notice notice-%1$s is-dismissible"><p>%2$s</p></div>',
                esc_attr( $type ),
                esc_html( $message )
            );
        }
    }
}


    

    /* -------------------------------------------------------------------------
     * RENDER DASHBOARD
     * ---------------------------------------------------------------------- */
    public static function render_dashboard() {
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_die( __( 'Permission denied', 'dpg' ) );
        }

        self::show_admin_notices();
        $tab = self::get_current_tab();

        echo '<div class="wrap dpg-admin-wrap">';
        echo '<h1>' . esc_html__( 'Dynamic SEO', 'dpg' ) . '</h1>';
        self::render_tab_navigation( $tab );
        echo '<div class="dpg-tab-content-wrapper">';

        switch ( $tab ) {
            case 'guide':
                self::render_guide_tab();
                break;
            case 'create':
                self::render_create_tab();
                break;
            case 'edit':
                self::render_edit_tab();
                break;
            case 'deleted':
                self::render_deleted_tab();
                break;
            case 'sitemap':
                self::render_sitemap_tab();
                break;
            case 'settings':
                self::render_compliant_settings_section();
                break;
            default:
                self::render_templates_tab();
                break;
        }

        echo '</div></div>';
    }

    /* -------------------------------------------------------------------------
     * GET CURRENT TAB
     * ---------------------------------------------------------------------- */
    private static function get_current_tab() {
        return isset( $_GET['tab'] ) ? sanitize_key( wp_unslash( $_GET['tab'] ) ) : '';
    }

    /* -------------------------------------------------------------------------
     * RENDER TAB NAVIGATION
     * ---------------------------------------------------------------------- */
    private static function render_tab_navigation( $active_tab ) {
        $tabs = [
            'guide'    => __( 'How To Guide',     'dpg' ),
            ''         => __( 'Templates',        'dpg' ),
            'create'   => __( 'Create Template',  'dpg' ),
            'deleted'  => __( 'Deleted Templates','dpg' ),
            'sitemap'  => __( 'Sitemap',          'dpg' ),
            'settings' => __( 'Settings',         'dpg' ),
        ];

        echo '<nav class="nav-tab-wrapper">';
        foreach ( $tabs as $key => $label ) {
            $class = ( $active_tab === $key ) ? ' nav-tab-active' : '';
            $url   = $key === ''
                ? admin_url( 'admin.php?page=dpg-dashboard' )
                : add_query_arg( 'tab', $key, admin_url( 'admin.php?page=dpg-dashboard' ) );

            printf(
                '<a href="%1$s" class="nav-tab%2$s">%3$s</a>',
                esc_url( $url ),
                esc_attr( $class ),
                esc_html( $label )
            );
        }
        echo '</nav>';
    }

/**
 * Renders the “Create Template” tab.
 */
private static function render_create_tab() {
    // If the user has reached their template limit, show a notice and return early.
    if ( ! self::can_create_code() ) {
        ?>
        <div class="dpg-tab-content-wrapper">
            <div class="dpg-form-container">
                <div class="dpg-form-header">
                    <h2 class="dpg-form-title"><?php esc_html_e( 'Create New Template', 'dpg' ); ?></h2>
                    <p class="dpg-form-description">
                        <?php esc_html_e( 'Sorry, this plugin only allows 1 template.', 'dpg' ); ?>
                    </p>
                </div>
                <div class="dpg-form-section">
                    <div class="dpg-section-content">
                        <div class="dpg-notice dpg-notice-info" style="background:rgba(99,102,241,0.05);border:1px solid rgba(99,102,241,0.2);border-radius:8px;padding:20px;text-align:center;">
                            <h3 style="margin:0 0 12px 0;font-size:18px;color:var(--dpg-primary);font-weight:600;">
                                📋 <?php esc_html_e( 'Sorry, this plugin only allows 1 template.', 'dpg' ); ?>
                            </h3>
                            <p style="margin:0;font-size:14px;color:var(--dpg-gray-600);">
                                <?php esc_html_e( 'Delete an existing template to create a new one.', 'dpg' ); ?>
                            </p>
                            <div style="margin-top:20px;display:flex;gap:12px;justify-content:center;">
                                <a href="<?php echo esc_url( admin_url( 'admin.php?page=dpg-dashboard' ) ); ?>" class="dpg-button dpg-button-primary">
                                    📋 <?php esc_html_e( 'Manage Templates', 'dpg' ); ?>
                                </a>
                                <a href="<?php echo esc_url( admin_url( 'admin.php?page=dpg-dashboard&tab=guide' ) ); ?>" class="dpg-button dpg-button-secondary">
                                    📖 <?php esc_html_e( 'View Guide', 'dpg' ); ?>
                                </a>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        <?php
        return;
    }

    // Otherwise, render the actual creation form
    ?>
    <div class="dpg-tab-content-wrapper">
        <?php
        if ( isset( self::$form_instances['create'] ) ) {
            self::$form_instances['create']->render();
        } else {
            // Fallback if for some reason the form class isn’t available
            ?>
            <div class="dpg-form-container">
                <div class="dpg-form-header">
                    <h2 class="dpg-form-title"><?php esc_html_e( 'Create New Template', 'dpg' ); ?></h2>
                    <p class="dpg-form-description"><?php esc_html_e( 'Generate multiple SEO-optimized pages automatically.', 'dpg' ); ?></p>
                </div>
                <div class="dpg-form-section">
                    <div class="dpg-section-header">
                        <h3 class="dpg-section-title" data-section="form-error"><?php esc_html_e( 'Form Error', 'dpg' ); ?></h3>
                    </div>
                    <div class="dpg-section-content">
                        <div class="dpg-notice dpg-notice-error" style="background:rgba(239,68,68,0.05);border:1px solid rgba(239,68,68,0.1);border-radius:8px;padding:20px;text-align:center;">
                            <p style="margin:0;font-size:16px;color:var(--dpg-error);font-weight:600;">
                                ⚠️ <?php esc_html_e( 'Create form not available.', 'dpg' ); ?>
                            </p>
                        </div>
                    </div>
                </div>
            </div>
            <?php
        }
        ?>
    </div>
    <?php
}


 private static function render_edit_tab() {
        $ctx = sanitize_text_field( wp_unslash( $_GET['ctx'] ?? '' ) );
        
        ?>
        <div class="dpg-tab-content-wrapper">
            <?php
            if ( $ctx && ! self::is_master_template( $ctx ) && isset( self::$form_instances['edit_class'] ) ) {
                $edit_class = self::$form_instances['edit_class'];
                $edit_config = self::$form_instances['edit_config'] ?? [];
                $form = new $edit_class( $ctx, $edit_config );
                $form->render();
            } else {
                ?>
                <div class="dpg-form-container">
                    <div class="dpg-form-header">
                        <h2 class="dpg-form-title"><?php esc_html_e( 'Edit Template', 'dpg' ); ?></h2>
                        <p class="dpg-form-description"><?php esc_html_e( 'Template not found or cannot be edited.', 'dpg' ); ?></p>
                    </div>
                    <div class="dpg-form-section">
                        <div class="dpg-section-header">
                            <h3 class="dpg-section-title" data-section="form-error">Template Error</h3>
                        </div>
                        <div class="dpg-section-content">
                            <div class="dpg-notice dpg-notice-error" style="background:rgba(239,68,68,0.05);border:1px solid rgba(239,68,68,0.1);border-radius:8px;padding:20px;text-align:center;">
                                <p style="margin:0;font-size:16px;color:var(--dpg-error);font-weight:600;">
                                    ⚠️ <?php esc_html_e( 'Template not found or cannot be edited.', 'dpg' ); ?>
                                </p>
                                <a href="<?php echo esc_url( admin_url( 'admin.php?page=dpg-dashboard' ) ); ?>" class="dpg-button dpg-button-primary" style="margin-top:16px;">
                                    📋 <?php esc_html_e( 'Back to Templates', 'dpg' ); ?>
                                </a>
                            </div>
                        </div>
                    </div>
                </div>
                <?php
            }
            ?>
        </div>
        <?php
    }


/* -------------------------------------------------------------------------
 * TAB: Deleted Templates
 * ---------------------------------------------------------------------- */
private static function render_deleted_tab() {
    $deleted = get_option( self::OPTION_DELETED_TEMPLATES, [] );
    ?>
    <div class="dpg-tab-content-wrapper">
      <div class="dpg-form-container">
        <div class="dpg-form-header">
          <h2 class="dpg-form-title"><?php esc_html_e( 'Deleted Templates', 'dpg' ); ?></h2>
          <p class="dpg-form-description">
            <?php esc_html_e( 'Templates you’ve removed—here you can restore or permanently delete them.', 'dpg' ); ?>
          </p>
        </div>
        <div class="dpg-form-section">
          <div class="dpg-section-header">
            <h3 class="dpg-section-title" data-section="deleted-templates-list">
              <?php esc_html_e( 'Deleted Templates List', 'dpg' ); ?>
            </h3>
          </div>
          <div class="dpg-section-content">
            <?php if ( empty( $deleted ) ) : ?>
              <div class="dpg-notice dpg-notice-info">
                <p>🗑️ <?php esc_html_e( 'No deleted templates found.', 'dpg' ); ?></p>
              </div>
            <?php else: ?>
              <?php self::render_deleted_templates_table( $deleted ); ?>
              <form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>" style="margin-top:16px;">
                <?php wp_nonce_field( 'dpg_clear_deleted_templates' ); ?>
                <input type="hidden" name="action" value="dpg_clear_deleted_templates">
                <button class="button button-secondary" onclick="return confirm('<?php echo esc_js( __( 'Clear all deleted templates? This cannot be undone.', 'dpg' ) ); ?>');">
                  <?php esc_html_e( 'Clear All Deleted', 'dpg' ); ?>
                </button>
              </form>
            <?php endif; ?>
          </div>
        </div>
      </div>
    </div>
    <?php
}

private static function render_deleted_templates_table( $deleted ) {
    ?>
    <table class="widefat fixed striped dpg-deleted-templates-table">
      <thead>
        <tr>
          <th><?php esc_html_e( 'Template Name', 'dpg' ); ?></th>
          <th><?php esc_html_e( 'Deleted Date',  'dpg' ); ?></th>
          <th><?php esc_html_e( 'URLs Affected', 'dpg' ); ?></th>
          <th><?php esc_html_e( 'Actions',       'dpg' ); ?></th>
        </tr>
      </thead>
      <tbody>
        <?php foreach ( $deleted as $ctx => $info ) :
          $name        = $info['data']['name']        ?? $ctx;
          $date        = $info['deleted_date']        ?? '';
          $urls        = $info['urls']                ?? [];
          $restore_url = wp_nonce_url( add_query_arg(
              [ 'action' => 'dpg_restore_template', 'ctx' => $ctx ],
              admin_url( 'admin-post.php' )
            ), 'dpg_restore_template' );
          $perm_url    = wp_nonce_url( add_query_arg(
              [ 'action' => 'dpg_permanent_delete', 'ctx' => $ctx ],
              admin_url( 'admin-post.php' )
            ), 'dpg_permanent_delete' );
        ?>
        <tr>
          <td>
            <strong><?php echo esc_html( $name ); ?></strong>
            <div class="row-actions"><span><?php printf( __( 'ID: %s', 'dpg' ), esc_html( $ctx ) ); ?></span></div>
          </td>
          <td><time><?php echo esc_html( date_i18n( 'Y-m-d H:i:s', strtotime( $date ) ) ); ?></time></td>
          <td>
            <?php if ( $urls ) : ?>
              <details>
                <summary><?php printf( _n( '%d URL', '%d URLs', count( $urls ), 'dpg' ), count( $urls ) ); ?></summary>
                <ul>
                  <?php foreach ( array_slice( $urls, 0, 5 ) as $u ) : ?>
                    <li><code><?php echo esc_html( $u ); ?></code></li>
                  <?php endforeach; ?>
                  <?php if ( count( $urls ) > 5 ) : ?>
                    <li><em><?php printf( __( '...and %d more', 'dpg' ), count( $urls ) - 5 ); ?></em></li>
                  <?php endif; ?>
                </ul>
              </details>
            <?php else: ?>
              <span><?php esc_html_e( 'No URLs', 'dpg' ); ?></span>
            <?php endif; ?>
          </td>
          <td>
            <a href="<?php echo esc_url( $restore_url ); ?>"
               class="button button-primary button-small"
               onclick="return confirm('<?php echo esc_js( __( 'Restore this template? URLs will go live again.', 'dpg' ) ); ?>');">
              <?php esc_html_e( 'Restore', 'dpg' ); ?>
            </a>
            <a href="<?php echo esc_url( $perm_url ); ?>"
               class="button button-link-delete button-small"
               onclick="return confirm('<?php echo esc_js( __( 'Delete permanently? This cannot be undone.', 'dpg' ) ); ?>');">
              <?php esc_html_e( 'Delete Permanently', 'dpg' ); ?>
            </a>
          </td>
        </tr>
        <?php endforeach; ?>
      </tbody>
    </table>
    <?php
}




    /**
 * Front-end hook: redirect any deleted-template URLs to something sensible.
 */
public static function handle_deleted_template_redirects() {
    // Only on the front end
    if ( is_admin() ) {
        return;
    }

    // Grab the request path, no leading/trailing slashes
    $request_uri = $_SERVER['REQUEST_URI'] ?? '';
    $parsed      = wp_parse_url( $request_uri );
    $path        = trim( $parsed['path'] ?? '', '/' );

    // If WP lives in a subdirectory, strip that base off
    if ( function_exists( 'dpg_get_wp_base_path_compliant' ) ) {
        $base = trim( dpg_get_wp_base_path_compliant(), '/' );
    } else {
        $base = trim( parse_url( home_url(), PHP_URL_PATH ), '/' );
    }
    if ( $base && strpos( $path, $base . '/' ) === 0 ) {
        $path = substr( $path, strlen( $base . '/' ) );
    }

    // Bail if path is empty or not our dynamic pattern
    if ( empty( $path ) || ! self::is_potential_dynamic_url( $path ) ) {
        return;
    }

    // Do we have any deleted templates?
    $deleted = get_option( self::OPTION_DELETED_TEMPLATES, [] );
    if ( ! $deleted ) {
        return;
    }

    // Build the full URL
    $current_url = home_url( '/' . $path . '/' );

    // If it's in our deleted list, redirect
    foreach ( $deleted as $ctx_id => $info ) {
        $urls = $info['urls'] ?? [];
        if ( in_array( $current_url, $urls, true ) ) {
            self::redirect_deleted_url( $ctx_id, $current_url, $info );
            return;
        }
    }
}

/**
 * Helper: detect whether a request-path looks like one of our dynamic pages.
 */
private static function is_potential_dynamic_url( $path ) {
    // Skip obvious WP/core assets
    $skip = [
        'wp-admin','wp-content','wp-includes','wp-json',
        '.php','xmlrpc.php','sitemap','robots.txt','favicon.ico'
    ];
    foreach ( $skip as $frag ) {
        if ( strpos( $path, $frag ) !== false ) {
            return false;
        }
    }
    // service-area: foo-bar
    if ( preg_match( '/^[a-z0-9-]+-[a-z0-9-]+$/', $path ) ) {
        return true;
    }
    // keyword-only: foo
    if ( preg_match( '/^[a-z0-9-]+$/', $path ) ) {
        return true;
    }
    return false;
}


/**
 * Fires a 301 to the best replacement for a deleted URL.
 */
private static function redirect_deleted_url( $ctx_id, $current_url, $info ) {
    // Determine where to send them
    $dest = self::get_redirect_destination_for_deleted_template( $ctx_id, $info );

    // Log for debugging if enabled
    if ( get_option( 'dpg_debug_mode', false ) ) {
        error_log( sprintf(
            '[DPG] Redirecting deleted URL %s (template %s) → %s',
            $current_url,
            $info['data']['name'] ?? $ctx_id,
            $dest
        ) );
    }

    // Use WP redirect (sanitized) rather than raw headers
    wp_safe_redirect( esc_url_raw( $dest ), 301 );
    exit;
}



/**
 * Choose a redirect destination for a deleted template.
 */
private static function get_redirect_destination_for_deleted_template( $ctx_id, $info ) {
    // 1) is there another template that can stand in?
    $replacement = self::find_replacement_template_url( $ctx_id, $info );
    if ( $replacement ) {
        return $replacement;
    }

    // 2) if this was a service-area, try a “Services” page
    if ( substr( $ctx_id, -5 ) === '-area' ) {
        $svc = self::find_services_page();
        if ( $svc ) {
            return $svc;
        }
    }

    // 3) try a relevant category/tag page
    $cat = self::find_relevant_category( $info['data']['name'] ?? '' );
    if ( $cat ) {
        return $cat;
    }

    // 4) fallback to homepage
    return home_url( '/' );
}

/**
 * Look for another template of the same type to redirect to.
 */
private static function find_replacement_template_url( $ctx_id, $info ) {
    $templates = self::get_user_codes();
    $is_area   = substr( $ctx_id, -5 ) === '-area';

    foreach ( $templates as $file ) {
        $other = basename( $file, '.json' );
        if ( $other === $ctx_id ) {
            continue;
        }
        $this_is_area = substr( $other, -5 ) === '-area';
        if ( $this_is_area !== $is_area ) {
            continue;
        }
        $data = json_decode( file_get_contents( $file ), true );
        $items = $data['items'] ?? [];
        if ( ! empty( $items ) ) {
            $first = is_array( $items[0] ) ? $items[0]['name'] : $items[0];
            $slug  = sanitize_title( $first );
            if ( $this_is_area ) {
                $base = substr( $other, 0, -5 );
                return home_url( "/{$base}-{$slug}/" );
            } else {
                return home_url( "/{$slug}/" );
            }
        }
    }
    return null;
}

/**
 * Try to find a published “Services” page on the site.
 */
private static function find_services_page() {
    $slugs = [ 'services','our-services','what-we-do','solutions' ];
    foreach ( $slugs as $s ) {
        $p = get_page_by_path( $s );
        if ( $p && $p->post_status === 'publish' ) {
            return get_permalink( $p );
        }
    }
    // fallback: any page with “service” in the title
    $pages = get_pages([
        'post_status' => 'publish',
        'number'      => 50,
    ]);
    foreach ( $pages as $p ) {
        if ( stripos( $p->post_title, 'service' ) !== false ) {
            return get_permalink( $p );
        }
    }
    return null;
}

/**
 * Try to find a category whose name or slug matches a keyword from the template name.
 */
private static function find_relevant_category( $template_name ) {
    if ( ! $template_name ) {
        return null;
    }
    $words = array_filter(
        explode( ' ', strtolower( $template_name ) ),
        fn( $w ) => strlen( $w ) > 3
    );
    if ( empty( $words ) ) {
        return null;
    }
    $cats = get_categories([ 'number' => 50 ]);
    foreach ( $cats as $cat ) {
        $name = strtolower( $cat->name );
        $slug = strtolower( $cat->slug );
        foreach ( $words as $w ) {
            if ( strpos( $name, $w ) !== false || strpos( $slug, $w ) !== false ) {
                return get_category_link( $cat );
            }
        }
    }
    return null;
}


/**
 * Export all per-URL score overrides as a CSV.
 */
public static function handle_export_page_scores() {
    if ( ! current_user_can( 'manage_options' ) ) {
        wp_die( __( 'Permission denied', 'dpg' ) );
    }
    check_admin_referer( 'dpg_export_nonce' );

    header( 'Content-Type: text/csv; charset=utf-8' );
    header( 'Content-Disposition: attachment; filename=dpg-page-scores.csv' );
    $out = fopen( 'php://output', 'w' );
    fputcsv( $out, [ 'template_id','item_index','url','score_override' ] );

    foreach ( glob( DPG_DATA_DIR . '*.json' ) as $file ) {
        $ctx  = basename( $file, '.json' );
        $data = json_decode( file_get_contents( $file ), true );
        if ( empty( $data['items'] ) ) {
            continue;
        }
        foreach ( $data['items'] as $i => $item ) {
            $score = is_array( $item ) && isset( $item['score'] ) ? $item['score'] : '';
            $name  = is_array( $item ) ? $item['name'] : $item;
            $slug  = sanitize_title( $name );
            $url   = ( substr( $ctx, -5 ) === '-area' )
                ? home_url( '/' . substr( $ctx, 0, -5 ) . '-' . $slug . '/' )
                : home_url( '/' . $slug . '/' );
            fputcsv( $out, [ $ctx, $i, $url, $score ] );
        }
    }

    fclose( $out );
    exit;
}

/**
 * Import per-URL score overrides from a user-uploaded CSV.
 */
public static function handle_import_page_scores() {
    if ( ! current_user_can( 'manage_options' ) ) {
        wp_die( __( 'Permission denied', 'dpg' ) );
    }
    check_admin_referer( 'dpg_import_nonce' );

    if ( empty( $_FILES['csv']['tmp_name'] ) || ! is_readable( $_FILES['csv']['tmp_name'] ) ) {
        wp_die( __( 'No CSV provided or unreadable.', 'dpg' ) );
    }

    $h = fopen( $_FILES['csv']['tmp_name'], 'r' );
    fgetcsv( $h ); // skip header
    while ( $row = fgetcsv( $h ) ) {
        list( $ctx, $idx, /* $url */, $score ) = $row;
        $json = DPG_DATA_DIR . "{$ctx}.json";
        if ( ! file_exists( $json ) ) {
            continue;
        }
        $data = json_decode( file_get_contents( $json ), true );
        if ( ! isset( $data['items'][ $idx ] ) ) {
            continue;
        }
        if ( is_array( $data['items'][ $idx ] ) ) {
            $data['items'][ $idx ]['score'] = floatval( $score );
        } else {
            $data['items'][ $idx ] = [
                'name'  => $data['items'][ $idx ],
                'score' => floatval( $score ),
            ];
        }
        file_put_contents( $json, wp_json_encode( $data, JSON_PRETTY_PRINT|JSON_UNESCAPED_UNICODE ) );
    }
    fclose( $h );

    flush_rewrite_rules();
    wp_safe_redirect( add_query_arg( 'scores_imported','1', admin_url( 'admin.php?page=dpg-dashboard&tab=sitemap' ) ) );
    exit;
}

/**
 * Save a single page’s score override (from the Sitemap tab).
 */
public static function handle_update_page_score() {
    if ( ! current_user_can( 'manage_options' ) ) {
        wp_die( __( 'Permission denied', 'dpg' ) );
    }
    check_admin_referer( 'dpg_update_page_score_nonce' );

    $ctx   = sanitize_text_field( $_POST['ctx_id']     ?? '' );
    $idx   = intval(            $_POST['item_index'] ?? 0 );
    $score = floatval(          $_POST['item_score'] ?? 0 );

    $json = DPG_DATA_DIR . "{$ctx}.json";
    if ( ! file_exists( $json ) ) {
        wp_die( __( 'Template data not found.', 'dpg' ) );
    }

    $data = json_decode( file_get_contents( $json ), true );
    if ( ! isset( $data['items'][ $idx ] ) ) {
        wp_die( __( 'Page item not found.', 'dpg' ) );
    }

    if ( is_array( $data['items'][ $idx ] ) ) {
        $data['items'][ $idx ]['score'] = $score;
    } else {
        $data['items'][ $idx ] = [
            'name'  => $data['items'][ $idx ],
            'score' => $score,
        ];
    }

    file_put_contents( $json, wp_json_encode( $data, JSON_PRETTY_PRINT|JSON_UNESCAPED_UNICODE ) );
    flush_rewrite_rules();
    wp_safe_redirect( add_query_arg( 'sitemap_saved','1', admin_url( 'admin.php?page=dpg-dashboard&tab=sitemap' ) ) );
    exit;
}


    /* -------------------------------------------------------------------------
     * RENDER SETTINGS (NEW COMPLIANT)
     * ---------------------------------------------------------------------- */
private static function render_compliant_settings_section() {
    ?>
    <div class="notice notice-info is-dismissible" style="position: relative; padding: 20px; margin-bottom: 20px; background: #f0f9ff; border-left: 4px solid #2196F3;">
        <p style="margin: 0 0 10px; font-weight: bold;">
            <?php esc_html_e( 'You’re currently using the Free version. Unlock advanced features by upgrading to Pro!', 'dpg' ); ?>
        </p>
        <a href="https://dynamicseopro.com/" target="_blank" class="button button-primary">
            <?php esc_html_e( 'Upgrade to Pro', 'dpg' ); ?>
        </a>
    </div>

    <?php
    if ( isset( self::$form_instances['settings'] ) ) {
        self::$form_instances['settings']->render();
    } else {
        ?>
        <form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>">
            <?php wp_nonce_field( 'dpg_save_settings' ); ?>
            <input type="hidden" name="action" value="dpg_save_settings">
            <p><?php esc_html_e( 'No settings form available.', 'dpg' ); ?></p>
            <button type="submit" class="button"><?php esc_html_e( 'Save Settings', 'dpg' ); ?></button>
        </form>
        <?php
    }
}




/**
 * -------------------------------------------------------------------------
 * TAB: Templates List
 * -------------------------------------------------------------------------
 */
private static function render_templates_tab() {
    $user_templates = self::get_user_codes();
    $debug_mode     = get_option( 'dpg_debug_mode', false );
    ?>
    <div class="dpg-tab-content-wrapper">
        <div class="dpg-form-container">
            <div class="dpg-form-header">
                <h2 class="dpg-form-title">
                    <?php esc_html_e( 'Templates Overview', 'dpg' ); ?>
                </h2>
                <p class="dpg-form-description">
                    <?php esc_html_e( 'Manage your dynamic SEO templates. Each template generates multiple pages based on your items list.', 'dpg' ); ?>
                </p>
            </div>

            <div class="dpg-form-section">
                <div class="dpg-section-header">
                    <h3 class="dpg-section-title" data-section="your-templates">
                        <?php esc_html_e( 'Your Templates', 'dpg' ); ?>
                    </h3>
                </div>
                <div class="dpg-section-content">
                    <?php if ( empty( $user_templates ) ) : ?>
                        <div class="dpg-notice dpg-notice-info">
                            <p>
                                📝 <?php esc_html_e( 'No templates found. Create your first template to get started.', 'dpg' ); ?>
                            </p>
                            <a href="<?php echo esc_url( admin_url( 'admin.php?page=dpg-dashboard&tab=create' ) ); ?>"
                               class="button button-primary">
                                <?php esc_html_e( 'Create Your First Template', 'dpg' ); ?>
                            </a>
                        </div>
                    <?php else : ?>
                        <?php
                        // this is the seven-column table we just re-defined above
                        self::render_templates_table( $user_templates, $debug_mode );
                        ?>
                    <?php endif; ?>
                </div>
            </div>
        </div>
    </div>
    <?php
}


/* -------------------------------------------------------------------------
 * RENDER TEMPLATES TABLE
 * ---------------------------------------------------------------------- */
private static function render_templates_table( $user_templates, $debug_mode ) { ?>

    <div style="overflow-x:auto;border:1px solid var(--dpg-gray-200);
                border-radius:8px;box-shadow:var(--dpg-shadow-sm);">
        <table class="widefat fixed striped dpg-templates-table" style="margin:0;border:none;min-width:100%;table-layout:fixed;">
            <colgroup>
                <col style="width: 20%;">  <!-- Template Name -->
                <col style="width: 10%;">  <!-- Type -->
                <col style="width: 15%;">  <!-- URL Pattern -->
                <col style="width: 8%;">   <!-- # Items -->
                <col style="width: 10%;">  <!-- SEO Score -->
                <col style="width: 20%;">  <!-- Example URL -->
                <col style="width: 17%;">  <!-- Actions -->
            </colgroup>
            <thead style="background:var(--dpg-gray-50);">
            <tr>
                <th style="padding:12px 8px;border-bottom:2px solid var(--dpg-gray-200);font-size:13px;">
                    <?php esc_html_e( 'Template Name', 'dpg' ); ?>
                </th>
                <th style="padding:12px 8px;border-bottom:2px solid var(--dpg-gray-200);font-size:13px;">
                    <?php esc_html_e( 'Type', 'dpg' ); ?>
                </th>
                <th style="padding:12px 8px;border-bottom:2px solid var(--dpg-gray-200);font-size:13px;">
                    <?php esc_html_e( 'URL Pattern', 'dpg' ); ?>
                </th>
                <th style="padding:12px 8px;border-bottom:2px solid var(--dpg-gray-200);font-size:13px;text-align:center;">
                    <?php esc_html_e( '# Items', 'dpg' ); ?>
                </th>
                <th style="padding:12px 8px;border-bottom:2px solid var(--dpg-gray-200);font-size:13px;text-align:center;">
                    <?php esc_html_e( 'SEO Score', 'dpg' ); ?>
                </th>
                <th style="padding:12px 8px;border-bottom:2px solid var(--dpg-gray-200);font-size:13px;">
                    <?php esc_html_e( 'Example URL', 'dpg' ); ?>
                </th>
                <th style="padding:12px 8px;border-bottom:2px solid var(--dpg-gray-200);font-size:13px;">
                    <?php esc_html_e( 'Actions', 'dpg' ); ?>
                </th>
            </tr>
            </thead>
            <tbody>
            <?php foreach ( $user_templates as $file ) :

                $ctx_id = basename( $file, '.json' );

                // skip system/master templates
                if ( self::is_master_template( $ctx_id ) ) {
                    continue;
                }

                $data = json_decode( file_get_contents( $file ), true );
                if ( ! is_array( $data ) ) {
                    continue;
                }

                $name    = $data['name'] ?? $ctx_id;
                $is_area = substr( $ctx_id, -5 ) === '-area';
                $items   = $data['items'] ?? [];
                $count   = count( $items );

                // FIXED: get or recalc SEO score - prioritize saved score from form submission
                $seo_score = null;

                // 1. Try to get the score that was saved during form submission
                if ( isset( $data['seo_score'] ) && is_numeric( $data['seo_score'] ) ) {
                    $seo_score = intval( $data['seo_score'] );
                }

                // 2. Fallback to cached score
                if ( $seo_score === null ) {
                    $seo_score = self::get_cached_seo_score( $ctx_id );
                }

                // 3. Last resort: calculate (but this will be the 15% capped version)
                if ( $seo_score === null ) {
                    $seo_score = self::calculate_template_seo_score( array_merge(
                        $data,
                        [
                            'hard_meta'       => $data['hard_meta']      ?? [],
                            'template_html'   => $data['template_html']  ?? '',
                            'template_source' => $data['template_source']?? 'custom',
                            'source_page_id'  => $data['source_page_id'] ?? '',
                        ]
                    ) );
                    self::set_cached_seo_score( $ctx_id, $seo_score );
                }

                // build example URL
                $example_url = '—';
                if ( $count ) {
                    $first_item = is_array( $items[0] ) ? $items[0]['name'] : $items[0];
                    $first_slug = sanitize_title( $first_item );
                    $example_url = $is_area
                        ? home_url( sprintf( '/%s-%s/', substr( $ctx_id, 0, -5 ), $first_slug ) )
                        : home_url( sprintf( '/%s/', $first_slug ) );
                }

                // pattern display
                $url_pattern = $is_area
                    ? '/' . substr( $ctx_id, 0, -5 ) . '-{location}/'
                    : '/{keyword}/';

                // action links
                $edit_url = add_query_arg(
                    [ 'page' => 'dpg-dashboard', 'tab' => 'edit', 'ctx' => $ctx_id ],
                    admin_url( 'admin.php' )
                );
                $delete_url = wp_nonce_url(
                    add_query_arg( [ 'action' => 'dpg_delete_template', 'ctx' => $ctx_id ],
                    admin_url( 'admin-post.php' ) ),
                    'dpg_delete_template'
                );
                ?>
                <tr onmouseover="this.style.backgroundColor='var(--dpg-gray-50)'"
                    onmouseout="this.style.backgroundColor=''">

                    <td style="padding:12px 8px;word-wrap:break-word;overflow:hidden;">
                        <strong style="font-size:14px;line-height:1.3;"><?php echo esc_html( $name ); ?></strong>
                        <div class="row-actions" style="font-size:11px;color:var(--dpg-gray-700);margin-top:2px;">
                            <?php printf( esc_html__( 'ID: %s', 'dpg' ), esc_html( $ctx_id ) ); ?>
                        </div>
                    </td>

                    <td style="padding:12px 8px;text-align:center;">
                        <span style="display:inline-block;padding:2px 8px;border-radius:12px;
                                     font-size:10px;font-weight:600;text-transform:uppercase;
                                     letter-spacing:0.3px;white-space:nowrap;<?php
                            echo $is_area
                                ? 'background:rgba(99,102,241,0.1);color:var(--dpg-primary);'
                                : 'background:rgba(16,185,129,0.1);color:var(--dpg-success);';
                        ?>">
                            <?php echo $is_area ? 'AREA' : 'KEYWORD'; ?>
                        </span>
                    </td>

                    <td style="padding:12px 8px;word-break:break-all;">
                        <code style="padding:4px 6px;border-radius:4px;
                                     border:1px solid var(--dpg-gray-200);
                                     background:var(--dpg-gray-50);font-family:monospace;
                                     font-size:11px;color:var(--dpg-primary);
                                     word-break:break-all;display:block;">
                            <?php echo esc_html( $url_pattern ); ?>
                        </code>
                    </td>

                    <td style="padding:12px 8px;text-align:center;">
                        <span style="display:inline-block;padding:4px 8px;border-radius:12px;
                                     background:var(--dpg-primary);color:#fff;font-weight:700;
                                     font-size:12px;min-width:24px;">
                            <?php echo esc_html( $count ); ?>
                        </span>
                    </td>

                    <td style="padding:12px 8px;text-align:center;">
                        <?php if ( $seo_score !== null ) :
                            $level = self::get_seo_score_level( $seo_score );
                            $icon  = self::get_seo_score_icon(  $seo_score ); ?>
                            <span style="display:inline-flex;align-items:center;gap:4px;
                                         padding:4px 8px;border-radius:12px;font-weight:700;
                                         font-size:11px;white-space:nowrap;<?php
                                echo $level === 'good'
                                    ? 'background:rgba(16,185,129,0.1);color:var(--dpg-success);'
                                    : ( $level === 'ok'
                                        ? 'background:rgba(245,158,11,0.1);color:var(--dpg-warning);'
                                        : 'background:rgba(239,68,68,0.1);color:var(--dpg-error);' );
                            ?>">
                                <?php echo esc_html( $seo_score . '%' ); ?> <?php echo esc_html( $icon ); ?>
                            </span>
                        <?php else : ?>
                            <span style="color:var(--dpg-gray-500);font-style:italic;">—</span>
                        <?php endif; ?>
                    </td>

                    <td style="padding:12px 8px;word-break:break-all;">
                        <?php if ( $example_url !== '—' ) : ?>
                            <a
                                href="<?php echo esc_url( $example_url ); ?>"
                                target="_blank"
                                style="font-weight:500;display:block;margin-bottom:2px;
                                    font-size:12px;text-decoration:none;
                                    overflow:hidden;text-overflow:ellipsis;white-space:nowrap;"
                                title="<?php echo esc_attr( $example_url ); ?>"
                            >
                                🔗 <?php echo esc_html( wp_parse_url( $example_url, PHP_URL_PATH ) ); ?>
                            </a>
                        <?php else : ?>
                            <span style="color:var(--dpg-gray-500);font-style:italic;">—</span>
                        <?php endif; ?>
                    </td>

                    <td style="padding:12px 8px;">
                        <div style="display:flex;gap:4px;flex-wrap:nowrap;">
                            <a href="<?php echo esc_url( $edit_url ); ?>"
                               class="dpg-button dpg-button-small"
                               style="padding:4px 8px;font-size:11px;text-decoration:none;
                                      white-space:nowrap;display:inline-flex;align-items:center;gap:2px;">
                                ✏️ <?php esc_html_e( 'Edit', 'dpg' ); ?>
                            </a>
                            <a href="<?php echo esc_url( $delete_url ); ?>"
                               class="dpg-button dpg-button-small"
                               style="padding:4px 8px;font-size:11px;text-decoration:none;
                                      white-space:nowrap;display:inline-flex;align-items:center;gap:2px;"
                               onclick="return confirm('<?php echo esc_js( __( 'Are you sure you want to delete this template? This can be restored later.', 'dpg' ) ); ?>');">
                                🗑️ <?php esc_html_e( 'Delete', 'dpg' ); ?>
                            </a>
                        </div>
                    </td>
                </tr>
            <?php endforeach; ?>
            </tbody>
        </table>
    </div>

<?php }


    /* -------------------------------------------------------------------------
     * CALCULATE & CACHE SEO SCORE
     * ---------------------------------------------------------------------- */
private static function calculate_template_seo_score( array $template_data ): ?int {

	/* 0. quick sanity check */
	if ( empty( $template_data ) ) {
		return null;
	}

	/* 1. pull the pieces the JS analyser needs --------------------------- */
	$focus = trim( (string) ( $template_data['focus_keyphrase'] ?? '' ) );

	$meta  = $template_data['hard_meta'] ?? [];
	$title = trim( (string) ( $meta['title']       ?? '' ) );
	$desc  = trim( (string) ( $meta['description'] ?? '' ) );

	$html  = (string) ( $template_data['template_html'] ?? '' );

	$sample_item = 'Example Item';
	if ( ! empty( $template_data['items'] ) ) {
		$first       = $template_data['items'][0];
		$sample_item = is_array( $first ) ? $first['name'] : $first;
	}

	$replace = static function ( $str, $sample ) {
		return str_ireplace( '{item}', $sample, $str );
	};

	$title = $replace( $title,  $sample_item );
	$desc  = $replace( $desc,   $sample_item );
	$html  = $replace( $html,   $sample_item );
	$focus = $replace( $focus,  $sample_item );

	$plain_text = wp_strip_all_tags( $html );
	$word_count = str_word_count( $plain_text );

	/* 2. run the same basic checks (weights identical to JS) ------------- */
	$checks = [];

	$checks['focus_keyphrase']            = $focus !== '' ? 100 : 0;
	$checks['content_length']             = $word_count >= 300 ? 100
	                                     : ( $word_count >= 150 ? 75
	                                     : ( $word_count >= 50  ? 25 : 0 ) );

	$kp_count = $focus ? preg_match_all( '/\b' . preg_quote( $focus, '/' ) . '\b/i', $plain_text ) : 0;
	$checks['keyphrase_frequency']        = $kp_count === 0 ? 0
	                                     : ( $kp_count === 1 ? 60
	                                     : ( $kp_count <= 4 ? 100 : 70 ) );

	$first200 = mb_strtolower( mb_substr( $plain_text, 0, 200 ) );
	$checks['keyphrase_first_paragraph']  = ( $focus && strpos( $first200, mb_strtolower( $focus ) ) !== false ) ? 100 : 0;

	$title_len = mb_strlen( $title );
	$checks['title_length']               = $title_len === 0 ? 0
	                                     : ( $title_len < 30 ? 60
	                                     : ( $title_len > 60 ? 80 : 100 ) );

	$checks['title_keyphrase']            = ( $focus && stripos( $title, $focus ) !== false ) ? 100 : 0;

	$desc_len  = mb_strlen( $desc );
	$checks['meta_description_length']    = $desc_len === 0 ? 0
	                                     : ( $desc_len < 120 ? 70
	                                     : ( $desc_len > 160 ? 85 : 100 ) );

	$checks['meta_description_keyphrase'] = ( $focus && stripos( $desc, $focus ) !== false ) ? 100 : 0;

	/* weighting identical to the front-end */
	$weights = [
		'focus_keyphrase'            => 15,
		'content_length'             => 5,
		'keyphrase_frequency'        => 10,
		'title_keyphrase'            => 15,
		'keyphrase_first_paragraph'  => 10,
		'title_length'               => 10,
		'meta_description_keyphrase' => 10,
		'meta_description_length'    => 10,
	];

	$total = 0; $max = 0;
	foreach ( $weights as $key => $w ) {
		$max   += $w;
		$total += ( $checks[ $key ] ?? 0 ) * $w / 100;
	}
	$score = $max ? round( ( $total / $max ) * 100 ) : 0;

	/* soft caps identical to JS */
	if ( $word_count < 10 )   $score = min( $score, 15 );
	elseif ( $word_count < 50 )  $score = min( $score, 35 );
	elseif ( $word_count < 100 ) $score = min( $score, 55 );
	elseif ( $word_count < 200 ) $score = min( $score, 75 );
	elseif ( $word_count < 300 ) $score = min( $score, 90 );

	$score = max( $focus ? 10 : 5, min( 100, $score ) );

	return (int) $score;
}
    private static function set_cached_seo_score( string $ctx_id, int $score ): void {
        if ( class_exists( 'DPG_DB' ) && method_exists( 'DPG_DB', 'save_score' ) ) {
            DPG_DB::save_score( $ctx_id, $score );
            return;
        }
        global $wpdb;
        $table = $wpdb->prefix . 'dpg_scores';
        $wpdb->query( $wpdb->prepare(
            "REPLACE INTO {$table} (ctx_id, last_score, updated_at) VALUES (%s, %d, %s)",
            sanitize_key( $ctx_id ),
            max( 0, min( 100, $score ) ),
            current_time( 'mysql' )
        ));
    }
    private static function get_cached_seo_score( string $ctx_id ): ?int {
        if ( class_exists( 'DPG_DB' ) && method_exists( 'DPG_DB', 'get_score' ) ) {
            $score = DPG_DB::get_score( $ctx_id );
            return is_numeric( $score ) ? (int) $score : null;
        }
        global $wpdb;
        $table = $wpdb->prefix . 'dpg_scores';
        $score = $wpdb->get_var( $wpdb->prepare(
            "SELECT last_score FROM {$table} WHERE ctx_id = %s",
            $ctx_id
        ));
        return is_numeric( $score ) ? (int) $score : null;
    }
    private static function get_seo_score_icon( int $score ): string {
        return $score >= 80 ? '✅' : ( $score >= 60 ? '⚠️' : '❌' );
    }
    private static function get_seo_score_level( int $score ): string {
        return $score >= 80 ? 'good' : ( $score >= 60 ? 'ok' : 'poor' );
    }

    /* -------------------------------------------------------------------------
     * HANDLERS FOR TEMPLATES
     * ---------------------------------------------------------------------- */
public static function handle_create_template() {
    if ( ! current_user_can( 'manage_options' ) ) {
        wp_die( esc_html__( 'Permission denied', 'dynamic-seo-pages') );
    }
    check_admin_referer( 'dpg_form_nonce' );

    // FREE VERSION: Check items limit
    $items_data  = wp_unslash( $_POST['dpg_items'] ?? '' );
    $items_array = array_filter( array_map( 'trim', explode( "\n", $items_data ) ) );

    if ( count( $items_array ) > 50 ) {
        // Redirect back to Create tab with error flag
        wp_safe_redirect( add_query_arg(
            [
                'page'            => 'dpg-dashboard',
                'tab'             => 'create',
                'dpg_items_error' => '1',
            ],
            admin_url( 'admin.php' )
        ) );
        exit;
    }

    if ( ! isset( self::$form_instances['create'] ) ) {
        wp_die( esc_html__( 'Create form not available.', 'dpg' ) );
    }

    $result = self::$form_instances['create']->process( wp_unslash( $_POST ) );

    // Cache SEO score (existing code)…
    $ctx_id = sanitize_key( wp_unslash( $_POST['dpg_ctx_id'] ?? '' ) );
    if ( $ctx_id ) {
        $json_file = DPG_DATA_DIR . "{$ctx_id}.json";
        if ( file_exists( $json_file ) ) {
            $tpl_data = json_decode( file_get_contents( $json_file ), true );
            $score = self::calculate_template_seo_score( array_merge(
                $tpl_data,
                [
                    'hard_meta'       => $tpl_data['hard_meta']        ?? [],
                    'template_html'   => $tpl_data['template_html']    ?? '',
                    'template_source' => $tpl_data['template_source']  ?? 'custom',
                    'source_page_id'  => $tpl_data['source_page_id']   ?? '',
                ]
            ) );
            self::set_cached_seo_score( $ctx_id, $score );
        }
    }

    return $result;
}


public static function handle_update_template() {
    if ( ! current_user_can( 'manage_options' ) ) {
        wp_die( esc_html__( 'Permission denied', 'dynamic-seo-pages') );
    }

    // Nonce verification
    $nonce_ok =
        ( isset( $_POST['dpg_nonce'] )  && wp_verify_nonce( $_POST['dpg_nonce'], 'dpg_update_template' ) ) ||
        ( isset( $_POST['_wpnonce'] )   && wp_verify_nonce( $_POST['_wpnonce'],  'dpg_form_nonce'      ) );

    // FREE VERSION: Check items limit for update
    $items_data  = wp_unslash( $_POST['dpg_items'] ?? '' );
    $items_array = array_filter( array_map( 'trim', explode( "\n", $items_data ) ) );

    if ( count( $items_array ) > 50 ) {
        // Redirect back to Edit tab with error flag
        $ctx = sanitize_key( wp_unslash( $_POST['dpg_ctx_id'] ?? '' ) );
        wp_safe_redirect( add_query_arg(
            [
                'page'            => 'dpg-dashboard',
                'tab'             => 'edit',
                'ctx'             => $ctx,
                'dpg_items_error' => '1',
            ],
            admin_url( 'admin.php' )
        ) );
        exit;
    }

    if ( empty( $_POST['dpg_ctx_id'] ) || ! isset( self::$form_instances['edit_class'] ) ) {
        wp_die( esc_html__( 'Edit form not available.', 'dpg' ) );
    }

    $ctx_id      = sanitize_key( wp_unslash( $_POST['dpg_ctx_id'] ) );
    $edit_class  = self::$form_instances['edit_class'];
    $edit_config = self::$form_instances['edit_config'] ?? [];

    try {
        $form   = new $edit_class( $ctx_id, $edit_config );
        $result = $form->process( wp_unslash( $_POST ) );
    } catch ( Exception $e ) {
        if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
            error_log( '[DPG] Edit-form exception: ' . $e->getMessage() );
        }
        wp_die(
            esc_html__( 'Error updating template:', 'dpg' ) . ' ' . esc_html( $e->getMessage() )
        );
    }

    // Refresh cached score (existing code)…
    $json_file = DPG_DATA_DIR . "{$ctx_id}.json";
    if ( file_exists( $json_file ) ) {
        $tpl_data = json_decode( file_get_contents( $json_file ), true );
        $score = self::calculate_template_seo_score( array_merge(
            $tpl_data,
            [
                'hard_meta'       => $tpl_data['hard_meta']        ?? [],
                'template_html'   => $tpl_data['template_html']    ?? '',
                'template_source' => $tpl_data['template_source']  ?? 'custom',
                'source_page_id'  => $tpl_data['source_page_id']   ?? '',
            ]
        ) );
        self::set_cached_seo_score( $ctx_id, $score );
    }

    return $result;
}


private static function render_guide_tab() {
    ?>
    <div class="dpg-tab-content-wrapper">
        <div class="dpg-form-container">
            <div class="dpg-form-header">
                <h2 class="dpg-form-title"><?php esc_html_e( 'How To Guide', 'dpg' ); ?></h2>
                <p class="dpg-form-description">
                    <?php esc_html_e(
                        'Dynamic SEO Pages Generator lets you define a single template and a list of items (keywords or service areas). As soon as you save a template, the plugin automatically creates friendly URLs for each item, injects your SEO meta tags, and generates a consolidated XML sitemap with per-page priorities.',
                        'dpg'
                    ); ?>
                </p>
            </div>

            <!-- 
            // VIDEO SECTION - Commented out for WordPress.org submission
            // Uncomment after plugin approval if desired
            <div class="dpg-form-section">
                <div class="dpg-section-content" style="text-align:center; margin:20px 0;">
                    <iframe
                        width="560" height="315"
                        style="max-width:100%; border-radius:8px; box-shadow:0 10px 30px rgba(0,0,0,0.1);"
                        src="https://www.youtube.com/embed/qRYwSyaF_8I?si=LDPmEwDIcdsjCf_c"
                        title="<?php echo esc_attr__( 'How To Guide Video', 'dpg' ); ?>"
                        frameborder="0"
                        allow="accelerometer; autoplay; clipboard-write; encrypted-media; gyroscope; picture-in-picture; web-share"
                        allowfullscreen>
                    </iframe>
                </div>
            </div>
            -->

            <div class="dpg-form-section">
                <div class="dpg-section-header">
                    <h3 class="dpg-section-title" data-section="quick-start">
                        <?php esc_html_e( 'Quick Start Guide', 'dpg' ); ?>
                    </h3>
                </div>
                <div class="dpg-section-content">
                    <div class="dpg-guide-intro" style="background: #f0f9ff; border: 1px solid #e0f2fe; border-radius: 8px; padding: 20px; margin-bottom: 24px;">
                        <p style="margin: 0; font-size: 16px; color: #0369a1;">
                            <strong>💡 <?php esc_html_e( 'What you\'ll create:', 'dpg' ); ?></strong>
                            <?php esc_html_e( 'From one template, automatically generate multiple SEO-optimized pages with unique URLs, meta tags, and sitemap entries.', 'dpg' ); ?>
                        </p>
                    </div>

                    <ol style="font-size: 15px; line-height: 1.8; padding-left: 20px; counter-reset: step-counter;">
                        <li style="margin-bottom: 16px; padding-left: 8px;">
                            <strong><?php esc_html_e( 'Navigate to Create Template', 'dpg' ); ?></strong><br>
                            <span style="color: #6b7280; font-size: 14px;">
                                <?php esc_html_e( 'Click on the "Create Template" tab in the Dynamic SEO menu.', 'dpg' ); ?>
                            </span>
                        </li>
                        
                        <li style="margin-bottom: 16px; padding-left: 8px;">
                            <strong><?php esc_html_e( 'Choose Your Template Type', 'dpg' ); ?></strong><br>
                            <span style="color: #6b7280; font-size: 14px;">
                                • <strong><?php esc_html_e( 'Service Area:', 'dpg' ); ?></strong> <?php esc_html_e( 'For location-based services (e.g., "plumber-chicago", "lawyer-miami")', 'dpg' ); ?><br>
                                • <strong><?php esc_html_e( 'Keyword:', 'dpg' ); ?></strong> <?php esc_html_e( 'For keyword targeting (e.g., "best-lawyer", "cheap-flights")', 'dpg' ); ?>
                            </span>
                        </li>
                        
                        <li style="margin-bottom: 16px; padding-left: 8px;">
                            <strong><?php esc_html_e( 'Create Your HTML Template', 'dpg' ); ?></strong><br>
                            <span style="color: #6b7280; font-size: 14px;">
                                <?php esc_html_e( 'Write your template HTML and include the {item} placeholder where you want dynamic content. Example:', 'dpg' ); ?><br>
                                <code style="background: #f3f4f6; padding: 4px 8px; border-radius: 4px; font-size: 12px; display: inline-block; margin-top: 4px;">
                                    &lt;h1&gt;Professional Services in {item}&lt;/h1&gt;
                                </code>
                            </span>
                        </li>
                        
                        <li style="margin-bottom: 16px; padding-left: 8px;">
                            <strong><?php esc_html_e( 'Add Your Items List', 'dpg' ); ?></strong><br>
                            <span style="color: #6b7280; font-size: 14px;">
                                <?php esc_html_e( 'Enter one item per line (cities, keywords, etc.). You can optionally add priority scores:', 'dpg' ); ?><br>
                                <code style="background: #f3f4f6; padding: 4px 8px; border-radius: 4px; font-size: 12px; display: inline-block; margin-top: 4px;">
                                    <?php esc_html_e( 'Chicago|0.9', 'dpg' ); ?><br>
                                    <?php esc_html_e( 'Miami|0.8', 'dpg' ); ?><br>
                                    <?php esc_html_e( 'Houston', 'dpg' ); ?>
                                </code>
                            </span>
                        </li>
                        
                        <li style="margin-bottom: 16px; padding-left: 8px;">
                            <strong><?php esc_html_e( 'Configure SEO Settings', 'dpg' ); ?></strong><br>
                            <span style="color: #6b7280; font-size: 14px;">
                                • <strong><?php esc_html_e( 'Focus Keyphrase:', 'dpg' ); ?></strong> <?php esc_html_e( 'Your main keyword with {item} placeholder', 'dpg' ); ?><br>
                                • <strong><?php esc_html_e( 'SEO Title:', 'dpg' ); ?></strong> <?php esc_html_e( 'Page title template (30-60 characters)', 'dpg' ); ?><br>
                                • <strong><?php esc_html_e( 'Meta Description:', 'dpg' ); ?></strong> <?php esc_html_e( 'Description template (120-160 characters)', 'dpg' ); ?>
                            </span>
                        </li>
                        
                        <li style="margin-bottom: 16px; padding-left: 8px;">
                            <strong><?php esc_html_e( 'Save and Generate', 'dpg' ); ?></strong><br>
                            <span style="color: #6b7280; font-size: 14px;">
                                <?php esc_html_e( 'Click "Create Template" and your pages will be automatically generated with:', 'dpg' ); ?><br>
                                • <?php esc_html_e( 'SEO-friendly URLs', 'dpg' ); ?><br>
                                • <?php esc_html_e( 'Unique meta tags for each page', 'dpg' ); ?><br>
                                • <?php esc_html_e( 'XML sitemap entries', 'dpg' ); ?><br>
                                • <?php esc_html_e( 'Social media meta tags', 'dpg' ); ?>
                            </span>
                        </li>
                    </ol>
                </div>
            </div>

            <div class="dpg-form-section">
                <div class="dpg-section-header">
                    <h3 class="dpg-section-title" data-section="example">
                        <?php esc_html_e( 'Example: Local Service Business', 'dpg' ); ?>
                    </h3>
                </div>
                <div class="dpg-section-content">
                    <div class="dpg-example-box" style="background: #f9fafb; border: 1px solid #e5e7eb; border-radius: 8px; padding: 20px;">
                        <p><strong><?php esc_html_e( 'Scenario:', 'dpg' ); ?></strong> <?php esc_html_e( 'A plumbing company serves 3 cities', 'dpg' ); ?></p>
                        
                        <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 20px; margin: 16px 0;">
                            <div>
                                <h4 style="margin: 0 0 8px 0; color: #374151;"><?php esc_html_e( 'Configuration:', 'dpg' ); ?></h4>
                                <ul style="margin: 0; padding-left: 16px; font-size: 14px; color: #6b7280;">
                                    <li><?php esc_html_e( 'Template Type: Service Area', 'dpg' ); ?></li>
                                    <li><?php esc_html_e( 'URL Slug: emergency-plumber', 'dpg' ); ?></li>
                                    <li><?php esc_html_e( 'Items: Chicago, Miami, Houston', 'dpg' ); ?></li>
                                </ul>
                            </div>
                            <div>
                                <h4 style="margin: 0 0 8px 0; color: #374151;"><?php esc_html_e( 'Generated URLs:', 'dpg' ); ?></h4>
                                <ul style="margin: 0; padding-left: 16px; font-size: 14px; color: #6b7280;">
                                    <li><code>/emergency-plumber-chicago/</code></li>
                                    <li><code>/emergency-plumber-miami/</code></li>
                                    <li><code>/emergency-plumber-houston/</code></li>
                                </ul>
                            </div>
                        </div>
                        
                        <p style="margin-top: 16px; font-size: 14px; color: #059669;">
                            <strong>✅ <?php esc_html_e( 'Result:', 'dpg' ); ?></strong> 
                            <?php esc_html_e( 'Three SEO-optimized landing pages with unique content, meta tags, and URLs - all from one template!', 'dpg' ); ?>
                        </p>
                    </div>
                </div>
            </div>

            <div class="dpg-form-section">
                <div class="dpg-section-header">
                    <h3 class="dpg-section-title" data-section="tips">
                        <?php esc_html_e( 'Pro Tips', 'dpg' ); ?>
                    </h3>
                </div>
                <div class="dpg-section-content">
                    <div class="dpg-tips-grid" style="display: grid; grid-template-columns: repeat(auto-fit, minmax(300px, 1fr)); gap: 16px;">
                        <div class="dpg-tip" style="background: #fefce8; border: 1px solid #fde047; border-radius: 8px; padding: 16px;">
                            <h4 style="margin: 0 0 8px 0; color: #a16207;">💡 <?php esc_html_e( 'SEO Optimization', 'dpg' ); ?></h4>
                            <p style="margin: 0; font-size: 14px; color: #92400e;">
                                <?php esc_html_e( 'Use your focus keyphrase naturally in your template. The plugin analyzes keyword density and provides real-time SEO scoring.', 'dpg' ); ?>
                            </p>
                        </div>
                        
                        <div class="dpg-tip" style="background: #f0f9ff; border: 1px solid #7dd3fc; border-radius: 8px; padding: 16px;">
                            <h4 style="margin: 0 0 8px 0; color: #0369a1;">🎯 <?php esc_html_e( 'URL Structure', 'dpg' ); ?></h4>
                            <p style="margin: 0; font-size: 14px; color: #0c4a6e;">
                                <?php esc_html_e( 'Choose URL slugs that match your business. Service areas create /slug-location/ URLs, keywords create /keyword/ URLs.', 'dpg' ); ?>
                            </p>
                        </div>
                        
                        <div class="dpg-tip" style="background: #f0fdf4; border: 1px solid #86efac; border-radius: 8px; padding: 16px;">
                            <h4 style="margin: 0 0 8px 0; color: #15803d;">📊 <?php esc_html_e( 'Priority Scores', 'dpg' ); ?></h4>
                            <p style="margin: 0; font-size: 14px; color: #14532d;">
                                <?php esc_html_e( 'Add |0.9 after important items to give them higher sitemap priority. Higher scores = better search engine crawling.', 'dpg' ); ?>
                            </p>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
    <?php
}

/**
 * Ajax: Auto-save a draft template
 */
public static function handle_auto_save() {
    check_ajax_referer( 'dpg_form_nonce', 'nonce' );
    if ( ! current_user_can( 'manage_options' ) ) {
        wp_send_json_error( 'Permission denied' );
    }

    // Delegate to your edit‐form class if it implements auto_save()
    $ctx = sanitize_text_field( wp_unslash( $_POST['ctx'] ?? '' ) );
    if ( isset( self::$form_instances['edit_class'], self::$form_instances['edit_config'] ) ) {
        $edit_class  = self::$form_instances['edit_class'];
        $edit_config = self::$form_instances['edit_config'];
        try {
            $form   = new $edit_class( $ctx, $edit_config );
            if ( method_exists( $form, 'auto_save' ) ) {
                $result = $form->auto_save( wp_unslash( $_POST ) );
                wp_send_json_success( $result );
            }
        } catch ( \Exception $e ) {
            wp_send_json_error( $e->getMessage() );
        }
    }

    wp_send_json_error( 'Auto-save not available.' );
}

/**
 * Ajax: Quick-save a template
 */
public static function handle_quick_save() {
    check_ajax_referer( 'dpg_form_nonce', 'nonce' );
    if ( ! current_user_can( 'manage_options' ) ) {
        wp_send_json_error( 'Permission denied' );
    }

    // Delegate to your edit‐form class if it implements quick_save()
    $ctx = sanitize_text_field( wp_unslash( $_POST['ctx'] ?? '' ) );
    if ( isset( self::$form_instances['edit_class'], self::$form_instances['edit_config'] ) ) {
        $edit_class  = self::$form_instances['edit_class'];
        $edit_config = self::$form_instances['edit_config'];
        try {
            $form   = new $edit_class( $ctx, $edit_config );
            if ( method_exists( $form, 'quick_save' ) ) {
                $result = $form->quick_save( wp_unslash( $_POST ) );
                wp_send_json_success( $result );
            }
        } catch ( \Exception $e ) {
            wp_send_json_error( $e->getMessage() );
        }
    }

    wp_send_json_error( 'Quick-save not available.' );
}

/**
 * Ajax: Clear plugin cache
 */
public static function handle_clear_cache() {
    check_ajax_referer( 'dpg_form_nonce', 'nonce' );
    if ( ! current_user_can( 'manage_options' ) ) {
        wp_send_json_error( 'Permission denied' );
    }

    // Flush rewrite rules & object cache
    flush_rewrite_rules();
    if ( function_exists( 'wp_cache_flush' ) ) {
        wp_cache_flush();
    }

    wp_send_json_success( 'Cache cleared.' );
}


public static function handle_delete_template() {
    if ( ! current_user_can( 'manage_options' ) ) {
        wp_die( esc_html__( 'Permission denied', 'dynamic-seo-pages') );
    }
    check_admin_referer( 'dpg_delete_template' );

    $ctx_id    = sanitize_key( wp_unslash( $_GET['ctx'] ?? '' ) );
    $json_file = DPG_DATA_DIR . "{$ctx_id}.json";
    $php_file  = DPG_TMPL_DIR . "{$ctx_id}.php";

    if ( ! file_exists( $json_file ) ) {
        wp_safe_redirect( add_query_arg( [
            'page'                  => 'dpg-dashboard',
            'delete_error_notfound' => '1',
        ], admin_url( 'admin.php' ) ) );
        exit;
    }

    // 1) record in “deleted” option
    $data    = json_decode( file_get_contents( $json_file ), true );
    $deleted = get_option( self::OPTION_DELETED_TEMPLATES, [] );
    $deleted[ $ctx_id ] = [
        'data'         => $data,
        'deleted_date' => current_time( 'mysql' ),
        'deleted_by'   => get_current_user_id(),
        'urls'         => self::get_template_urls( $ctx_id, $data ),
    ];
    update_option( self::OPTION_DELETED_TEMPLATES, $deleted );

    // 2) remove files via WP_Filesystem
    if ( ! function_exists( 'WP_Filesystem' ) ) {
        require_once ABSPATH . 'wp-admin/includes/file.php';
    }
    WP_Filesystem();
    global $wp_filesystem;
    $wp_filesystem->delete( $json_file );
    $wp_filesystem->delete( $php_file );

    // 3) flush routing & cache
    flush_rewrite_rules();
    if ( function_exists( 'wp_cache_flush' ) ) {
        wp_cache_flush();
    }

    wp_safe_redirect( add_query_arg( [
        'page'    => 'dpg-dashboard',
        'deleted' => '1',
    ], admin_url( 'admin.php' ) ) );
    exit;
}



public static function handle_restore_template() {
    if ( ! current_user_can( 'manage_options' ) ) {
        wp_die( esc_html__( 'Permission denied', 'dynamic-seo-pages') );
    }
    check_admin_referer( 'dpg_restore_template' );

    $ctx_id  = sanitize_key( wp_unslash( $_GET['ctx'] ?? '' ) );
    $deleted = get_option( self::OPTION_DELETED_TEMPLATES, [] );

    // If it wasn’t in the deleted list
    if ( ! isset( $deleted[ $ctx_id ] ) ) {
        wp_safe_redirect( esc_url_raw( add_query_arg([
            'page'                     => 'dpg-dashboard',
            'tab'                      => 'deleted',
            'restore_error_notfound'   => '1',
        ], admin_url( 'admin.php' ) ) ) );
        exit;
    }

    // If they’re at their template limit
    if ( ! self::can_create_code() ) {
        wp_safe_redirect( esc_url_raw( add_query_arg([
            'page'           => 'dpg-dashboard',
            'tab'            => 'deleted',
            'restore_error'  => '1',
        ], admin_url( 'admin.php' ) ) ) );
        exit;
    }

    // Write back JSON + PHP stub via WP_Filesystem
    if ( ! function_exists( 'WP_Filesystem' ) ) {
        require_once ABSPATH . 'wp-admin/includes/file.php';
    }
    WP_Filesystem();
    global $wp_filesystem;

    $template_data = $deleted[ $ctx_id ]['data'];
    $json_file     = DPG_DATA_DIR . "{$ctx_id}.json";
    $php_file      = DPG_TMPL_DIR . "{$ctx_id}.php";

    $json_content = wp_json_encode( $template_data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE );
    $wp_filesystem->put_contents( $json_file, $json_content, FS_CHMOD_FILE );

    $template_name = sanitize_text_field( $template_data['name'] ?? $ctx_id );
    $stub  = "<?php\n/**\n * Template: {$template_name}\n * Context : {$ctx_id}\n * Restored: " . current_time( 'mysql' ) . "\n */\n\n";
    $stub .= "if ( ! defined('ABSPATH') || ! isset(\$dpg_current_item) ) { status_header(404); exit; }\n\n";
    $stub .= "global \$dpg_current_item, \$dpg_current_slug;\n";
    $wp_filesystem->put_contents( $php_file, $stub, FS_CHMOD_FILE );

    // Remove from deleted list
    unset( $deleted[ $ctx_id ] );
    update_option( self::OPTION_DELETED_TEMPLATES, $deleted );

    // Flush rewrite rules
    flush_rewrite_rules();

    // Redirect back with success flag
    wp_safe_redirect( esc_url_raw( add_query_arg([
        'page'     => 'dpg-dashboard',
        'restored' => '1',
    ], admin_url( 'admin.php' ) ) ) );
    exit;
}


/**
 * 🔄 REPLACED: Permanent delete without .htaccess
 */
public static function handle_permanent_delete() {
    if ( ! current_user_can( 'manage_options' ) ) {
        wp_die( esc_html__( 'Permission denied', 'dynamic-seo-pages') );
    }
    check_admin_referer( 'dpg_permanent_delete' );

    $ctx_id  = sanitize_key( wp_unslash( $_GET['ctx'] ?? '' ) );
    $deleted = get_option( self::OPTION_DELETED_TEMPLATES, [] );

    // If it wasn’t in the deleted list
    if ( ! isset( $deleted[ $ctx_id ] ) ) {
        wp_safe_redirect( esc_url_raw( add_query_arg([
            'page'                       => 'dpg-dashboard',
            'tab'                        => 'deleted',
            'permanent_delete_notfound'  => '1',
        ], admin_url( 'admin.php' ) ) ) );
        exit;
    }

    // Actually remove it
    unset( $deleted[ $ctx_id ] );
    update_option( self::OPTION_DELETED_TEMPLATES, $deleted );

    // Redirect back with success flag
    wp_safe_redirect( esc_url_raw( add_query_arg([
        'page'              => 'dpg-dashboard',
        'tab'               => 'deleted',
        'permanent_deleted' => '1',
    ], admin_url( 'admin.php' ) ) ) );
    exit;
}



    /* -------------------------------------------------------------------------
     * GET TEMPLATE URLS
     * ---------------------------------------------------------------------- */
    private static function get_template_urls( $ctx_id, $template_data ) {
        $urls  = [];
        $items = $template_data['items'] ?? [];
        if ( substr( $ctx_id, -5 ) === '-area' ) {
            $base = substr( $ctx_id, 0, -5 );
            foreach ( $items as $item ) {
                $name = is_array( $item ) ? $item['name'] : $item;
                $urls[] = home_url( '/' . $base . '-' . sanitize_title( $name ) . '/' );
            }
        } else {
            foreach ( $items as $item ) {
                $name = is_array( $item ) ? $item['name'] : $item;
                $urls[] = home_url( '/' . sanitize_title( $name ) . '/' );
            }
        }
        return $urls;
    }

    /* -------------------------------------------------------------------------
     * HANDLE SEO ANALYSIS
     * ---------------------------------------------------------------------- */
    public static function handle_seo_analysis() {
        check_ajax_referer( 'dpg_form_nonce', 'nonce' );
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( 'Permission denied' );
        }
        $template_data = wp_unslash( $_POST['template_data'] ?? [] );
        if ( empty( $template_data ) ) {
            wp_send_json_error( 'No template data provided' );
        }
        $score = self::calculate_template_seo_score( $template_data );
        if ( $score === null ) {
            wp_send_json_error( 'Could not calculate SEO score' );
        }
        wp_send_json_success( [
            'score' => $score,
            'level' => self::get_seo_score_level( $score ),
            'icon'  => self::get_seo_score_icon( $score )
        ] );
    }

    /* -------------------------------------------------------------------------
     * HANDLE SAVE SITEMAP
     * ---------------------------------------------------------------------- */
    public static function handle_save_sitemap() {
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_die( esc_html__( 'Permission denied', 'dynamic-seo-pages') );
        }
        check_admin_referer( 'dpg_save_sitemap_nonce' );

        if ( ! empty( $_POST['dpg_reset_score'] ) ) {
            update_option( self::OPTION_GLOBAL_SCORE, 0.9 );
            self::clear_individual_overrides();
            flush_rewrite_rules();
            if ( class_exists( 'DPG_Frontend' ) && method_exists( 'DPG_Frontend', 'generate_sitemap' ) ) {
                DPG_Frontend::generate_sitemap();
            }
            wp_safe_redirect( add_query_arg( 'sitemap_saved', '1', admin_url( 'admin.php?page=dpg-dashboard&tab=sitemap' ) ) );
            exit;
        }

        $raw    = str_replace( ',', '.', sanitize_text_field( $_POST['dpg_global_score'] ?? '' ) );
        $global = is_numeric( $raw ) ? floatval( $raw ) : 0.9;
        $global = max( 0.01, min( 1.0, $global ) );
        update_option( self::OPTION_GLOBAL_SCORE, $global );

        foreach ( glob( DPG_DATA_DIR . '*.json' ) as $json_file ) {
            $data = json_decode( file_get_contents( $json_file ), true );
            if ( ! is_array( $data ) || empty( $data['items'] ) ) {
                continue;
            }
            foreach ( $data['items'] as $i => $entry ) {
                $name = is_array( $entry ) ? $entry['name'] : $entry;
                $data['items'][ $i ] = [
                    'name'  => sanitize_text_field( $name ),
                    'score' => $global,
                ];
            }
            file_put_contents( $json_file, wp_json_encode( $data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE ) );
        }

        flush_rewrite_rules();
        if ( class_exists( 'DPG_Frontend' ) && method_exists( 'DPG_Frontend', 'generate_sitemap' ) ) {
            DPG_Frontend::generate_sitemap();
        }

        wp_safe_redirect( add_query_arg( 'sitemap_saved', '1', admin_url( 'admin.php?page=dpg-dashboard&tab=sitemap' ) ) );
        exit;
    }

    /* -------------------------------------------------------------------------
     * CLEAR INDIVIDUAL OVERRIDES
     * ---------------------------------------------------------------------- */
    public static function clear_individual_overrides() {
        foreach ( glob( DPG_DATA_DIR . '*.json' ) as $json_file ) {
            $data = json_decode( file_get_contents( $json_file ), true );
            if ( ! is_array( $data ) || empty( $data['items'] ) ) {
                continue;
            }
            $updated = false;
            foreach ( $data['items'] as $i => $entry ) {
                if ( is_array( $entry ) && isset( $entry['score'] ) ) {
                    $data['items'][ $i ] = $entry['name'];
                    $updated = true;
                }
            }
            if ( $updated ) {
                file_put_contents( $json_file, wp_json_encode( $data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE ) );
            }
        }
        if ( class_exists( 'DPG_Frontend' ) && method_exists( 'DPG_Frontend', 'generate_sitemap' ) ) {
            DPG_Frontend::generate_sitemap();
        }
    }


        /* -------------------------------------------------------------------------
    * Render the per-URL overrides table on the Sitemap tab
    * ---------------------------------------------------------------------- */
    private static function render_url_overrides() {
        // Gather every dynamic URL and its override
        $rows = [];
        foreach ( glob( DPG_DATA_DIR . '*.json' ) as $file ) {
            $ctx_id = basename( $file, '.json' );
            if ( self::is_master_template( $ctx_id ) ) {
                continue;
            }
            $data  = json_decode( file_get_contents( $file ), true );
            $items = $data['items'] ?? [];
            foreach ( $items as $i => $entry ) {
                $name  = is_array( $entry ) ? $entry['name'] : $entry;
                $score = is_array( $entry ) && isset( $entry['score'] ) ? $entry['score'] : '';
                $slug  = sanitize_title( $name );
                $url   = ( substr( $ctx_id, -5 ) === '-area' )
                    ? home_url( '/' . substr( $ctx_id, 0, -5 ) . '-' . $slug . '/' )
                    : home_url( '/' . $slug . '/' );
                $rows[] = [ $ctx_id, $i, $url, $score ];
            }
        }

        // Filtering, pagination...
        $chosen_ctx = sanitize_key( wp_unslash( $_GET['template'] ?? '' ) );
        if ( $chosen_ctx !== '' ) {
            $rows = array_filter( $rows, fn( $r ) => $r[0] === $chosen_ctx );
        }
        $per_page = 50;
        $total    = count( $rows );
        $paged    = max( 1, absint( $_GET['paged'] ?? 1 ) );
        $pages    = max( 1, ceil( $total / $per_page ) );
        $offset   = ( $paged - 1 ) * $per_page;
        $rows     = array_slice( $rows, $offset, $per_page );
        ?>
        <div class="dpg-form-section">
            <div class="dpg-section-header">
                <h3 class="dpg-section-title" data-section="per-url-score-overrides">
                    <?php esc_html_e( 'Per-URL Score Overrides', 'dpg' ); ?>
                </h3>
            </div>
            <div class="dpg-section-content">
                <p class="dpg-section-description">
                    <?php esc_html_e( 'Override the global score for specific pages. Leave blank to use the global score.', 'dpg' ); ?>
                </p>

                <div class="dpg-url-controls" style="display:flex;justify-content:space-between;align-items:center;margin-bottom:20px;padding:16px;border:1px solid var(--dpg-gray-200);background:var(--dpg-gray-50);border-radius:8px;">
                    <form method="get" action="<?php echo esc_url( admin_url( 'admin.php' ) ); ?>">
                        <input type="hidden" name="page"  value="dpg-dashboard">
                        <input type="hidden" name="tab"   value="sitemap">
                        <select name="template" onchange="this.form.submit();" class="dpg-select" style="min-width:200px;">
                            <option value=""><?php esc_html_e( 'All templates', 'dpg' ); ?></option>
                            <?php foreach ( self::get_user_codes() as $path ) :
                                $id       = basename( $path, '.json' );
                                $selected = selected( $chosen_ctx, $id, false );
                            ?>
                            <option value="<?php echo esc_attr( $id ); ?>" <?php echo $selected; ?>>
                                <?php echo esc_html( $id ); ?>
                            </option>
                            <?php endforeach; ?>
                        </select>
                    </form>

                    <div style="display:flex;gap:12px;">
                        <a href="<?php echo esc_url( wp_nonce_url( admin_url( 'admin-post.php?action=dpg_export_scores' ), 'dpg_export_nonce' ) ); ?>" class="dpg-button dpg-button-secondary" style="background:var(--dpg-success);color:#fff;">
                            📥 <?php esc_html_e( 'Export CSV', 'dpg' ); ?>
                        </a>
                        <form method="post" enctype="multipart/form-data" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>" style="display:flex;gap:8px;align-items:center;">
                            <?php wp_nonce_field( 'dpg_import_nonce' ); ?>
                            <input type="hidden" name="action" value="dpg_import_scores">
                            <input type="file" name="csv" accept=".csv" required style="min-width:200px;">
                            <button type="submit" class="dpg-button dpg-button-secondary" style="background:var(--dpg-warning);color:#fff;">
                                📤 <?php esc_html_e( 'Import CSV', 'dpg' ); ?>
                            </button>
                        </form>
                    </div>
                </div>

                <?php if ( empty( $rows ) ) : ?>
                    <div class="dpg-notice dpg-notice-info" style="border:1px solid rgba(99,102,241,0.1);background:rgba(99,102,241,0.05);border-radius:8px;padding:16px;color:var(--dpg-primary);">
                        <p style="margin:0;font-weight:600;">ℹ️ <?php esc_html_e( 'No pages found.', 'dpg' ); ?></p>
                    </div>
                <?php else : ?>
                    <div style="overflow-x:auto;border:1px solid var(--dpg-gray-200);border-radius:8px;">
                        <table class="widefat striped dpg-urls-table" style="margin:0;border:none;">
                            <thead style="background:var(--dpg-gray-50);">
                                <tr>
                                    <th style="padding:16px;"><?php esc_html_e( 'URL', 'dpg' ); ?></th>
                                    <th style="padding:16px;"><?php esc_html_e( 'Score Override', 'dpg' ); ?></th>
                                    <th style="padding:16px;"><?php esc_html_e( 'Actions', 'dpg' ); ?></th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ( $rows as list( $ctx, $idx, $url, $score ) ) : ?>
                                    <tr style="border-bottom:1px solid var(--dpg-gray-200);">
                                        <td style="padding:16px;">
                                            <a href="<?php echo esc_url( $url ); ?>" target="_blank" style="color:var(--dpg-primary);text-decoration:none;font-weight:500;">
                                                <?php echo esc_html( $url ); ?>
                                            </a>
                                        </td>
                                        <td style="padding:16px;">
                                            <form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>">
                                                <?php wp_nonce_field( 'dpg_update_page_score_nonce' ); ?>
                                                <input type="hidden" name="action"     value="dpg_update_page_score">
                                                <input type="hidden" name="ctx_id"     value="<?php echo esc_attr( $ctx ); ?>">
                                                <input type="hidden" name="item_index" value="<?php echo esc_attr( $idx ); ?>">
                                                <div style="display:flex;gap:8px;align-items:center;">
                                                    <input
                                                        type="number"
                                                        name="item_score"
                                                        class="dpg-input dpg-input--numeric"
                                                        placeholder="<?php esc_attr_e( 'Global', 'dpg' ); ?>"
                                                        value="<?php echo esc_attr( $score ); ?>"
                                                        step="0.1"
                                                        min="0.1"
                                                        max="1"
                                                        style="max-width:120px;"
                                                    />
                                                    <button type="submit" class="dpg-button dpg-button-primary"><?php esc_html_e( 'Save', 'dpg' ); ?></button>
                                                </div>
                                            </form>
                                        </td>
                                        <td style="padding:16px;">
                                            <a href="<?php echo esc_url( $url ); ?>" target="_blank" class="dpg-button dpg-button-secondary">
                                                🔗 <?php esc_html_e( 'View', 'dpg' ); ?>
                                            </a>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>

                    <?php if ( $pages > 1 ) : ?>
                        <div style="display:flex;justify-content:center;gap:8px;margin-top:20px;">
                            <?php for ( $p = 1; $p <= $pages; $p++ ) :
                                $link   = add_query_arg(
                                            array_filter([
                                                'page'     => 'dpg-dashboard',
                                                'tab'      => 'sitemap',
                                                'template' => $chosen_ctx ?: null,
                                                'paged'    => $p,
                                            ]),
                                            admin_url( 'admin.php' )
                                        );
                                $active = $p === $paged;
                            ?>
                                <a
                                    href="<?php echo esc_url( $link ); ?>"
                                    class="dpg-page-link<?php echo $active ? ' current' : ''; ?>"
                                    style="padding:8px 12px;border-radius:6px;text-decoration:none;font-weight:600;<?php echo $active ? 'background:var(--dpg-primary);color:#fff;' : 'background:var(--dpg-gray-100);color:var(--dpg-gray-700);'; ?>"
                                >
                                    <?php echo esc_html( $p ); ?>
                                </a>
                            <?php endfor; ?>
                        </div>
                    <?php endif; ?>

                <?php endif; ?>
            </div>
        </div>
        <?php
    }

    /* -------------------------------------------------------------------------
    * TAB: Settings
    * ---------------------------------------------------------------------- */
 private static function render_settings_tab() {
        ?>
        <div class="dpg-tab-content-wrapper">
            <?php
            if ( isset( self::$form_instances['settings'] ) ) {
                self::$form_instances['settings']->render();
            } else {
                ?>
                <div class="dpg-form-container">
                    <div class="dpg-form-header">
                        <h2 class="dpg-form-title"><?php esc_html_e( 'Settings', 'dpg' ); ?></h2>
                        <p class="dpg-form-description"><?php esc_html_e( 'Configure your plugin settings and license.', 'dpg' ); ?></p>
                    </div>
                    <div class="dpg-form-section">
                        <div class="dpg-section-header">
                            <h3 class="dpg-section-title" data-section="form-error">Settings Error</h3>
                        </div>
                        <div class="dpg-section-content">
                            <div class="dpg-notice dpg-notice-error" style="background:rgba(239,68,68,0.05);border:1px solid rgba(239,68,68,0.1);border-radius:8px;padding:20px;text-align:center;">
                                <p style="margin:0;font-size:16px;color:var(--dpg-error);font-weight:600;">
                                    ⚠️ <?php esc_html_e( 'Settings form not available.', 'dpg' ); ?>
                                </p>
                            </div>
                        </div>
                    </div>
                </div>
                <?php
            }
            ?>
        </div>
        <?php
    }

    




private static function get_seo_score_from_templates_table( string $ctx_id ): ?int {
    // Get numeric ID from context
    $contexts = get_option( 'dpg_contexts', [] );
    $match = wp_list_filter( $contexts, [ 'ctx_id' => $ctx_id ] );
    
    if ( empty( $match ) ) {
        return null;
    }
    
    $numeric_id = absint( array_shift( $match )['id'] );
    
    global $wpdb;
    $templates_table = $wpdb->prefix . 'dpg_templates';
    
    $score = $wpdb->get_var( $wpdb->prepare(
        "SELECT seo_score FROM {$templates_table} WHERE id = %d",
        $numeric_id
    ) );
    
    return is_numeric( $score ) ? (int) $score : null;
}

private static function get_cached_seo_score_enhanced( string $ctx_id ): ?int {
    // Priority 1: Check dpg_scores table (most recent)
    if ( class_exists( 'DPG_DB' ) && method_exists( 'DPG_DB', 'get_score' ) ) {
        $score = DPG_DB::get_score( $ctx_id );
        if ( is_numeric( $score ) ) {
            return (int) $score;
        }
    }
    
    // Priority 2: Check dpg_templates table
    $templates_score = self::get_seo_score_from_templates_table( $ctx_id );
    if ( $templates_score !== null ) {
        return $templates_score;
    }
    
    // Priority 3: Fallback to wp_options (old method)
    $option_score = get_option( 'dpg_seo_score_' . $ctx_id, null );
    if ( is_numeric( $option_score ) ) {
        return (int) $option_score;
    }
    
    return null;
}
    /* -------------------------------------------------------------------------
     * RENDER SITEMAP TAB
     * ---------------------------------------------------------------------- */
private static function render_sitemap_tab() {
    $global_score = (float) get_option( self::OPTION_GLOBAL_SCORE, 0.9 );
    $sitemap_url  = home_url( '/dynamic-pages-sitemap.xml' );
    ?>
    <div class="dpg-tab-content-wrapper">
        <div class="dpg-form-container">
            <div class="dpg-form-header">
                <h2 class="dpg-form-title"><?php esc_html_e( 'Dynamic Sitemap', 'dpg' ); ?></h2>
                <p class="dpg-form-description"><?php esc_html_e( 'Manage your dynamic sitemap settings and page priorities.', 'dpg' ); ?></p>
            </div>

            <div class="dpg-form-section">
                <div class="dpg-section-header">
                    <h3 class="dpg-section-title" data-section="sitemap-url">
                        <?php esc_html_e( 'Sitemap URL', 'dpg' ); ?>
                    </h3>
                </div>
                <div class="dpg-section-content">
                    <div class="dpg-sitemap-url-container" style="display:flex;align-items:center;gap:16px;padding:16px;border:1px solid var(--dpg-gray-200);background:var(--dpg-gray-50);border-radius:8px;">
                        <code class="dpg-sitemap-url" style="flex:1;padding:12px;border:1px solid var(--dpg-gray-300);border-radius:6px;background:#fff;font-family:monospace;font-weight:600;color:var(--dpg-primary);">
                            <?php echo esc_html( $sitemap_url ); ?>
                        </code>
                        <a href="<?php echo esc_url( $sitemap_url ); ?>"
                           target="_blank"
                           class="dpg-button dpg-button-secondary"
                           style="padding:12px 20px;border-radius:6px;font-weight:600;color:#fff;background:var(--dpg-primary);text-decoration:none;transition:all .3s;">
                            📄 <?php esc_html_e( 'View Sitemap', 'dpg' ); ?>
                        </a>
                        <a href="<?php echo esc_url( $sitemap_url ); ?>"
                           download="dynamic-pages-sitemap.xml"
                           class="dpg-button dpg-button-secondary"
                           style="padding:12px 20px;border-radius:6px;font-weight:600;color:#fff;background:var(--dpg-success);text-decoration:none;transition:all .3s;">
                            ⬇️ <?php esc_html_e( 'Download Sitemap', 'dpg' ); ?>
                        </a>
                    </div>
                </div>
            </div>

            <?php
            // render the global‐score form and the per‐URL overrides table:
            self::render_global_score_form( $global_score );
            self::render_url_overrides();
            ?>
        </div>
    </div>
    <?php
}



    /* -------------------------------------------------------------------------
     * RENDER GLOBAL SCORE FORM
     * ---------------------------------------------------------------------- */
    private static function render_global_score_form( $global_score ) {
        ?>
        <form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>">
            <?php wp_nonce_field( 'dpg_save_sitemap_nonce' ); ?>
            <input type="hidden" name="action" value="dpg_save_sitemap">
            <label for="dpg_global_score"><?php esc_html_e( 'Global Score (0.01–1.0)', 'dpg' ); ?></label>
            <input type="number" name="dpg_global_score" id="dpg_global_score" value="<?php echo esc_attr( $global_score ); ?>" step="0.01" min="0.01" max="1.0">
            <button type="submit" class="button button-primary"><?php esc_html_e( 'Save Score', 'dpg' ); ?></button>
            <button type="submit" name="dpg_reset_score" value="1" class="button button-secondary" onclick="return confirm('<?php echo esc_js( __( 'Reset to 0.9 and clear overrides?', 'dpg' ) ); ?>');"><?php esc_html_e( 'Reset All to 0.9', 'dpg' ); ?></button>
        </form>
        <?php
    }



    /* -------------------------------------------------------------------------
     * CLEAR ALL DELETED TEMPLATES
     * ---------------------------------------------------------------------- */
    public static function clear_all_deleted_templates() {
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_die( esc_html__( 'Permission denied', 'dpg' ) );
        }
        check_admin_referer( 'dpg_clear_deleted_templates' );
        delete_option( self::OPTION_DELETED_TEMPLATES );
        wp_safe_redirect( add_query_arg( 'deleted_cleared', '1', admin_url( 'admin.php?page=dpg-dashboard&tab=deleted' ) ) );
        exit;
    }

    /* -------------------------------------------------------------------------
     * HANDLE REFRESH RULES
     * ---------------------------------------------------------------------- */
    public static function handle_refresh_rules() {
        if ( isset( $_GET['dpg_refresh_rules'] ) && '1' === $_GET['dpg_refresh_rules'] ) {
            if ( ! current_user_can( 'manage_options' ) ) {
                wp_die( esc_html__( 'Permission denied', 'dpg' ) );
            }
            global $wp_rewrite;
            $wp_rewrite->init();
            flush_rewrite_rules( true );
            wp_safe_redirect( add_query_arg( 'rules_refreshed', '1', admin_url( 'admin.php?page=dpg-dashboard' ) ) );
            exit;
        }
    }



    /* -------------------------------------------------------------------------
     * TEMPLATE UTILITIES
     * ---------------------------------------------------------------------- */
    private static function is_master_template( $ctx_id ) {
        return dpg_is_master_template( $ctx_id );
    }
    private static function get_user_codes() {
        $files = glob( DPG_DATA_DIR . '*.json' );
        if ( ! $files ) {
            return [];
        }
        return array_filter( $files, function( $file ) {
            $ctx = basename( $file, '.json' );
            return ! dpg_is_master_template( $ctx );
        } );
    }
/**
 * Get maximum templates allowed (obfuscated)
 */
private static function get_max_code() {
  
    $base = (int) floor( sqrt( 4 ) ); 
    $offset = (int) ceil( log( 2.718, M_E ) ); 
    return $base - $offset; 
}


private static function listvars() {
    $base  = (int) ( 6 * 10 - 10 );
    $check = defined( 'DPG_VERSION' ) ? 0 : 100;
    return $base + $check; 
}


private static function can_create_code() {
    $active_templates = count( self::get_user_codes() );
    return $active_templates < self::get_max_code();
}

    private static function get_remaining_code_slots() {
        return max( 0, self::get_max_code() - count( self::get_user_codes() ) );
    }


} // end class DPG_Admin

// Initialize the admin class.
DPG_Admin::init();


