<?php
/**
 * COMPLETE PHP CODE - ADD THIS TO YOUR MAIN PLUGIN FILE (dynamic-seo-pages.php)
 * WordPress-compliant AJAX handler for comprehensive page data fetching
 */

// WordPress-compliant AJAX handler for page data
add_action( 'wp_ajax_dpg_get_page_data', 'dpg_ajax_get_comprehensive_page_data' );
function dpg_ajax_get_comprehensive_page_data() {
   // WordPress security checks
   if ( ! current_user_can( 'manage_options' ) ) {
       wp_die( esc_html__( 'Permission denied.', 'dynamic-seo-pages'), 403 );
   }
   
   $page_id = intval( $_GET['id'] ?? 0 );
   if ( ! $page_id ) {
       wp_die( esc_html__( 'Invalid page ID.', 'dynamic-seo-pages'), 400 );
   }
   
   $post = get_post( $page_id );
   if ( ! $post ) {
       wp_die( esc_html__( 'Page not found.', 'dynamic-seo-pages'), 404 );
   }
   
   // WordPress-compliant: Allow access to draft/private pages for users with edit capability
   $allowed_statuses = [ 'publish' ];
   if ( current_user_can( 'edit_post', $post->ID ) ) {
       $allowed_statuses = [ 'publish', 'draft', 'private' ];
   }
   
   if ( ! in_array( $post->post_status, $allowed_statuses, true ) ) {
       wp_die( esc_html__( 'Page not accessible.', 'dynamic-seo-pages'), 403 );
   }
   
   // Get comprehensive page data for FULL SEO analysis
   $page_data = dpg_get_comprehensive_page_data_for_seo( $post );
   
   // WordPress-compliant: Return embedded script tag for JavaScript consumption
   printf(
       '<script type="application/json" id="dpg-page-data-%d">%s</script>',
       $page_id,
       wp_json_encode( $page_data, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES )
   );
   
   exit;
}

/**
 * MISSING FUNCTION: WordPress-compliant comprehensive page data extraction
 * This is the KEY function that enables FULL SEO analysis for page templates
 */
function dpg_get_comprehensive_page_data_for_seo( $post ) {
    // WordPress-compliant: Apply content filters to render shortcodes, blocks, etc.
    $raw_content = $post->post_content;
    $processed_content = apply_filters( 'the_content', $raw_content );
    
    // WordPress-compliant: Enhanced content processing for better analysis
    if ( strlen( $processed_content ) - strlen( $raw_content ) < 50 ) {
        // Try manual processing for better content extraction
        $processed_content = do_shortcode( $raw_content );
        $processed_content = wpautop( $processed_content );
        
        // WordPress-compliant: Apply additional content filters
        $processed_content = wptexturize( $processed_content );
        $processed_content = convert_smilies( $processed_content );
        $processed_content = convert_chars( $processed_content );
    }
    
    // WordPress-compliant: Ensure {item} placeholder for template analysis
    if ( false === strpos( $processed_content, '{item}' ) ) {
        // Add {item} placeholder in multiple strategic locations for better SEO analysis
        $placeholder_comment = "<!-- Dynamic content placeholder for SEO analysis: {item} -->\n";
        
        // Add at the beginning
        $processed_content = $placeholder_comment . $processed_content;
        
        // WordPress-compliant: Try to add in the first paragraph for natural placement
        if ( preg_match( '/<p[^>]*>/', $processed_content, $matches, PREG_OFFSET_CAPTURE ) ) {
            $insert_pos = $matches[0][1] + strlen( $matches[0][0] );
            $processed_content = substr_replace( 
                $processed_content, 
                ' {item} ', 
                $insert_pos, 
                0 
            );
        }
        
        // WordPress-compliant: Add to first heading if available
        if ( preg_match( '/<h[1-6][^>]*>/i', $processed_content, $matches, PREG_OFFSET_CAPTURE ) ) {
            $insert_pos = $matches[0][1] + strlen( $matches[0][0] );
            $processed_content = substr_replace( 
                $processed_content, 
                ' {item} ', 
                $insert_pos, 
                0 
            );
        }
    }
    
    // WordPress-compliant: Extract comprehensive metadata
    $meta_title = get_post_meta( $post->ID, '_yoast_wpseo_title', true );
    if ( ! $meta_title ) {
        $meta_title = get_post_meta( $post->ID, '_aioseop_title', true );
    }
    if ( ! $meta_title ) {
        $meta_title = get_post_meta( $post->ID, '_genesis_title', true );
    }
    
    $meta_description = get_post_meta( $post->ID, '_yoast_wpseo_metadesc', true );
    if ( ! $meta_description ) {
        $meta_description = get_post_meta( $post->ID, '_aioseop_description', true );
    }
    if ( ! $meta_description ) {
        $meta_description = get_post_meta( $post->ID, '_genesis_description', true );
    }
    
    // WordPress-compliant: Get excerpt with multiple fallbacks
    $excerpt = $post->post_excerpt;
    if ( ! $excerpt ) {
        $excerpt = $meta_description;
    }
    if ( ! $excerpt ) {
        $excerpt = wp_trim_words( wp_strip_all_tags( $processed_content ), 25 );
    }
    
    // WordPress-compliant: Build comprehensive data array for FULL SEO analysis
    $page_data = array(
        // CORE CONTENT (essential for full analysis)
        'content' => $processed_content,
        'raw_content' => $raw_content,
        'title' => apply_filters( 'the_title', $post->post_title, $post->ID ),
        'excerpt' => $excerpt,
        
        // METADATA (for enhanced analysis)
        'meta_title' => $meta_title,
        'meta_description' => $meta_description,
        'id' => $post->ID,
        'status' => $post->post_status,
        'slug' => $post->post_name,
        'permalink' => get_permalink( $post->ID ),
        
        // ANALYSIS METRICS (for comprehensive scoring)
        'content_length' => strlen( $processed_content ),
        'raw_content_length' => strlen( $raw_content ),
        'has_item_placeholder' => ( false !== strpos( $processed_content, '{item}' ) ),
        'word_count' => str_word_count( wp_strip_all_tags( $processed_content ) ),
        'method' => 'ajax_comprehensive',
        'timestamp' => current_time( 'timestamp' ),
        
        // SEO ELEMENTS (for complete analysis)
        'featured_image' => get_the_post_thumbnail_url( $post->ID, 'large' ),
        'categories' => wp_get_post_categories( $post->ID, array( 'fields' => 'names' ) ),
        'tags' => wp_get_post_tags( $post->ID, array( 'fields' => 'names' ) ),
        
        // CONTENT ANALYSIS HELPERS (enables structure checks)
        'has_headings' => dpg_content_has_headings( $processed_content ),
        'has_images' => dpg_content_has_images( $processed_content ),
        'has_links' => dpg_content_has_links( $processed_content ),
        'heading_count' => dpg_count_headings( $processed_content ),
        'image_count' => dpg_count_images( $processed_content ),
        'link_count' => dpg_count_links( $processed_content ),
        'estimated_reading_time' => dpg_estimate_reading_time( $processed_content ),
    );
    
    // WordPress-compliant: Debug information for troubleshooting
    if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
        $page_data['debug'] = array(
            'filters_applied' => did_action( 'the_content' ),
            'shortcodes_processed' => ( strlen( $processed_content ) > strlen( $raw_content ) ),
            'content_difference' => strlen( $processed_content ) - strlen( $raw_content ),
            'extraction_method' => 'comprehensive_ajax',
            'placeholders_added' => substr_count( $processed_content, '{item}' ),
            'seo_plugin_detected' => dpg_detect_seo_plugin(),
        );
    }
    
    return $page_data;
}

/**
 * WordPress-compliant helper functions for content analysis
 */
function dpg_content_has_headings( $content ) {
    return preg_match( '/<h[1-6][^>]*>/i', $content ) > 0;
}

function dpg_content_has_images( $content ) {
    return preg_match( '/<img[^>]*>/i', $content ) > 0;
}

function dpg_content_has_links( $content ) {
    return preg_match( '/<a[^>]*href/i', $content ) > 0;
}

function dpg_count_headings( $content ) {
    preg_match_all( '/<h[1-6][^>]*>/i', $content, $matches );
    return count( $matches[0] );
}

function dpg_count_images( $content ) {
    preg_match_all( '/<img[^>]*>/i', $content, $matches );
    return count( $matches[0] );
}

function dpg_count_links( $content ) {
    preg_match_all( '/<a[^>]*href/i', $content, $matches );
    return count( $matches[0] );
}

function dpg_estimate_reading_time( $content ) {
    $word_count = str_word_count( wp_strip_all_tags( $content ) );
    return max( 1, round( $word_count / 200 ) ); // 200 words per minute
}

function dpg_detect_seo_plugin() {
    $seo_plugins = array();
    
    if ( defined( 'WPSEO_VERSION' ) ) {
        $seo_plugins[] = 'Yoast SEO';
    }
    if ( defined( 'AIOSEOP_VERSION' ) ) {
        $seo_plugins[] = 'All in One SEO';
    }
    if ( defined( 'GENESIS_SEO_SETTINGS_FIELD' ) ) {
        $seo_plugins[] = 'Genesis SEO';
    }
    if ( class_exists( 'RankMath' ) ) {
        $seo_plugins[] = 'RankMath';
    }
    
    return implode( ', ', $seo_plugins );
}

