<?php
// File: wp-content/plugins/dynamic-seo-pages/includes/class-dpg-frontend.php

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class DPG_Frontend {

    /* --------------------------------------------------------------------- */
    /*  BOOTSTRAP                                                            */
    /* --------------------------------------------------------------------- */
    public static function init() {
        // 0 = start output buffering before any header output
        add_action( 'template_redirect', [ __CLASS__, 'remove_conflicting_tags' ], 0 );

        // 1 = then route the request
        add_action( 'template_redirect', [ __CLASS__, 'template_router' ], 1 );

        // set up our shortcode
        add_shortcode( 'dpg_item', [ __CLASS__, 'shortcode_item' ] );

        // sitemap regen
        add_action( 'init', [ __CLASS__, 'generate_sitemap' ] );
        
        // NEW: Register rewrite rules automatically
        add_action( 'init', [ __CLASS__, 'register_rewrite_rules' ], 5 );
        
        // NEW: Auto-flush rewrite rules when needed
        add_action( 'init', [ __CLASS__, 'maybe_flush_rewrite_rules' ] );
    }

    public static function maybe_flush_rewrite_rules() {
        // Check if we need to flush rewrite rules
        $last_flush = get_option( 'dpg_last_rewrite_flush', 0 );
        $current_time = time();
        
        // Flush once per day or when plugin is updated
        if ( ( $current_time - $last_flush ) > DAY_IN_SECONDS ) {
            flush_rewrite_rules( false ); // false = don't hard flush
            update_option( 'dpg_last_rewrite_flush', $current_time );
            
            if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                error_log( 'DPG: Rewrite rules auto-flushed' );
            }
        }
    }

    /**
     * Enhanced WordPress-compliant base path detection
     */
    private static function get_wp_base_path() {
        static $base_path = null;
        
        if ( null !== $base_path ) {
            return $base_path;
        }
        
        // Method 1: WordPress-standard using home_url()
        $home_url = home_url();
        $home_parsed = wp_parse_url( $home_url );
        $home_path = isset( $home_parsed['path'] ) ? $home_parsed['path'] : '';
        
        // Method 2: WordPress-standard using site_url() as fallback
        $site_url = site_url();
        $site_parsed = wp_parse_url( $site_url );
        $site_path = isset( $site_parsed['path'] ) ? $site_parsed['path'] : '';
        
        if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
            error_log( 'DPG: Debug path detection:' );
            error_log( 'DPG: home_url() = ' . $home_url );
            error_log( 'DPG: site_url() = ' . $site_url );
            error_log( 'DPG: home_path = ' . $home_path );
            error_log( 'DPG: site_path = ' . $site_path );
            error_log( 'DPG: REQUEST_URI = ' . ( $_SERVER['REQUEST_URI'] ?? 'NOT_SET' ) );
        }
        
        // Priority 1: Use home_path if it exists and is not root
        if ( $home_path && $home_path !== '/' ) {
            $base_path = trim( $home_path, '/' );
            if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                error_log( 'DPG: Using home_path base: "' . $base_path . '"' );
            }
            return $base_path;
        }
        
        // Priority 2: Use site_path if it exists and is not root
        if ( $site_path && $site_path !== '/' ) {
            $base_path = trim( $site_path, '/' );
            if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                error_log( 'DPG: Using site_path base: "' . $base_path . '"' );
            }
            return $base_path;
        }
        
        // Priority 3: Auto-detect from server variables
        if ( isset( $_SERVER['REQUEST_URI'] ) && isset( $_SERVER['SCRIPT_NAME'] ) ) {
            $request_uri = wp_unslash( $_SERVER['REQUEST_URI'] );
            $script_name = wp_unslash( $_SERVER['SCRIPT_NAME'] );
            
            // Extract directory from script name
            $script_dir = dirname( $script_name );
            if ( $script_dir !== '/' ) {
                // Remove /wp-admin, /wp-content etc. to get base
                $script_dir = preg_replace( '#/(wp-admin|wp-content|wp-includes).*$#', '', $script_dir );
                if ( $script_dir && $script_dir !== '/' ) {
                    $base_path = trim( $script_dir, '/' );
                    if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                        error_log( 'DPG: Using auto-detected base: "' . $base_path . '"' );
                    }
                    return $base_path;
                }
            }
        }
        
        // Default: Root installation
        $base_path = '';
        if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
            error_log( 'DPG: Using default root base: "' . $base_path . '"' );
        }
        
        return $base_path;
    }

    /**
     * Return full, sanitized current URL.
     */
    private static function get_current_url(): string {
        // parse_request_url() will strip off the WP base‐path and any query
        $clean = self::parse_request_url();

        // ensure it begins with a slash (and preserve a trailing slash if present)
        $path = '/' . ltrim( $clean, '/' );

        return esc_url_raw( home_url( $path ) );
    }

    /**
     * Enhanced URL parsing that accounts for subdirectory installations
     */
    private static function parse_request_url() {
        $request_uri = wp_unslash( $_SERVER['REQUEST_URI'] ?? '' );
        $base_path = self::get_wp_base_path();
        
        if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
            error_log( 'DPG: Parsing URL - REQUEST_URI: ' . $request_uri );
            error_log( 'DPG: Base path: "' . $base_path . '"' );
        }
        
        // Remove base path from request URI if it exists
        if ( $base_path ) {
            $base_pattern = '/' . preg_quote( $base_path, '/' ) . '/';
            $clean_uri = preg_replace( '#^' . $base_pattern . '#', '/', $request_uri );
            if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                error_log( 'DPG: Clean URI after base removal: ' . $clean_uri );
            }
        } else {
            $clean_uri = $request_uri;
        }
        
        // Remove query string
        $clean_uri = strtok( $clean_uri, '?' );
        
        // Remove leading/trailing slashes and parse
        $clean_uri = trim( $clean_uri, '/' );
        
        if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
            error_log( 'DPG: Final clean URI: "' . $clean_uri . '"' );
        }
        
        return $clean_uri;
    }

    /* --------------------------------------------------------------------- */
    /*  MAIN ROUTER                                                          */
    /* --------------------------------------------------------------------- */
    public static function template_router() {
        // Method 1: Try to get from WordPress query vars
        $ctx_id   = sanitize_key( get_query_var( 'dpg_ctx' ) );
        $item_raw = sanitize_text_field( get_query_var( 'dpg_item' ) );

        // Method 2: Fallback in case some installs strip early filters
        global $wp_query;
        if ( ! $ctx_id && isset( $wp_query->query_vars['dpg_ctx'] ) ) {
            $ctx_id = sanitize_key( $wp_query->query_vars['dpg_ctx'] );
        }
        if ( ! $item_raw && isset( $wp_query->query_vars['dpg_item'] ) ) {
            $item_raw = sanitize_text_field( $wp_query->query_vars['dpg_item'] );
        }

        // Method 3: Enhanced manual URL parsing for subdirectory installations
        if ( ! $ctx_id || ! $item_raw ) {
            $parsed_url = self::parse_request_url();
            
            if ( $parsed_url ) {
                // Try to match against known templates
                list( $manual_ctx, $manual_item ) = self::manual_url_parse( $parsed_url );
                
                if ( $manual_ctx && $manual_item ) {
                    $ctx_id = $manual_ctx;
                    $item_raw = $manual_item;
                    
                    if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                        error_log( 'DPG: Manual parsing successful - ctx: ' . $ctx_id . ', item: ' . $item_raw );
                    }
                }
            }
        }

        if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
            error_log( 'DPG: Final router values - ctx: ' . $ctx_id . ', item: ' . $item_raw );
        }

        if ( ! get_option( 'dpg_pages_active', false ) ) {
            return;
        }

        // NEW: IMMEDIATELY set 200 status when we have a valid dynamic page request
        if ( ( $ctx_id && $item_raw ) || ( $item_raw && ! $ctx_id ) ) {
            // Verify this is actually a valid page before setting 200 status
            if ( self::verify_valid_dynamic_page( $ctx_id, $item_raw ) ) {
                self::set_200_status_early();
            }
        }

        // Service-area style: both ctx and item present
        if ( $ctx_id && $item_raw ) {
            self::handle_service_area_request( $ctx_id, $item_raw );
        }
        // Keyword-only style
        elseif ( $item_raw && ! $ctx_id ) {
            self::handle_keyword_request( $item_raw );
        }
    }

    private static function verify_valid_dynamic_page( $ctx_id, $item_raw ) {
        // For keyword-only requests
        if ( ! $ctx_id && $item_raw ) {
            $json_file = DPG_DATA_DIR . 'keyword.json';
            return self::check_item_exists_in_template( $json_file, $item_raw );
        }
        
        // For service-area requests
        if ( $ctx_id && $item_raw ) {
            $json_file = DPG_DATA_DIR . "{$ctx_id}.json";
            return self::check_item_exists_in_template( $json_file, $item_raw );
        }
        
        return false;
    }

    // Check if item actually exists in template
    private static function check_item_exists_in_template( $json_file, $item_raw ) {
        global $wp_filesystem;
        if ( ! function_exists( 'WP_Filesystem' ) ) {
            require_once ABSPATH . 'wp-admin/includes/file.php';
        }
        WP_Filesystem();

        if ( ! $wp_filesystem->exists( $json_file ) ) {
            return false;
        }

        $contents = $wp_filesystem->get_contents( $json_file );
        if ( false === $contents ) {
            return false;
        }

        $cfg = json_decode( $contents, true );
        if ( ! is_array( $cfg['items'] ?? null ) ) {
            return false;
        }

        $requested = sanitize_title( $item_raw );
        foreach ( $cfg['items'] as $entry ) {
            $name = is_array( $entry ) ? $entry['name'] : $entry;
            if ( sanitize_title( $name ) === $requested ) {
                return true;
            }
        }
        
        return false;
    }

    /**
     * Automatically register rewrite rules for all dynamic pages
     * This prevents the need for manual .htaccess editing
     */
    public static function register_rewrite_rules() {
        if ( ! is_dir( DPG_DATA_DIR ) ) {
            return;
        }

        global $wp_filesystem;
        if ( ! function_exists( 'WP_Filesystem' ) ) {
            require_once ABSPATH . 'wp-admin/includes/file.php';
        }
        WP_Filesystem();

        $json_files = glob( DPG_DATA_DIR . '*.json' );
        if ( ! $json_files ) {
            return;
        }

        foreach ( $json_files as $file ) {
            $ctx_id = sanitize_file_name( basename( $file, '.json' ) );
            
            // Skip master templates
            if ( dpg_is_master_template( $ctx_id ) ) {
                continue;
            }
            
            if ( ! $wp_filesystem->exists( $file ) ) {
                continue;
            }
            
            $content = $wp_filesystem->get_contents( $file );
            if ( false === $content ) {
                continue;
            }
            
            $data = json_decode( $content, true );
            if ( ! is_array( $data ) || empty( $data['items'] ) ) {
                continue;
            }

            // Register rewrite rules for each item
            foreach ( $data['items'] as $item ) {
                $item_name = is_array( $item ) ? $item['name'] : $item;
                $item_slug = sanitize_title( $item_name );
                
                if ( substr( $ctx_id, -5 ) === '-area' ) {
                    // Service-area pattern: base-item
                    $base = substr( $ctx_id, 0, -5 );
                    $url_pattern = $base . '-' . $item_slug;
                    
                    add_rewrite_rule(
                        '^' . preg_quote( $url_pattern ) . '/?$',
                        'index.php?dpg_ctx=' . $ctx_id . '&dpg_item=' . $item_slug,
                        'top'
                    );
                } else {
                    // Keyword pattern: direct item
                    add_rewrite_rule(
                        '^' . preg_quote( $item_slug ) . '/?$',
                        'index.php?dpg_ctx=' . $ctx_id . '&dpg_item=' . $item_slug,
                        'top'
                    );
                }
            }
        }
        
        // Register our custom query vars
        add_filter( 'query_vars', function( $vars ) {
            $vars[] = 'dpg_ctx';
            $vars[] = 'dpg_item';
            return $vars;
        });
    }

    /**
     * Manual URL parsing for when rewrite rules fail
     */
    private static function manual_url_parse( $clean_uri ) {
        if ( ! $clean_uri ) {
            return [ null, null ];
        }
        
        $parts = explode( '/', $clean_uri );
        if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
            error_log( 'DPG: URL parts: ' . implode( ', ', $parts ) );
        }
        
        // Check all available templates
        $json_files = glob( DPG_DATA_DIR . '*.json' );
        if ( ! $json_files ) {
            return [ null, null ];
        }
        
        global $wp_filesystem;
        if ( ! function_exists( 'WP_Filesystem' ) ) {
            require_once ABSPATH . 'wp-admin/includes/file.php';
        }
        WP_Filesystem();
        
        foreach ( $json_files as $file ) {
            $ctx_id = sanitize_file_name( basename( $file, '.json' ) );
            
            // Skip master templates
            if ( dpg_is_master_template( $ctx_id ) ) {
                continue;
            }
            
            if ( ! $wp_filesystem->exists( $file ) ) {
                continue;
            }
            
            $content = $wp_filesystem->get_contents( $file );
            if ( false === $content ) {
                continue;
            }
            
            $data = json_decode( $content, true );
            if ( ! is_array( $data ) || empty( $data['items'] ) ) {
                continue;
            }
            
            // Check if this is a service-area template (ends with -area)
            if ( substr( $ctx_id, -5 ) === '-area' ) {
                $base = substr( $ctx_id, 0, -5 );
                
                // Pattern: base-item (e.g., "best-seo-plugin-cape-town")
                if ( count( $parts ) === 1 && strpos( $parts[0], $base . '-' ) === 0 ) {
                    $potential_item = substr( $parts[0], strlen( $base . '-' ) );
                    
                    // Check if this item exists in the template
                    foreach ( $data['items'] as $item ) {
                        $item_name = is_array( $item ) ? $item['name'] : $item;
                        if ( sanitize_title( $item_name ) === $potential_item ) {
                            if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                                error_log( 'DPG: Found service-area match - ctx: ' . $ctx_id . ', item: ' . $potential_item );
                            }
                            return [ $ctx_id, $potential_item ];
                        }
                    }
                }
            } else {
                // Keyword template - direct match
                if ( count( $parts ) === 1 ) {
                    $potential_item = $parts[0];
                    
                    // Check if this item exists in the template
                    foreach ( $data['items'] as $item ) {
                        $item_name = is_array( $item ) ? $item['name'] : $item;
                        if ( sanitize_title( $item_name ) === $potential_item ) {
                            if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                                error_log( 'DPG: Found keyword match - ctx: ' . $ctx_id . ', item: ' . $potential_item );
                            }
                            return [ $ctx_id, $potential_item ];
                        }
                    }
                }
            }
        }
        
        return [ null, null ];
    }

    // Early HTTP status fix
    private static function set_200_status_early() {
        // Set 200 status immediately when we detect a valid dynamic page
        status_header(200);
        
        // Tell WordPress this is not a 404
        global $wp_query;
        $wp_query->is_404 = false;
        $wp_query->is_page = true;
        $wp_query->is_singular = true;
        
        // Remove any 404 template loading
        add_filter('404_template', '__return_false', 999);
    }

    private static function fix_http_status() {
        // Override the 404 status that WordPress sets by default
        status_header(200);
        
        // Also tell WordPress this is not a 404
        global $wp_query;
        $wp_query->is_404 = false;
        $wp_query->is_page = true;
        $wp_query->is_singular = true;
    }

    /* --------------------------------------------------------------------- */
    /*  SERVICE-AREA HANDLER                                                 */
    /* --------------------------------------------------------------------- */
    private static function handle_service_area_request( string $ctx_id, string $item_raw ) {
        $ctx_id     = sanitize_file_name( $ctx_id );
        $json_file  = DPG_DATA_DIR . "{$ctx_id}.json";

        global $wp_filesystem;
        if ( ! function_exists( 'WP_Filesystem' ) ) {
            require_once ABSPATH . 'wp-admin/includes/file.php';
        }
        WP_Filesystem();

        if ( ! $wp_filesystem->exists( $json_file ) ) {
            self::do_404();
        }

        $contents = $wp_filesystem->get_contents( $json_file );
        if ( false === $contents ) {
            self::do_404();
        }

        $cfg   = json_decode( $contents, true );
        $items = is_array( $cfg['items'] ?? null ) ? $cfg['items'] : [];

        // NEW: Fix HTTP status before rendering
        self::fix_http_status();

        self::render_dynamic_page( $items, $item_raw, $ctx_id, $cfg );
    }

    /* --------------------------------------------------------------------- */
    /*  KEYWORD HANDLER                                                      */
    /* --------------------------------------------------------------------- */
    private static function handle_keyword_request( string $item_raw ) {
        $json_file = DPG_DATA_DIR . 'keyword.json';

        global $wp_filesystem;
        if ( ! function_exists( 'WP_Filesystem' ) ) {
            require_once ABSPATH . 'wp-admin/includes/file.php';
        }
        WP_Filesystem();

        if ( ! $wp_filesystem->exists( $json_file ) ) {
            return;
        }

        $contents = $wp_filesystem->get_contents( $json_file );
        if ( false === $contents ) {
            return;
        }

        $cfg   = json_decode( $contents, true );
        $items = is_array( $cfg['items'] ?? null ) ? $cfg['items'] : [];

        // NEW: Fix HTTP status before rendering  
        self::fix_http_status();

        self::render_dynamic_page( $items, $item_raw, 'keyword', $cfg );
    }

    /* --------------------------------------------------------------------- */
    /*  DISABLE OTHER SEO PLUGINS                                            */
    /* --------------------------------------------------------------------- */
private static function disable_other_seo_plugins() {
    // Only disable on dynamic pages
    if (!self::is_dynamic_page_request()) {
        return;
    }
    
    // Disable Yoast SEO
    add_filter( 'wpseo_frontend_presenters', '__return_empty_array', 99 );
    add_filter( 'wpseo_opengraph_output', '__return_false', 99 );
    add_filter( 'wpseo_twitter_output', '__return_false', 99 );
    add_filter( 'wpseo_json_ld_output', '__return_false', 99 );
    
    // Remove Yoast's title filter
    global $wpseo_front;
    if ( isset( $wpseo_front ) && is_object( $wpseo_front ) ) {
        remove_filter( 'pre_get_document_title', array( $wpseo_front, 'title' ), 15 );
    }

    // Disable RankMath
    add_filter( 'rank_math/frontend/title', '__return_false', 99 );
    add_filter( 'rank_math/frontend/description', '__return_false', 99 );
    add_filter( 'rank_math/opengraph/facebook', '__return_false', 99 );
    add_filter( 'rank_math/opengraph/twitter', '__return_false', 99 );
    add_filter( 'rank_math/json_ld', '__return_empty_array', 99 );

    // Disable SEOPress
    add_filter( 'seopress_titles_title', '__return_false', 99 );
    add_filter( 'seopress_titles_desc', '__return_false', 99 );
    add_filter( 'seopress_social_og_title', '__return_false', 99 );
    add_filter( 'seopress_social_og_desc', '__return_false', 99 );
    add_filter( 'seopress_social_twitter_title', '__return_false', 99 );
    add_filter( 'seopress_social_twitter_desc', '__return_false', 99 );

    // Disable All in One SEO
    add_filter( 'aioseo_title', '__return_false', 99 );
    add_filter( 'aioseo_description', '__return_false', 99 );
    add_filter( 'aioseo_og_title', '__return_false', 99 );
    add_filter( 'aioseo_og_description', '__return_false', 99 );
    add_filter( 'aioseo_twitter_title', '__return_false', 99 );
    add_filter( 'aioseo_twitter_description', '__return_false', 99 );
}

    /* --------------------------------------------------------------------- */
    /*  CORE RENDERER – SMART WRAPPER / SEO / SOCIAL                         */
    /* --------------------------------------------------------------------- */
public static function render_dynamic_page( array $items, $raw_slug, $ctx_id, array $cfg ) {

    /* ---------- locate the requested item ---------- */
    $requested = sanitize_title( $raw_slug );
    $current   = null;

    foreach ( $items as $entry ) {
        $name = is_array( $entry ) ? $entry['name'] : $entry;
        if ( sanitize_title( $name ) === $requested ) {
            $current = $name;
            break;
        }
    }
    if ( ! $current ) {
        self::do_404();
    }

    // Disable other SEO plugins for our dynamic pages
    // (This will now only run on dynamic pages due to the check inside the method)
    self::disable_other_seo_plugins();

    /* ---------- template options ---------- */
    $page_id        = intval( $cfg['page_id']        ?? 0 );
    $template_html  =            $cfg['template_html'] ?? '';
    $include_header = ! empty(   $cfg['include_header'] );
    $include_footer = ! empty(   $cfg['include_footer'] );
    $seo_mode       =            $cfg['seo_mode']      ?? 'hardcoded';
    $hard_meta      = is_array(  $cfg['hard_meta']     ?? null ) ? $cfg['hard_meta'] : [];

    $GLOBALS['dpg_current_item'] = $current;
    $GLOBALS['dpg_current_slug'] = sanitize_title( $current );

    /* ---------- Get template content with proper structure ---------- */
    $template_content = self::get_complete_template_content( $cfg, $current, $include_header, $include_footer );

    /* ---------- Check if template contains full HTML document ---------- */
    $contains_shell = preg_match( '/<!doctype\s+html|<html\b|<head\b/i', $template_content );
    $wrap_document = ( ! $include_header && ! $contains_shell );

    $GLOBALS['dpg_wrap_document']      = $wrap_document;
    $GLOBALS['dpg_inline_head_tags']   = $wrap_document;

    /* ---------- SEO / SOCIAL setup for all cases ---------- */
    if ( $seo_mode === 'hardcoded' ) {

        $rep = [ '{item}' => $current ];

        global $dpg_seo_title,  $dpg_seo_description, $dpg_seo_keywords,
               $dpg_og_title,   $dpg_og_description,  $dpg_og_image,
               $dpg_tw_title,   $dpg_tw_description,  $dpg_tw_image;

        $dpg_seo_title       = strtr( $hard_meta['title']       ?? '', $rep );
        $dpg_seo_description = strtr( $hard_meta['description'] ?? '', $rep );
        $dpg_seo_keywords    = strtr( $hard_meta['keywords']    ?? '', $rep );

        $dpg_og_title       = strtr( $hard_meta['og_title']       ?? $dpg_seo_title,       $rep );
        $dpg_og_description = strtr( $hard_meta['og_description'] ?? $dpg_seo_description, $rep );
        $dpg_og_image       =            $hard_meta['og_image']    ?? '';

        $dpg_tw_title       = strtr( $hard_meta['twitter_title']       ?? $dpg_og_title,       $rep );
        $dpg_tw_description = strtr( $hard_meta['twitter_description'] ?? $dpg_og_description, $rep );
        $dpg_tw_image       =            $hard_meta['twitter_image']    ?? $dpg_og_image;

        // Always setup SEO hooks for header/footer mode or page-based templates
        if ( $include_header || $page_id ) {
            self::setup_seo_hooks();
        }
    }

    /* -----------------------------------------------------------------
     |  NEW: Treat this dynamic page as a real WP page (not a 404)
     |  so that get_header(), body_class(), etc. all see the correct
     |  page context and CSS selectors fire exactly as on your template.
     ----------------------------------------------------------------- */
    if ( $page_id ) {
        // Override the 404 status
        status_header( 200 );
        // Flip WP_Query->is_404 = false, set is_page, is_singular, queried_object, and $post
        self::setup_page_rendering( $page_id, $current );
    }

    /* ---------- output ---------- */
    if ( $page_id ) {
        // For page-based templates, let WordPress handle header/footer
        if ( $include_header ) {
            get_header();
        }

        echo $template_content;

        if ( $include_footer ) {
            get_footer();
        }
    } else {
        // Custom HTML templates
        if ( $wrap_document ) {
            self::print_document_open();
        }

        if ( $include_header ) {
            get_header();
        }

        echo $template_content;

        if ( $include_footer ) {
            get_footer();
        }

        if ( $wrap_document ) {
            self::print_document_close();
        }
    }

    exit;
}

    private static function get_complete_template_content( array $cfg, string $current, bool $include_header, bool $include_footer ): string {
        $page_id = intval( $cfg['page_id'] ?? $cfg['source_page_id'] ?? 0 );
        $template_html = $cfg['template_html'] ?? '';

        // 1) Custom HTML snippet
        if ( ! $page_id && $template_html ) {
            $content = str_replace( '{item}', esc_html( $current ), $template_html );
            
            // NEW: Process shortcodes in HTML templates
            $content = do_shortcode( $content );
            
            if ( ! $include_header && ! $include_footer ) {
                return self::wrap_content_with_page_structure( $content );
            }
            return $content;
        }

        // 2) Page-based template - use the working method from pro version
        if ( $page_id ) {
            $post = get_post( $page_id );
            if ( ! $post || $post->post_status !== 'publish' ) {
                return '<div class="dpg-error">Template page not found.</div>';
            }
            return self::get_complete_rendered_page_content( $post, $current, $include_header, $include_footer );
        }

        // 3) Fallback
        return '<div class="dpg-error">No HTML template defined.</div>';
    }

    private static function get_complete_rendered_page_content( WP_Post $post_obj, string $current, bool $include_header, bool $include_footer ) {
        global $wp_query, $post;

        // 1) Backup globals
        $orig_query = $wp_query;
        $orig_post  = $post;

        // 2) Set up a complete page simulation
        $wp_query = new WP_Query([
            'page_id'   => $post_obj->ID,
            'post_type' => 'page',
            'posts_per_page' => 1,
        ]);
        
        // Make this the main query
        $wp_query->queried_object = $post_obj;
        $wp_query->queried_object_id = $post_obj->ID;
        $wp_query->is_page = true;
        $wp_query->is_singular = true;
        $wp_query->is_404 = false;
        $wp_query->posts = [ $post_obj ];
        $wp_query->post_count = 1;
        $wp_query->found_posts = 1;
        
        $post = $post_obj;
        setup_postdata( $post );

        // 3) Inject our {item} placeholder replacement at priority 9
        $placeholder = function( $html ) use ( $current ) {
            return str_replace( '{item}', esc_html( $current ), $html );
        };
        add_filter( 'the_content', $placeholder, 9 );

        // 4) Buffer everything with proper main wrapper
        ob_start();

        // Get the post classes for proper styling
        $post_classes = implode( ' ', get_post_class( '', $post_obj->ID ) );

        // Add the main wrapper with proper classes
        echo '<main id="content" class="site-main ' . esc_attr( $post_classes ) . '">';
        echo '<div class="page-content">';
        
        // Get the content - this will trigger all page builder hooks
        if ( have_posts() ) {
            while ( have_posts() ) {
                the_post();
                the_content();
            }
        }
        
        echo '</div>'; // Close page-content
        echo '</main>'; // Close main

        $output = ob_get_clean();

        // 5) Cleanup: remove only our placeholder filter
        remove_filter( 'the_content', $placeholder, 9 );

        // 6) Restore globals
        wp_reset_postdata();
        $wp_query = $orig_query;
        $post = $orig_post;

        return $output;
    }

    private static function setup_page_rendering( $page_id, $current ) {
        global $wp_query, $post;
        
        // Get the page
        $page_post = get_post( $page_id );
        if ( ! $page_post || $page_post->post_status !== 'publish' ) {
            return;
        }
        
        // Set up WordPress to think we're viewing this page
        $wp_query->queried_object = $page_post;
        $wp_query->queried_object_id = $page_id;
        $wp_query->is_page = true;
        $wp_query->is_singular = true;
        $wp_query->is_404 = false;
        $wp_query->posts = [ $page_post ];
        $wp_query->post_count = 1;
        $wp_query->found_posts = 1;
        
        // Set the global post
        $post = $page_post;
        setup_postdata( $post );
        
        // UNIVERSAL SOLUTION: Simulate the actual page request to get ALL body classes
        add_filter( 'body_class', function( $classes ) use ( $page_id, $page_post ) {
            // Get the REAL body classes by simulating a proper page request
            $real_classes = self::get_real_page_body_classes( $page_id );
            
            if ( ! empty( $real_classes ) ) {
                // Use the real classes from the actual page
                $classes = array_unique( array_merge( $classes, $real_classes ) );
            } else {
                // Fallback: use WordPress post classes
                $page_classes = get_post_class( '', $page_id );
                $classes = array_unique( array_merge( $classes, $page_classes ) );
            }
            
            // Remove any 404 classes
            $classes = array_filter( $classes, function( $class ) {
                return $class !== 'error404';
            });
            
            return $classes;
        }, 999 );
        
        // Add our {item} replacement filter with high priority
        add_filter( 'the_content', function( $content ) use ( $current ) {
            return str_replace( '{item}', esc_html( $current ), $content );
        }, 5 );
        
        // Store that we're in DPG page rendering mode
        $GLOBALS['dpg_rendering_page'] = true;
    }

    /**
     * Get the real body classes by making an internal request to the actual page
     * This captures ALL body classes from any page builder or theme
     */
    private static function get_real_page_body_classes( $page_id ) {
        static $cached_classes = [];
        
        // Check cache first
        if ( isset( $cached_classes[ $page_id ] ) ) {
            return $cached_classes[ $page_id ];
        }
        
        // Get the page URL
        $page_url = get_permalink( $page_id );
        if ( ! $page_url ) {
            return [];
        }
        
        // Make an internal request to get the actual HTML
        $response = wp_remote_get( $page_url, [
            'timeout' => 10,
            'sslverify' => false,
            'headers' => [
                'User-Agent' => 'DPG-Internal-Request'
            ]
        ] );
        
        if ( is_wp_error( $response ) ) {
            return [];
        }
        
        $html = wp_remote_retrieve_body( $response );
        if ( empty( $html ) ) {
            return [];
        }
        
        // Extract body classes from the HTML
        $classes = [];
        if ( preg_match( '/<body[^>]*class=["\']([^"\']*)["\'][^>]*>/i', $html, $matches ) ) {
            $class_string = $matches[1];
            $classes = array_filter( explode( ' ', $class_string ) );
        }
        
        // Cache the result
        $cached_classes[ $page_id ] = $classes;
        
        return $classes;
    }

    /* --------------------------------------------------------------------- */
    /*  NEW METHOD: Wrap Content with Basic Page Structure                  */
    /* --------------------------------------------------------------------- */
    private static function wrap_content_with_page_structure( string $content ): string {
        return sprintf(
            '<div class="dpg-page-wrapper">
               <main id="content" class="site-main">
                 <div class="page-content">%s</div>
               </main>
             </div>',
            $content
        );
    }

    /* --------------------------------------------------------------------- */
    /*  REMOVE CONFLICTING TAGS                                              */
    /* --------------------------------------------------------------------- */
    /**
     * Start output buffering early and strip out any conflicting SEO/head tags,
     * then remove *only* the duplicate <title> occurrences (leaving the last).
     */
public static function remove_conflicting_tags() {
    // First check if this is actually a dynamic page request
    if (!self::is_dynamic_page_request()) {
        return; // Don't do anything for regular WordPress pages
    }
    
    // Only start output buffering for dynamic pages
    ob_start( function( $buffer ) {
        // 1) Yoast SEO comments & meta/scripts
        $buffer = preg_replace(
            '#<!-- This site is optimized with the Yoast SEO plugin.*?<!-- / Yoast SEO plugin\. -->#is',
            '',
            $buffer
        );
        $buffer = preg_replace(
            '#<meta[^>]*class=[\'"]yoast-seo-meta-tag[\'"][^>]*>#i',
            '',
            $buffer
        );
        $buffer = preg_replace(
            '#<script[^>]*class=[\'"]yoast-schema-graph[\'"][^>]*>.*?</script>#is',
            '',
            $buffer
        );

        // 2) Rank Math comments & tags
        $buffer = preg_replace(
            '#<!-- This site is optimized with the Rank Math.*?<!-- /Rank Math WordPress SEO plugin -->#is',
            '',
            $buffer
        );

        // 3) Core WP head tags: generator, shortlink, REST API, RSD, wlwmanifest
        $buffer = preg_replace('#<meta\s+name=(["\'])generator\1[^>]*>#i', '', $buffer);
        $buffer = preg_replace('#<link\s+rel=(["\'])shortlink\1[^>]*>#i',   '', $buffer);
        $buffer = preg_replace('#<link\s+rel=(["\'])(https?://api\.w\.org/)[^>]*>#i', '', $buffer);
        $buffer = preg_replace('#<link\s+rel=(["\'])EditURI\1[^>]*>#i',      '', $buffer);
        $buffer = preg_replace('#<link\s+rel=(["\'])wlwmanifest\1[^>]*>#i',  '', $buffer);

        // 4) Collapse multiple <title>…</title> down to *just* the last one
        if ( preg_match_all('#<title\b[^>]*>.*?</title>#is', $buffer, $matches) ) {
            $total = count( $matches[0] );
            if ( $total > 1 ) {
                for ( $i = 0; $i < $total - 1; $i++ ) {
                    // remove each duplicate *once*, in order
                    $pattern = '#' . preg_quote( $matches[0][$i], '#' ) . '#';
                    $buffer  = preg_replace( $pattern, '', $buffer, 1 );
                }
            }
        }

        return $buffer;
    } );
}

/**
 * 2. ADD NEW METHOD: Check if current request is for a dynamic page
 */
private static function is_dynamic_page_request() {
    // Method 1: Check WordPress query vars
    $ctx_id   = sanitize_key( get_query_var( 'dpg_ctx' ) );
    $item_raw = sanitize_text_field( get_query_var( 'dpg_item' ) );

    if ($ctx_id || $item_raw) {
        return true;
    }

    // Method 2: Check global query vars as fallback
    global $wp_query;
    if ( isset( $wp_query->query_vars['dpg_ctx'] ) || isset( $wp_query->query_vars['dpg_item'] ) ) {
        return true;
    }

    // Method 3: Manual URL parsing to detect dynamic pages
    $parsed_url = self::parse_request_url();
    if ($parsed_url) {
        list( $manual_ctx, $manual_item ) = self::manual_url_parse( $parsed_url );
        if ($manual_ctx || $manual_item) {
            return true;
        }
    }

    return false;
}

    /* --------------------------------------------------------------------- */
    /*  SETUP SEO HOOKS FOR HEADER/FOOTER MODE                              */
    /* --------------------------------------------------------------------- */
    private static function setup_seo_hooks() {
        global $dpg_seo_title;

        // Override document title with highest priority
        add_filter( 'pre_get_document_title', function() use ( $dpg_seo_title ) {
            return $dpg_seo_title;
        }, 999 );

        // Override wp_title as backup
        add_filter( 'wp_title', function() use ( $dpg_seo_title ) {
            return $dpg_seo_title;
        }, 999 );

        // Add our SEO tags early but after charset
        add_action( 'wp_head', [ __CLASS__, 'output_seo_tags' ], 2 );
    }

    /* --------------------------------------------------------------------- */
    /*  OUTPUT SEO TAGS                                                      */
    /* --------------------------------------------------------------------- */
    public static function output_seo_tags() {
        global $dpg_seo_title,  $dpg_seo_description, $dpg_seo_keywords,
               $dpg_og_title,   $dpg_og_description,  $dpg_og_image,
               $dpg_tw_title,   $dpg_tw_description,  $dpg_tw_image;

        // Only output if we have SEO data
        if ( empty( $dpg_seo_title ) && empty( $dpg_seo_description ) ) {
            return;
        }

        echo "\n<!-- This page is optimized with the Dynamic SEO plugin Version 1.3.7 https://dynamicseopro.com/-->\n";

        // Determine if we're in "wrapper" (inline) mode—only then
        // do we need to print viewport/robots (to avoid duplication).
        $inline = ! empty( $GLOBALS['dpg_inline_head_tags'] );
        if ( $inline ) {
            echo '<meta name="viewport" content="width=device-width, initial-scale=1">' . "\n";
            echo '<meta name="robots" content="index, follow, max-image-preview:large, max-snippet:-1, max-video-preview:-1" />' . "\n";
        }

        /* Standard SEO */
        if ( $dpg_seo_title ) {
            echo "<title>" . esc_html( $dpg_seo_title ) . "</title>\n";
        }
        if ( $dpg_seo_description ) {
            echo '<meta name="description" content="' .
                 esc_attr( $dpg_seo_description ) . "\" />\n";
        }
        if ( $dpg_seo_keywords ) {
            echo '<meta name="keywords" content="' .
                 esc_attr( $dpg_seo_keywords ) . "\" />\n";
        }

        /* Open Graph – optional */
        if ( $dpg_og_title || $dpg_og_description || $dpg_og_image ) {
            echo '<meta property="og:type" content="website" />' . "\n";
            if ( $dpg_og_title ) {
                echo '<meta property="og:title" content="' .
                     esc_attr( $dpg_og_title ) . "\" />\n";
            }
            if ( $dpg_og_description ) {
                echo '<meta property="og:description" content="' .
                     esc_attr( $dpg_og_description ) . "\" />\n";
            }
            if ( $dpg_og_image ) {
                echo '<meta property="og:image" content="' .
                     esc_url( $dpg_og_image ) . "\" />\n";
            }
        }

        /* Twitter Card – optional */
        if ( $dpg_tw_title || $dpg_tw_description || $dpg_tw_image ) {
            echo '<meta name="twitter:card" content="summary_large_image" />' . "\n";
            if ( $dpg_tw_title ) {
                echo '<meta name="twitter:title" content="' .
                     esc_attr( $dpg_tw_title ) . "\" />\n";
            }
            if ( $dpg_tw_description ) {
                echo '<meta name="twitter:description" content="' .
                     esc_attr( $dpg_tw_description ) . "\" />\n";
            }
            if ( $dpg_tw_image ) {
                echo '<meta name="twitter:image" content="' .
                     esc_url( $dpg_tw_image ) . "\" />\n";
            }
        }

        // Canonical URL
        echo '<link rel="canonical" href="' . self::get_current_url() . "\" />\n";

        echo "<!-- DPG SEO Tags End -->\n\n";
    }

    private static function maybe_output_custom_scripts() {
        // Only output custom scripts when in standalone document mode
        // (when include_header is false and we're wrapping the document)
        $custom_scripts = get_option( 'dpg_custom_scripts', '' );
        
        if ( ! empty( $custom_scripts ) && ! empty( $GLOBALS['dpg_wrap_document'] ) ) {
            echo "\n<!-- DPG Custom Scripts Start -->\n";
            echo $custom_scripts . "\n";
            echo "<!-- DPG Custom Scripts End -->\n\n";
        }
    }

    /* --------------------------------------------------------------------- */
    /*  SHORTCODE  [dpg_item field="slug|name"]                              */
    /* --------------------------------------------------------------------- */
    public static function shortcode_item( $atts ) {

        $atts = shortcode_atts( [ 'field' => 'name' ], $atts, 'dpg_item' );

        return empty( $GLOBALS['dpg_current_item'] ) ? '' :
            ( $atts['field'] === 'slug'
                ? esc_html( $GLOBALS['dpg_current_slug'] )
                : esc_html( $GLOBALS['dpg_current_item'] ) );
    }

    /* --------------------------------------------------------------------- */
    /*  SEO + SOCIAL TAGS (Updated method)                                   */
    /* --------------------------------------------------------------------- */
    private static function maybe_output_seo_tags() {

        if ( get_option( 'dpg_seo_mode', 'hardcoded' ) !== 'hardcoded' ) {
            return;
        }

        $inline = ! empty( $GLOBALS['dpg_inline_head_tags'] );

        if ( $inline ) {        // wrapper mode – print immediately
            self::output_seo_tags();
        }
        // For header/footer mode, SEO hooks are set up in setup_seo_hooks()
    }

    /* --------------------------------------------------------------------- */
    /*  DOCUMENT WRAPPER HELPERS                                             */
    /* --------------------------------------------------------------------- */
    private static function print_document_open() {

        $lang = esc_attr( get_bloginfo( 'language' ) ?: 'en' );
        echo "<!DOCTYPE html>\n";
        echo "<html lang=\"{$lang}\">\n<head>\n";
        echo '<meta charset="' . esc_attr( get_bloginfo( 'charset' ) ) . "\">\n";

        /* SEO + Social tags (prints immediately in wrapper mode) */
        self::maybe_output_seo_tags();

        /* Custom Scripts - Only in standalone document mode */
        self::maybe_output_custom_scripts();

        echo "</head>\n<body>\n";
    }

    private static function print_document_close() {
        echo "\n</body>\n</html>";
    }

    /* --------------------------------------------------------------------- */
    /*  404 HELPER                                                           */
    /* --------------------------------------------------------------------- */
    private static function do_404() {
        status_header( 404 );
        $tpl = get_404_template();
        if ( $tpl ) {
            include $tpl;
        } else {
            wp_die( esc_html__( '404 Not Found', 'dynamic-seo-pages') );
        }
        exit;
    }

    /* --------------------------------------------------------------------- */
    /*  SITEMAP GENERATION                                                   */
    /* --------------------------------------------------------------------- */
/**
 * SITEMAP GENERATION
 */
public static function generate_sitemap() {
    if ( ! is_dir( DPG_DATA_DIR ) ) {
        return;
    }

    global $wp_filesystem;
    if ( ! function_exists( 'WP_Filesystem' ) ) {
        require_once ABSPATH . 'wp-admin/includes/file.php';
    }
    WP_Filesystem();

    $global = floatval( get_option( 'dpg_global_score', 0.9 ) );
    $global = max( 0.01, min( 1.0, $global ) );
    $domain = untrailingslashit( home_url() );
    $date   = gmdate( 'c' );

    $xml  = '<?xml version="1.0" encoding="UTF-8"?>' . "\n";
    $xml .= '<urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9">' . "\n";

    foreach ( glob( DPG_DATA_DIR . '*.json' ) as $file ) {
        if ( ! $wp_filesystem->exists( $file ) ) {
            continue;
        }
        $ctx     = sanitize_file_name( basename( $file, '.json' ) );
        $content = $wp_filesystem->get_contents( $file );
        if ( false === $content ) {
            continue;
        }
        $cfg = json_decode( $content, true );
        if ( ! is_array( $cfg['items'] ?? null ) ) {
            continue;
        }

        foreach ( $cfg['items'] as $entry ) {
            $name = sanitize_text_field(
                is_array( $entry ) && isset( $entry['name'] )
                    ? $entry['name']
                    : (string) $entry
            );
            $slug = sanitize_title( $name );

            if ( substr( $ctx, -5 ) === '-area' ) {
                $base = sanitize_title( substr( $ctx, 0, -5 ) );
                $loc  = "{$domain}/{$base}-{$slug}/";
            } else {
                $loc = "{$domain}/{$slug}/";
            }

            $prio = is_array( $entry ) && isset( $entry['score'] ) && is_numeric( $entry['score'] )
                ? max( 0.01, min( 1.0, floatval( $entry['score'] ) ) )
                : $global;

            $xml .= self::sitemap_url( $loc, $date, $prio );
        }
    }

    $xml .= "</urlset>\n";

    // **Always write into your WP directory**, wherever it lives:
    $sitemap_file = untrailingslashit( ABSPATH ) . '/dynamic-pages-sitemap.xml';

    $wp_filesystem->put_contents( $sitemap_file, $xml, FS_CHMOD_FILE );
}



    /**
     * Format a single <url> entry for the sitemap.
     */
    private static function sitemap_url( string $loc, string $date, float $priority ): string {
        $priority = number_format( $priority, 2, '.', '' );
        return
              "  <url>\n"
            . "    <loc>"     . esc_url( $loc      ) . "</loc>\n"
            . "    <lastmod>" . esc_html( $date    ) . "</lastmod>\n"
            . "    <priority>{$priority}</priority>\n"
            . "  </url>\n";
    }
}

// Fire it up
DPG_Frontend::init();