<?php
// File: wp-content/plugins/dynamic-seo-pages/includes/class-dpg-metabox.php

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class DPG_Metabox {

    const META_KEY = 'dpg_context_id';

    public static function init() {
        add_action( 'add_meta_boxes',   [ __CLASS__, 'register_metabox' ] );
        add_action( 'save_post_page',   [ __CLASS__, 'save_metabox' ], 10, 2 );
    }

    /**
     * Register the metabox on the "page" post type.
     */
    public static function register_metabox() {
        add_meta_box(
            'dpg_page_template',
            __( 'Dynamic SEO Template', 'dynamic-seo-pages'),
            [ __CLASS__, 'render_metabox' ],
            'page',
            'side',
            'high'
        );
    }

    /**
     * Render the dropdown of available contexts (derived from data/*.json).
     *
     * @param WP_Post $post
     */
    public static function render_metabox( $post ) {
        wp_nonce_field( 'dpg_page_template', 'dpg_page_template_nonce' );

        // Find all JSON context files
        $json_files = glob( DPG_DATA_DIR . '*.json' );
        if ( empty( $json_files ) ) {
            echo '<p>' . esc_html__( 'No Dynamic SEO contexts defined.', 'dynamic-seo-pages') . '</p>';
            return;
        }

        // Get currently assigned context
        $current = get_post_meta( $post->ID, self::META_KEY, true );
        $current = sanitize_key( $current );

        echo '<select name="' . esc_attr( self::META_KEY ) . '" style="width:100%;">';
        echo '<option value="">' . esc_html__( '— Select a dynamic template —', 'dynamic-seo-pages') . '</option>';

        foreach ( $json_files as $file ) {
            // Derive clean context ID
            $ctx_id = sanitize_file_name( wp_basename( $file, '.json' ) );
            if ( '' === $ctx_id ) {
                continue;
            }

            // Look for matching PHP template
            $php_path = DPG_TMPL_DIR . $ctx_id . '.php';
            $human_name = '';

            if ( is_readable( $php_path ) ) {
                // Read header lines for "Template: <Name>"
                $lines = file( $php_path, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES );
                foreach ( $lines as $line ) {
                    if ( preg_match( '/^\s*\*\s*Template:\s*(.+)$/i', $line, $m ) ) {
                        $human_name = sanitize_text_field( trim( $m[1] ) );
                        break;
                    }
                }
            }

            // Fallback to ID → words
            if ( '' === $human_name ) {
                $human_name = ucwords( str_replace( '-', ' ', $ctx_id ) );
            }

            // Determine slug base for display (strip "-area" if present)
            if ( substr( $ctx_id, -5 ) === '-area' ) {
                $slug_base = sanitize_title_with_dashes( substr( $ctx_id, 0, -5 ) );
            } else {
                $slug_base = sanitize_title_with_dashes( $ctx_id );
            }

            // Prevent assignment if another page is already using this context
            $existing = get_posts( [
                'post_type'      => 'page',
                'meta_key'       => self::META_KEY,
                'meta_value'     => $ctx_id,
                'post__not_in'   => [ $post->ID ],
                'fields'         => 'ids',
                'numberposts'    => 1,
            ] );

            if ( empty( $existing ) || $current === $ctx_id ) {
                printf(
                    '<option value="%1$s"%2$s>%3$s &mdash; <code>%4$s</code></option>',
                    esc_attr( $ctx_id ),
                    selected( $current, $ctx_id, false ),
                    esc_html( $human_name ),
                    esc_html( $slug_base )
                );
            }
        }

        echo '</select>';

        if ( $current ) {
            echo '<p><em>' . esc_html__( 'Only one page may use each template.', 'dynamic-seo-pages') . '</em></p>';
        }
    }

    /**
     * Save the selected context ID into post meta.
     *
     * @param int     $post_id
     * @param WP_Post $post
     */
    public static function save_metabox( $post_id, $post ) {
        if ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) {
            return;
        }
        if ( empty( $_POST['dpg_page_template_nonce'] ) ) {
            return;
        }
        if ( ! wp_verify_nonce( wp_unslash( $_POST['dpg_page_template_nonce'] ), 'dpg_page_template' ) ) {
            return;
        }
        if ( ! current_user_can( 'edit_post', $post_id ) ) {
            return;
        }

        $new_ctx = isset( $_POST[ self::META_KEY ] ) ? sanitize_key( wp_unslash( $_POST[ self::META_KEY ] ) ) : '';

        if ( $new_ctx ) {
            // Ensure no other page uses it
            $existing = get_posts( [
                'post_type'      => 'page',
                'meta_key'       => self::META_KEY,
                'meta_value'     => $new_ctx,
                'post__not_in'   => [ $post_id ],
                'fields'         => 'ids',
                'numberposts'    => 1,
            ] );

            if ( empty( $existing ) ) {
                update_post_meta( $post_id, self::META_KEY, $new_ctx );
            }
            // else: silently refuse to overwrite
        } else {
            delete_post_meta( $post_id, self::META_KEY );
        }
    }
}

DPG_Metabox::init();