/**
 * Dynamic SEO Pages - Final Working Hover Menu JavaScript
 * Compatible with existing CSS - no inline style conflicts
 */

jQuery(document).ready(function($) {
    
    /**
     * Initialize Hover Menu - Works with existing CSS
     */
function initializeHoverMenu() {
    // Find the DPG menu item
    var dpgMenuItem = $('#adminmenu').find('a[href*="dpg-dashboard"]').closest('.menu-top');
    
    if (!dpgMenuItem.length) {
        // Try alternative selector
        dpgMenuItem = $('#adminmenu').find('a').filter(function() {
            return $(this).attr('href') && $(this).attr('href').indexOf('dpg-dashboard') !== -1;
        }).closest('.menu-top');
    }
    
    if (!dpgMenuItem.length) {
        return;
    }
    
    // CRITICAL: Add the dpg-menu-item class that CSS is targeting
    dpgMenuItem.addClass('dpg-menu-item');
    
    // IMPORTANT: Hide any existing WordPress submenu
    dpgMenuItem.find('.wp-submenu').hide();
    
    // Check if dpgMenuData is available
    if (typeof dpgMenuData === 'undefined') {
        createFallbackMenu(dpgMenuItem);
        return;
    }
    
    // Check if menu already exists
    if (dpgMenuItem.find('.dpg-hover-menu').length > 0) {
        return;
    }
    
    // ✅ WordPress.org compliant - only internal links in hover menu
    var hoverMenuHTML = '<div class="dpg-hover-menu">' +
        '<a href="' + dpgMenuData.urls.dashboard + '" class="dpg-menu-item">' +
            dpgMenuData.strings.dashboard +
        '</a>' +
        '<a href="' + dpgMenuData.urls.create + '" class="dpg-menu-item">' +
            dpgMenuData.strings.create_template +
        '</a>' +
        '<a href="' + dpgMenuData.urls.settings + '" class="dpg-menu-item">' +
            dpgMenuData.strings.settings +
        '</a>' +
        // ✅ External links removed from hover menu
    '</div>';
    
    // Append menu to the menu item
    dpgMenuItem.append(hoverMenuHTML);
    
    // Let CSS handle the hover - just ensure positioning
    dpgMenuItem.css('position', 'relative');
}

/**
 * ✅ WordPress.org compliant fallback menu
 */
function createFallbackMenu(dpgMenuItem) {
    if (dpgMenuItem.find('.dpg-hover-menu').length > 0) {
        return;
    }
    
    var baseUrl = window.location.origin + window.location.pathname.replace(/\/[^\/]*$/, '/admin.php');
    
    // ✅ Only internal WordPress admin links
    var hoverMenuHTML = '<div class="dpg-hover-menu">' +
        '<a href="' + baseUrl + '?page=dpg-dashboard" class="dpg-menu-item">Dashboard</a>' +
        '<a href="' + baseUrl + '?page=dpg-create" class="dpg-menu-item">Create Template</a>' +
        '<a href="' + baseUrl + '?page=dpg-settings" class="dpg-menu-item">Settings</a>' +
        // ✅ External links removed from fallback menu
    '</div>';
    
    dpgMenuItem.append(hoverMenuHTML);
    dpgMenuItem.css('position', 'relative');
}
    
    /**
     * Initialize Export Backup functionality
     */
    function initializeExportBackup() {
        var exportBtn = document.getElementById('dpg-export-backup');
        if (!exportBtn) {
            return;
        }
        
        exportBtn.addEventListener('click', function(e) {
            e.preventDefault();
            
            var form = document.createElement('form');
            form.method = 'POST';
            form.action = (typeof dpgMenuData !== 'undefined' && dpgMenuData.ajaxurl) ? 
                          dpgMenuData.ajaxurl : '/wp-admin/admin-ajax.php';
            
            var actionInput = document.createElement('input');
            actionInput.type = 'hidden';
            actionInput.name = 'action';
            actionInput.value = 'dpg_export_backup';
            
            var nonceInput = document.createElement('input');
            nonceInput.type = 'hidden';
            nonceInput.name = 'nonce';
            nonceInput.value = (typeof dpgMenuData !== 'undefined' && dpgMenuData.nonce) ? 
                               dpgMenuData.nonce : '';
            
            form.appendChild(actionInput);
            form.appendChild(nonceInput);
            document.body.appendChild(form);
            form.submit();
            document.body.removeChild(form);
        });
    }
    
    /**
     * Initialize Cache Clear functionality
     */
    function initializeCacheClear() {
        var clearCacheBtn = document.getElementById('dpg-clear-cache');
        if (!clearCacheBtn) {
            return;
        }
        
        clearCacheBtn.addEventListener('click', function(e) {
            e.preventDefault();
            
            var originalText = this.textContent;
            this.textContent = 'Clearing...';
            this.disabled = true;
            
            $.ajax({
                url: (typeof dpgMenuData !== 'undefined' && dpgMenuData.ajaxurl) ? 
                     dpgMenuData.ajaxurl : '/wp-admin/admin-ajax.php',
                type: 'POST',
                data: {
                    action: 'dpg_clear_cache',
                    nonce: (typeof dpgMenuData !== 'undefined') ? 
                           (dpgMenuData.cache_nonce || dpgMenuData.nonce) : ''
                },
                success: function(response) {
                    if (response.success) {
                        clearCacheBtn.textContent = '✅ Cleared!';
                        setTimeout(function() {
                            clearCacheBtn.textContent = originalText;
                            clearCacheBtn.disabled = false;
                        }, 2000);
                    } else {
                        clearCacheBtn.textContent = '❌ Failed';
                        setTimeout(function() {
                            clearCacheBtn.textContent = originalText;
                            clearCacheBtn.disabled = false;
                        }, 2000);
                    }
                },
                error: function() {
                    clearCacheBtn.textContent = '❌ Error';
                    setTimeout(function() {
                        clearCacheBtn.textContent = originalText;
                        clearCacheBtn.disabled = false;
                    }, 2000);
                }
            });
        });
    }
    
    /**
     * Initialize Form Enhancements
     */
    function initializeFormEnhancements() {
        $('form.dpg-form, form[id*="dpg"]').off('.dpg-form').on('submit.dpg-form', function() {
            var submitBtn = $(this).find('input[type="submit"], button[type="submit"]');
            if (submitBtn.length) {
                var originalValue = submitBtn.val() || submitBtn.text();
                submitBtn.prop('disabled', true);
                
                if (submitBtn.is('input')) {
                    submitBtn.val('Processing...');
                } else {
                    submitBtn.text('Processing...');
                }
                
                setTimeout(function() {
                    submitBtn.prop('disabled', false);
                    if (submitBtn.is('input')) {
                        submitBtn.val(originalValue);
                    } else {
                        submitBtn.text(originalValue);
                    }
                }, 10000);
            }
        });
        
        $('textarea.dpg-auto-resize').each(function() {
            this.style.height = 'auto';
            this.style.height = this.scrollHeight + 'px';
        }).off('.dpg-resize').on('input.dpg-resize', function() {
            this.style.height = 'auto';
            this.style.height = this.scrollHeight + 'px';
        });
    }
    
    /**
     * Initialize Tooltips
     */
    function initializeTooltips() {
        $('[data-dpg-tooltip]').off('.dpg-tooltip').on('mouseenter.dpg-tooltip', function() {
            var tooltip = $('<div class="dpg-tooltip">' + $(this).data('dpg-tooltip') + '</div>');
            $('body').append(tooltip);
            
            var offset = $(this).offset();
            tooltip.css({
                position: 'absolute',
                top: offset.top - tooltip.outerHeight() - 5,
                left: offset.left + ($(this).outerWidth() / 2) - (tooltip.outerWidth() / 2),
                zIndex: 999999,
                background: '#333',
                color: '#fff',
                padding: '8px 12px',
                borderRadius: '4px',
                fontSize: '12px',
                whiteSpace: 'nowrap'
            });
        }).on('mouseleave.dpg-tooltip', function() {
            $('.dpg-tooltip').remove();
        });
    }
    
    /**
     * Initialize Tab Functionality
     */
    function initializeTabs() {
        $('.dpg-tabs .tab-nav a').off('.dpg-tabs').on('click.dpg-tabs', function(e) {
            e.preventDefault();
            
            var targetTab = $(this).attr('href');
            var tabContainer = $(this).closest('.dpg-tabs');
            
            tabContainer.find('.tab-nav a').removeClass('active');
            $(this).addClass('active');
            
            tabContainer.find('.tab-content').removeClass('active');
            tabContainer.find(targetTab).addClass('active');
        });
    }
    
    /**
     * Initialize all admin functionality
     */
    function initializeAdmin() {
        // Initialize hover menu with slight delay
        setTimeout(function() {
            initializeHoverMenu();
        }, 500);
        
        // Initialize other functionality
        initializeExportBackup();
        initializeCacheClear();
        initializeFormEnhancements();
        initializeTooltips();
        initializeTabs();
        
        if (typeof dpgMenuData !== 'undefined' && dpgMenuData.debug) {
      
        }
    }
    
    // Initialize when DOM is ready
    initializeAdmin();
    
    // Fallback initialization
    $(window).on('load', function() {
        setTimeout(function() {
            if ($('#adminmenu .dpg-hover-menu').length === 0) {
       
                initializeHoverMenu();
            }
        }, 1000);
    });
    
    // Re-initialize on AJAX complete for dynamic content
    $(document).ajaxComplete(function(event, xhr, settings) {
        setTimeout(function() {
            initializeTooltips();
            initializeFormEnhancements();
            
            if (settings.url && settings.url.indexOf('admin-ajax.php') !== -1) {
                if ($('#adminmenu .dpg-hover-menu').length === 0) {
                    initializeHoverMenu();
                }
            }
        }, 500);
    });
    
});
