
/**
 * Reusable UI Components for DPG Forms
 * Provides interactive components and utilities
 * UPDATED: Enhanced integration with form validation system
 */
// File: wp-content/plugins/dynamic-seo-pages/includes/forms/assets/js/components.js
(function($) {
    'use strict';

    class DPGComponents {
        constructor() {
            this.activeModals = [];
            this.activeDropdowns = [];
            this.tooltipInstances = [];
            this.collapsedSections = new Set();

            // Configuration from dpgForm
            this.config = window.dpgForm || {};
            this.debug = this.config.debug === true;

            this.init();
        }

        init() {
            this.bindGlobalEvents();
            this.initializeComponents();
            
            if (this.debug) {
              
            }
        }

    bindGlobalEvents() {
        // REMOVED: Tab functionality that conflicts with server-side navigation
        // $(document).on('click', '.dpg-tab', (e) => {
        //     e.preventDefault();
        //     this.handleTabClick($(e.currentTarget));
        // });

        // Modal functionality
        $(document).on('click', '[data-modal-target]', (e) => {
            e.preventDefault();
            const target = $(e.currentTarget).data('modal-target');
            this.openModal(target);
        });

        $(document).on('click', '.dpg-modal-close, .dpg-modal-overlay', (e) => {
            if (e.target === e.currentTarget) {
                this.closeModal($(e.currentTarget).closest('.dpg-modal-overlay'));
            }
        });

        // Dropdown functionality
        $(document).on('click', '.dpg-dropdown-toggle', (e) => {
            e.preventDefault();
            e.stopPropagation();
            this.toggleDropdown($(e.currentTarget).closest('.dpg-dropdown'));
        });

        // Close dropdowns when clicking outside
        $(document).on('click', (e) => {
            if (!$(e.target).closest('.dpg-dropdown').length) {
                this.closeAllDropdowns();
            }
        });

            // Collapsible sections (enhanced for form sections)
            $(document).on('click', '.dpg-section-toggle', (e) => {
                e.preventDefault();
                this.toggleSection($(e.currentTarget));
            });

            // Notification dismissal
            $(document).on('click', '.dpg-notice-dismiss', (e) => {
                this.dismissNotice($(e.currentTarget).closest('.dpg-notice'));
            });

            // Enhanced tooltip functionality
            $(document).on('mouseenter', '.dpg-tooltip, [data-tooltip]', (e) => {
                this.showTooltip($(e.currentTarget));
            });

            $(document).on('mouseleave', '.dpg-tooltip, [data-tooltip]', (e) => {
                this.hideTooltip($(e.currentTarget));
            });

            // Keyboard navigation
            $(document).on('keydown', (e) => {
                this.handleKeyboardNavigation(e);
            });

            // Progress bar animations
            $(document).on('dpg:update-progress', '.dpg-progress', (e, percentage) => {
                this.updateProgress($(e.target), percentage);
            });

            // Form enhancement events
            $(document).on('dpg:form-loaded', '.dpg-form', (e) => {
                this.enhanceForm($(e.target));
            });

            // Character counter events
            $(document).on('input', 'input[maxlength], textarea[maxlength]', (e) => {
                this.updateCharacterCounter($(e.target));
            });

            // Loading state management
            $(document).on('dpg:show-loading', (e, element, text) => {
                this.showLoading($(element), text);
            });

            $(document).on('dpg:hide-loading', (e, element) => {
                this.hideLoading($(element));
            });
        }

        initializeComponents() {
            // Initialize existing components on page load
            this.initializeTabs();
            this.initializeProgress();
            this.initializeTooltips();
            this.initializeAlerts();
            this.initializeSkeletonLoaders();
            this.initializeSections();
            this.initializeCharacterCounters();
            
            // Auto-enhance forms
            if ($('.dpg-form').length > 0) {
                $('.dpg-form').each((i, form) => {
                    this.enhanceForm($(form));
                });
            }

            if (this.debug) {
               
            }
        }

        // Enhanced Tab Component
        handleTabClick($tab) {
            const tabId = $tab.data('tab');
            const $tabContainer = $tab.closest('.dpg-tabs');
            const $contentContainer = $tabContainer.siblings('.dpg-tab-content-container');

            // Update tab states
            $tabContainer.find('.dpg-tab').removeClass('active').attr('aria-selected', 'false');
            $tab.addClass('active').attr('aria-selected', 'true');

            // Update content visibility
            if ($contentContainer.length) {
                $contentContainer.find('.dpg-tab-content').removeClass('active').hide();
                $contentContainer.find(`[data-tab="${tabId}"].dpg-tab-content`).addClass('active').show();
            } else {
                // Fallback to sibling approach
                $tab.closest('.dpg-tabs').parent().find('.dpg-tab-content').removeClass('active').hide();
                $tab.closest('.dpg-tabs').parent().find(`[data-tab="${tabId}"].dpg-tab-content`).addClass('active').show();
            }

            // Trigger custom event
            $tab.trigger('dpg:tab-changed', [tabId]);

            // Announce to screen readers
            this.announceToScreenReader(`Switched to ${$tab.text()} tab`);
        }

        initializeTabs() {
            $('.dpg-tabs').each((i, tabContainer) => {
                const $container = $(tabContainer);
                const $tabs = $container.find('.dpg-tab');
                
                // Add ARIA attributes
                $tabs.attr('role', 'tab');
                $container.attr('role', 'tablist');
                
                // Activate first tab if none active
                if (!$container.find('.dpg-tab.active').length && $tabs.length) {
                    $tabs.first().trigger('click');
                }
            });
        }

        // Enhanced Modal Component
        openModal(modalId) {
            const $modal = $(`#${modalId}`);
            if (!$modal.length) {
                if (this.debug) {
                    console.warn('[DPG Components] Modal not found:', modalId);
                }
                return;
            }

            // Create overlay if it doesn't exist
            let $overlay = $modal.closest('.dpg-modal-overlay');
            if (!$overlay.length) {
                $modal.wrap('<div class="dpg-modal-overlay"></div>');
                $overlay = $modal.parent();
            }

            $overlay.addClass('open');
            this.activeModals.push($overlay);

            // Focus management and accessibility
            this.trapFocus($overlay);
            this.makeAccessible($modal);
            
            // Prevent body scroll
            $('body').addClass('dpg-modal-open');

            // Trigger custom event
            $overlay.trigger('dpg:modal-opened');

            // Announce to screen readers
            const title = $modal.find('.dpg-modal-title').text() || 'Modal';
            this.announceToScreenReader(`${title} dialog opened`);

            return $overlay;
        }

        closeModal($modal) {
            if (!$modal || !$modal.length) {
                $modal = $('.dpg-modal-overlay.open').last();
            }

            if (!$modal.length) return;

            $modal.removeClass('open');
            
            // Remove from active modals
            this.activeModals = this.activeModals.filter(modal => modal[0] !== $modal[0]);

            // Restore body scroll if no active modals
            if (this.activeModals.length === 0) {
                $('body').removeClass('dpg-modal-open');
            }

            // Return focus to trigger if available
            const $trigger = $(`[data-modal-target="${$modal.find('.dpg-modal').attr('id')}"]`);
            if ($trigger.length) {
                $trigger.focus();
            }

            // Trigger custom event
            $modal.trigger('dpg:modal-closed');

            // Announce to screen readers
            this.announceToScreenReader('Dialog closed');

            return $modal;
        }

        createModal(options = {}) {
            const defaults = {
                id: 'dpg-modal-' + Date.now(),
                title: 'Modal',
                content: '',
                size: 'medium',
                closeButton: true,
                backdrop: true,
                keyboard: true,
                footer: ''
            };

            const config = { ...defaults, ...options };

            const modalHtml = `
                <div class="dpg-modal-overlay" id="${config.id}-overlay">
                    <div class="dpg-modal dpg-modal-${config.size}" id="${config.id}" role="dialog" aria-modal="true" aria-labelledby="${config.id}-title">
                        <div class="dpg-modal-header">
                            <h3 class="dpg-modal-title" id="${config.id}-title">${config.title}</h3>
                            ${config.closeButton ? '<button type="button" class="dpg-modal-close" aria-label="Close dialog">&times;</button>' : ''}
                        </div>
                        <div class="dpg-modal-body">
                            ${config.content}
                        </div>
                        ${config.footer ? `<div class="dpg-modal-footer">${config.footer}</div>` : ''}
                    </div>
                </div>
            `;

            const $modal = $(modalHtml).appendTo('body');

            // Auto-open if specified
            if (config.autoOpen !== false) {
                setTimeout(() => this.openModal(config.id), 10);
            }

            return $modal;
        }

        // Enhanced Dropdown Component
        toggleDropdown($dropdown) {
            if ($dropdown.hasClass('open')) {
                this.closeDropdown($dropdown);
            } else {
                this.closeAllDropdowns();
                this.openDropdown($dropdown);
            }
        }

        openDropdown($dropdown) {
            $dropdown.addClass('open');
            this.activeDropdowns.push($dropdown);

            // Update ARIA attributes
            $dropdown.find('.dpg-dropdown-toggle').attr('aria-expanded', 'true');

            // Position dropdown
            this.positionDropdown($dropdown);

            // Focus first item
            const $firstItem = $dropdown.find('.dpg-dropdown-item').first();
            if ($firstItem.length) {
                $firstItem.focus();
            }

            // Trigger custom event
            $dropdown.trigger('dpg:dropdown-opened');
        }

        closeDropdown($dropdown) {
            $dropdown.removeClass('open');
            this.activeDropdowns = this.activeDropdowns.filter(dd => dd[0] !== $dropdown[0]);

            // Update ARIA attributes
            $dropdown.find('.dpg-dropdown-toggle').attr('aria-expanded', 'false');

            // Trigger custom event
            $dropdown.trigger('dpg:dropdown-closed');
        }

        closeAllDropdowns() {
            this.activeDropdowns.forEach($dropdown => {
                $dropdown.removeClass('open');
                $dropdown.find('.dpg-dropdown-toggle').attr('aria-expanded', 'false');
            });
            this.activeDropdowns = [];
        }

        positionDropdown($dropdown) {
            const $menu = $dropdown.find('.dpg-dropdown-menu');
            const $toggle = $dropdown.find('.dpg-dropdown-toggle');
            
            if (!$menu.length || !$toggle.length) return;

            const toggleRect = $toggle[0].getBoundingClientRect();
            const menuHeight = $menu.outerHeight();
            const windowHeight = $(window).height();
            const scrollTop = $(window).scrollTop();
            
            // Reset positioning
            $menu.removeClass('dropup');

            // Check if dropdown would go below viewport
            if (toggleRect.bottom + menuHeight > windowHeight && toggleRect.top > menuHeight) {
                $menu.addClass('dropup');
            }
        }

        // Enhanced Section Toggle Component
        toggleSection($toggle) {
            const $header = $toggle.closest('.dpg-section-header');
            const $section = $header.parent();
            const $content = $section.find('.dpg-section-content');
            const isExpanded = $toggle.attr('aria-expanded') === 'true';
            const sectionId = $section.data('section') || $section.attr('id');

            if (isExpanded) {
                $content.slideUp(300);
                $toggle.attr('aria-expanded', 'false');
                $section.addClass('dpg-section-collapsed');
                $toggle.find('.dashicons').removeClass('dashicons-arrow-up-alt2').addClass('dashicons-arrow-down-alt2');
                
                if (sectionId) {
                    this.collapsedSections.add(sectionId);
                    this.setData('collapsed_sections', Array.from(this.collapsedSections));
                }
            } else {
                $content.slideDown(300);
                $toggle.attr('aria-expanded', 'true');
                $section.removeClass('dpg-section-collapsed');
                $toggle.find('.dashicons').removeClass('dashicons-arrow-down-alt2').addClass('dashicons-arrow-up-alt2');
                
                if (sectionId) {
                    this.collapsedSections.delete(sectionId);
                    this.setData('collapsed_sections', Array.from(this.collapsedSections));
                }
            }

            // Trigger custom event
            $section.trigger('dpg:section-toggled', [!isExpanded]);

            // Announce to screen readers
            const sectionTitle = $header.find('.dpg-section-title').text();
            this.announceToScreenReader(`${sectionTitle} section ${isExpanded ? 'collapsed' : 'expanded'}`);
        }

        initializeSections() {
            // Restore collapsed state from storage
            const collapsedSections = this.getData('collapsed_sections') || [];
            this.collapsedSections = new Set(collapsedSections);

            $('.dpg-section-collapsible').each((i, section) => {
                const $section = $(section);
                const sectionId = $section.data('section') || $section.attr('id');
                const $toggle = $section.find('.dpg-section-toggle');
                const $content = $section.find('.dpg-section-content');

                // Set initial state
                if (sectionId && this.collapsedSections.has(sectionId)) {
                    $content.hide();
                    $toggle.attr('aria-expanded', 'false');
                    $section.addClass('dpg-section-collapsed');
                    $toggle.find('.dashicons').removeClass('dashicons-arrow-up-alt2').addClass('dashicons-arrow-down-alt2');
                } else {
                    $toggle.attr('aria-expanded', 'true');
                    $toggle.find('.dashicons').removeClass('dashicons-arrow-down-alt2').addClass('dashicons-arrow-up-alt2');
                }
            });
        }

        // Enhanced Progress Component
        updateProgress($progressBar, percentage) {
            const $bar = $progressBar.find('.dpg-progress-bar');
            const clampedPercentage = Math.max(0, Math.min(100, percentage));
            
            $bar.css('width', `${clampedPercentage}%`);
            $progressBar.attr('aria-valuenow', clampedPercentage);
            
            // Update percentage text if it exists
            const $percentage = $progressBar.find('.dpg-progress-percentage');
            if ($percentage.length) {
                $percentage.text(`${Math.round(clampedPercentage)}%`);
            }

            // Add status classes
            $progressBar.removeClass('dpg-progress-low dpg-progress-medium dpg-progress-high');
            if (clampedPercentage < 30) {
                $progressBar.addClass('dpg-progress-low');
            } else if (clampedPercentage < 70) {
                $progressBar.addClass('dpg-progress-medium');
            } else {
                $progressBar.addClass('dpg-progress-high');
            }

            // Trigger complete event at 100%
            if (clampedPercentage >= 100) {
                $progressBar.trigger('dpg:progress-complete');
                this.announceToScreenReader('Progress completed');
            }
        }

        initializeProgress() {
            $('.dpg-progress').each((i, el) => {
                const $progress = $(el);
                const initialValue = $progress.data('value') || 0;
                
                // Add ARIA attributes
                $progress.attr({
                    'role': 'progressbar',
                    'aria-valuemin': '0',
                    'aria-valuemax': '100',
                    'aria-valuenow': initialValue
                });
                
                this.updateProgress($progress, initialValue);
            });
        }

        animateProgress($progressBar, from, to, duration = 1000) {
            const startTime = Date.now();
            const animate = () => {
                const elapsed = Date.now() - startTime;
                const progress = Math.min(elapsed / duration, 1);
                const currentValue = from + (to - from) * this.easeOutCubic(progress);
                
                this.updateProgress($progressBar, currentValue);
                
                if (progress < 1) {
                    requestAnimationFrame(animate);
                }
            };
            animate();
        }

        easeOutCubic(t) {
            return 1 - Math.pow(1 - t, 3);
        }

        // Enhanced Tooltip Component
        showTooltip($trigger) {
            // Get content from various sources
            const content = $trigger.find('.dpg-tooltip-content').html() || 
                          $trigger.data('tooltip') || 
                          $trigger.attr('title') || 
                          $trigger.attr('aria-label');
                          
            if (!content) return;

            // Clear title to prevent browser tooltip
            if ($trigger.attr('title')) {
                $trigger.data('original-title', $trigger.attr('title')).removeAttr('title');
            }

            const tooltipId = 'tooltip-' + Date.now();
            const position = $trigger.data('tooltip-position') || 'top';
            
            const $tooltip = $(`
                <div class="dpg-tooltip-popup dpg-tooltip-${position}" id="${tooltipId}" role="tooltip">
                    <div class="dpg-tooltip-arrow"></div>
                    <div class="dpg-tooltip-inner">${content}</div>
                </div>
            `).appendTo('body');

            this.positionTooltip($trigger, $tooltip, position);
            
            // Show with animation
            setTimeout(() => $tooltip.addClass('show'), 10);
            
            $trigger.data('tooltip-id', tooltipId).attr('aria-describedby', tooltipId);
            this.tooltipInstances.push({ trigger: $trigger, tooltip: $tooltip });
        }

        hideTooltip($trigger) {
            const tooltipId = $trigger.data('tooltip-id');
            if (!tooltipId) return;

            const $tooltip = $(`#${tooltipId}`);
            $tooltip.removeClass('show');
            
            // Restore original title
            const originalTitle = $trigger.data('original-title');
            if (originalTitle) {
                $trigger.attr('title', originalTitle).removeData('original-title');
            }
            
            $trigger.removeData('tooltip-id').removeAttr('aria-describedby');
            
            setTimeout(() => {
                $tooltip.remove();
                this.tooltipInstances = this.tooltipInstances.filter(
                    instance => instance.trigger[0] !== $trigger[0]
                );
            }, 200);
        }

        positionTooltip($trigger, $tooltip, position = 'top') {
            const triggerRect = $trigger[0].getBoundingClientRect();
            const tooltipRect = $tooltip[0].getBoundingClientRect();
            const windowWidth = $(window).width();
            const windowHeight = $(window).height();
            const scrollTop = $(window).scrollTop();
            const scrollLeft = $(window).scrollLeft();

            let top, left;

            switch (position) {
                case 'bottom':
                    top = triggerRect.bottom + scrollTop + 10;
                    left = triggerRect.left + scrollLeft + (triggerRect.width - tooltipRect.width) / 2;
                    break;
                case 'left':
                    top = triggerRect.top + scrollTop + (triggerRect.height - tooltipRect.height) / 2;
                    left = triggerRect.left + scrollLeft - tooltipRect.width - 10;
                    break;
                case 'right':
                    top = triggerRect.top + scrollTop + (triggerRect.height - tooltipRect.height) / 2;
                    left = triggerRect.right + scrollLeft + 10;
                    break;
                default: // top
                    top = triggerRect.top + scrollTop - tooltipRect.height - 10;
                    left = triggerRect.left + scrollLeft + (triggerRect.width - tooltipRect.width) / 2;
            }

            // Adjust for viewport boundaries
            if (left < 10) left = 10;
            if (left + tooltipRect.width > windowWidth - 10) {
                left = windowWidth - tooltipRect.width - 10;
            }
            if (top < scrollTop + 10) {
                top = triggerRect.bottom + scrollTop + 10;
                $tooltip.removeClass('dpg-tooltip-top').addClass('dpg-tooltip-bottom');
            }
            if (top + tooltipRect.height > scrollTop + windowHeight - 10) {
                top = triggerRect.top + scrollTop - tooltipRect.height - 10;
                $tooltip.removeClass('dpg-tooltip-bottom').addClass('dpg-tooltip-top');
            }

            $tooltip.css({ top: top + 'px', left: left + 'px' });
        }

        initializeTooltips() {
            // Initialize tooltips with data-tooltip attribute
            $('[data-tooltip]').each((i, el) => {
                $(el).addClass('dpg-tooltip');
            });
        }

        // Enhanced Alert/Notice Component
        createAlert(options = {}) {
            const defaults = {
                type: 'info',
                title: '',
                message: '',
                dismissible: true,
                autoHide: false,
                hideDelay: 5000,
                container: 'body',
                actions: []
            };

            const config = { ...defaults, ...options };
            const alertId = 'alert-' + Date.now();

            const actionsHtml = config.actions.length ? 
                '<div class="dpg-alert-actions">' + 
                config.actions.map(action => `<button type="button" class="dpg-button dpg-button-small ${action.class || ''}" data-action="${action.action || ''}">${action.text}</button>`).join('') +
                '</div>' : '';

            const alertHtml = `
                <div class="dpg-alert dpg-alert-${config.type}" id="${alertId}" role="alert" aria-live="polite">
                    <div class="dpg-alert-icon">${this.getAlertIcon(config.type)}</div>
                    <div class="dpg-alert-content">
                        ${config.title ? `<div class="dpg-alert-title">${config.title}</div>` : ''}
                        <div class="dpg-alert-message">${config.message}</div>
                        ${actionsHtml}
                    </div>
                    ${config.dismissible ? '<button type="button" class="dpg-alert-dismiss" aria-label="Dismiss alert">&times;</button>' : ''}
                </div>
            `;

            const $alert = $(alertHtml);
            $(config.container).append($alert);

            // Bind action handlers
            $alert.find('[data-action]').on('click', (e) => {
                const action = $(e.target).data('action');
                if (config.onAction && typeof config.onAction === 'function') {
                    config.onAction(action, $alert);
                }
            });

            // Show with animation
            setTimeout(() => $alert.addClass('show'), 10);

            // Auto-hide if specified
            if (config.autoHide) {
                setTimeout(() => this.dismissAlert($alert), config.hideDelay);
            }

            return $alert;
        }

        dismissAlert($alert) {
            $alert.removeClass('show');
            setTimeout(() => $alert.remove(), 300);
        }

        dismissNotice($notice) {
            $notice.fadeOut(300, () => $notice.remove());
        }

        getAlertIcon(type) {
            const icons = {
                info: '<span class="dashicons dashicons-info"></span>',
                success: '<span class="dashicons dashicons-yes-alt"></span>',
                warning: '<span class="dashicons dashicons-warning"></span>',
                error: '<span class="dashicons dashicons-dismiss"></span>'
            };
            return icons[type] || icons.info;
        }

        initializeAlerts() {
            // Auto-hide alerts after specified time
            $('.dpg-alert[data-auto-hide]').each((i, el) => {
                const $alert = $(el);
                const delay = parseInt($alert.data('auto-hide')) || 5000;
                setTimeout(() => this.dismissAlert($alert), delay);
            });
        }

        // Character Counter Component
        initializeCharacterCounters() {
            $('input[maxlength], textarea[maxlength]').each((i, el) => {
                this.addCharacterCounter($(el));
            });
        }

        addCharacterCounter($field) {
            const maxLength = parseInt($field.attr('maxlength'));
            if (!maxLength || $field.siblings('.dpg-character-counter').length) return;

            const $counter = $('<div class="dpg-character-counter"></div>');
            $field.after($counter);

            this.updateCharacterCounter($field);
        }

        updateCharacterCounter($field) {
            const maxLength = parseInt($field.attr('maxlength'));
            if (!maxLength) return;

            const $counter = $field.siblings('.dpg-character-counter');
            if (!$counter.length) return;

            const currentLength = $field.val().length;
            const remaining = maxLength - currentLength;
            const percentage = (currentLength / maxLength) * 100;

            $counter.text(`${currentLength}/${maxLength}`)
                   .removeClass('dpg-counter-warning dpg-counter-error dpg-counter-ok')
                   .addClass(
                       percentage >= 100 ? 'dpg-counter-error' :
                       percentage >= 90 ? 'dpg-counter-warning' : 'dpg-counter-ok'
                   );

            // Update ARIA attributes
            $field.attr('aria-describedby', $counter.attr('id') || ($counter.attr('id', 'counter-' + Date.now()), $counter.attr('id')));
        }

        // Enhanced Form Enhancement
        enhanceForm($form) {
            if ($form.data('dpg-enhanced')) return;

            // Add loading states to submit buttons
            $form.on('submit', () => {
                const $submitBtn = $form.find('[type="submit"]');
                if ($submitBtn.length) {
                    this.showLoading($submitBtn, this.config.strings?.saving || 'Saving...');
                }
            });

            // Add character counters
            $form.find('input[maxlength], textarea[maxlength]').each((i, el) => {
                this.addCharacterCounter($(el));
            });

            // Add validation indicators
            $form.find('input[required], textarea[required], select[required]').each((i, el) => {
                this.addValidationIndicator($(el));
            });

            // Enhance accessibility
            this.makeFormAccessible($form);

            // Mark as enhanced
            $form.data('dpg-enhanced', true);

            if (this.debug) {
               
            }
        }

        addValidationIndicator($field) {
            if ($field.siblings('.dpg-validation-indicator').length) return;

            const $indicator = $('<div class="dpg-validation-indicator" aria-hidden="true"></div>');
            $field.after($indicator);

            const updateIndicator = () => {
                const isValid = $field[0].checkValidity && $field[0].checkValidity();
                const hasValue = $field.val().length > 0;
                
                $indicator.removeClass('valid invalid empty')
                         .addClass(
                             !hasValue ? 'empty' :
                             isValid ? 'valid' : 'invalid'
                         )
                         .html(
                             !hasValue ? '' :
                             isValid ? '<span class="dashicons dashicons-yes"></span>' : 
                             '<span class="dashicons dashicons-warning"></span>'
                         );
            };

            $field.on('input blur change', updateIndicator);
            updateIndicator();
        }

        makeFormAccessible($form) {
            // Add form landmarks
            if (!$form.attr('role')) {
                $form.attr('role', 'form');
            }

            // Enhance field labels
            $form.find('input, textarea, select').each((i, el) => {
                const $field = $(el);
                const $label = $form.find(`label[for="${$field.attr('id')}"]`);
                
                if (!$label.length) {
                    // Create label from nearby text if missing
                    const $wrapper = $field.closest('.dpg-field-wrapper');
                    const labelText = $wrapper.find('.dpg-field-label').text();
                    if (labelText && !$field.attr('aria-label')) {
                        $field.attr('aria-label', labelText);
                    }
                }
            });

            // Enhance error messages
            $form.find('.dpg-error-message').each((i, el) => {
                const $error = $(el);
                if (!$error.attr('id')) {
                    $error.attr('id', 'error-' + Date.now());
                }
                
                const $field = $error.siblings('input, textarea, select');
                if ($field.length) {
                    $field.attr('aria-describedby', $error.attr('id'));
                }
            });
        }

        // Skeleton Loader Component
        createSkeleton(options = {}) {
            const defaults = {
                lines: 3,
                width: '100%',
                height: '1em',
                className: 'dpg-skeleton-text',
                animated: true
            };

            const config = { ...defaults, ...options };
            const animatedClass = config.animated ? ' dpg-skeleton-animated' : '';
            
            const skeletonHtml = Array.from({ length: config.lines }, (_, i) => {
                const width = i === config.lines - 1 ? '75%' : config.width;
                return `<div class="dpg-skeleton ${config.className}${animatedClass}" style="width: ${width}; height: ${config.height};" aria-hidden="true"></div>`;
            }).join('');

            return $(skeletonHtml);
        }

        showSkeleton($container, options = {}) {
            const $skeleton = this.createSkeleton(options);
            $container.html($skeleton).addClass('dpg-loading-skeleton');
            
            // Add ARIA label for screen readers
            $container.attr('aria-label', 'Loading content...');
        }

        hideSkeleton($container, content) {
            $container.removeClass('dpg-loading-skeleton').removeAttr('aria-label');
            $container.fadeOut(200, () => {
                $container.html(content).fadeIn(200);
            });
        }

        initializeSkeletonLoaders() {
            // Auto-replace skeleton content when real content loads
            $('.dpg-skeleton-container[data-load-url]').each((i, el) => {
                const $container = $(el);
                const url = $container.data('load-url');
                
                // Show skeleton while loading
                this.showSkeleton($container);
                
                $.get(url)
                    .done(response => {
                        this.hideSkeleton($container, response);
                    })
                    .fail(() => {
                        this.hideSkeleton($container, '<p class="dpg-error">Failed to load content.</p>');
                    });
            });
        }

        // Enhanced Keyboard Navigation
        handleKeyboardNavigation(e) {
            // Escape key closes modals and dropdowns
            if (e.which === 27) { // ESC
                if (this.activeModals.length > 0) {
                    this.closeModal();
                    e.preventDefault();
                } else if (this.activeDropdowns.length > 0) {
                    this.closeAllDropdowns();
                    e.preventDefault();
                }
            }

            // Tab navigation in modals
            if (e.which === 9 && this.activeModals.length > 0) { // TAB
                this.handleModalTabNavigation(e);
            }

            // Arrow key navigation in dropdowns
            if (this.activeDropdowns.length > 0 && [38, 40, 13, 32].includes(e.which)) { // UP/DOWN/ENTER/SPACE
                this.handleDropdownNavigation(e);
            }

            // Arrow key navigation in tabs
            if ([37, 39].includes(e.which) && $(e.target).hasClass('dpg-tab')) { // LEFT/RIGHT
                this.handleTabNavigation(e);
            }
        }

        handleModalTabNavigation(e) {
            const $modal = this.activeModals[this.activeModals.length - 1];
            const $focusableElements = $modal.find('input:not([disabled]), textarea:not([disabled]), select:not([disabled]), button:not([disabled]), a[href], [tabindex]:not([tabindex="-1"]):not([disabled])');
            
            if ($focusableElements.length === 0) return;

            const firstElement = $focusableElements.first()[0];
            const lastElement = $focusableElements.last()[0];

            if (e.shiftKey) {
                if (document.activeElement === firstElement) {
                    e.preventDefault();
                    lastElement.focus();
                }
            } else {
                if (document.activeElement === lastElement) {
                    e.preventDefault();
                    firstElement.focus();
                }
            }
        }

        handleDropdownNavigation(e) {
            const $dropdown = this.activeDropdowns[this.activeDropdowns.length - 1];
            const $items = $dropdown.find('.dpg-dropdown-item:not([disabled])');
            const currentIndex = $items.index($items.filter(':focus'));

            if (e.which === 40) { // DOWN
                e.preventDefault();
                const newIndex = currentIndex < $items.length - 1 ? currentIndex + 1 : 0;
                $items.eq(newIndex).focus();
            } else if (e.which === 38) { // UP
                e.preventDefault();
                const newIndex = currentIndex > 0 ? currentIndex - 1 : $items.length - 1;
                $items.eq(newIndex).focus();
            } else if (e.which === 13 || e.which === 32) { // ENTER/SPACE
                e.preventDefault();
                if (currentIndex >= 0) {
                    $items.eq(currentIndex).trigger('click');
                }
            }
        }

        handleTabNavigation(e) {
            const $currentTab = $(e.target);
            const $tabContainer = $currentTab.closest('.dpg-tabs');
            const $tabs = $tabContainer.find('.dpg-tab:not([disabled])');
            const currentIndex = $tabs.index($currentTab);

            if (e.which === 39) { // RIGHT
                e.preventDefault();
                const newIndex = currentIndex < $tabs.length - 1 ? currentIndex + 1 : 0;
                $tabs.eq(newIndex).focus().trigger('click');
            } else if (e.which === 37) { // LEFT
                e.preventDefault();
                const newIndex = currentIndex > 0 ? currentIndex - 1 : $tabs.length - 1;
                $tabs.eq(newIndex).focus().trigger('click');
            }
        }

        trapFocus($container) {
            const $focusableElements = $container.find('input:not([disabled]), textarea:not([disabled]), select:not([disabled]), button:not([disabled]), a[href], [tabindex]:not([tabindex="-1"]):not([disabled])');
            if ($focusableElements.length > 0) {
                $focusableElements.first().focus();
            }
        }

        // Enhanced Loading States
        showLoading($element, text = 'Loading...') {
            if ($element.hasClass('dpg-loading')) return;

            const originalText = $element.is('button') ? $element.text() : '';
            const $spinner = $('<span class="dpg-spinner" aria-hidden="true"></span>');
            const $loadingText = $(`<span class="dpg-loading-text">${text}</span>`);
            
            $element.addClass('dpg-loading')
                   .data('original-text', originalText)
                   .prop('disabled', true);

            if ($element.is('button')) {
                $element.html('').append($spinner, ' ', $loadingText);
            } else {
                $element.append($('<div class="dpg-loading-overlay"></div>').append($spinner, $loadingText));
            }

            // Update ARIA attributes
            $element.attr('aria-busy', 'true');
        }

        hideLoading($element) {
            if (!$element.hasClass('dpg-loading')) return;

            const originalText = $element.data('original-text') || '';
            
            $element.removeClass('dpg-loading')
                   .prop('disabled', false)
                   .removeAttr('aria-busy');

            if ($element.is('button')) {
                $element.text(originalText);
            } else {
                $element.find('.dpg-loading-overlay').remove();
            }
        }

        // Animation Utilities
        fadeIn($element, duration = 300) {
            return new Promise(resolve => {
                $element.fadeIn(duration, resolve);
            });
        }

        fadeOut($element, duration = 300) {
            return new Promise(resolve => {
                $element.fadeOut(duration, resolve);
            });
        }

        slideDown($element, duration = 300) {
            return new Promise(resolve => {
                $element.slideDown(duration, resolve);
            });
        }

        slideUp($element, duration = 300) {
            return new Promise(resolve => {
                $element.slideUp(duration, resolve);
            });
        }

        // Utility Methods
        debounce(func, wait, immediate = false) {
            let timeout;
            return function executedFunction(...args) {
                const later = () => {
                    timeout = null;
                    if (!immediate) func.apply(this, args);
                };
                const callNow = immediate && !timeout;
                clearTimeout(timeout);
                timeout = setTimeout(later, wait);
                if (callNow) func.apply(this, args);
            };
        }

        throttle(func, limit) {
            let inThrottle;
            return function executedFunction(...args) {
                if (!inThrottle) {
                    func.apply(this, args);
                    inThrottle = true;
                    setTimeout(() => inThrottle = false, limit);
                }
            };
        }

        // Enhanced Data Storage (using sessionStorage safely)
        setData(key, value) {
            try {
                const data = {
                    value: value,
                    timestamp: Date.now()
                };
                sessionStorage.setItem(`dpg_${key}`, JSON.stringify(data));
                return true;
            } catch (e) {
                if (this.debug) {
                    console.warn('[DPG Components] Failed to store data:', e);
                }
                return false;
            }
        }

        getData(key, maxAge = 3600000) { // 1 hour default
            try {
                const stored = sessionStorage.getItem(`dpg_${key}`);
                if (!stored) return null;

                const data = JSON.parse(stored);
                if (Date.now() - data.timestamp > maxAge) {
                    this.removeData(key);
                    return null;
                }

                return data.value;
            } catch (e) {
                if (this.debug) {
                    console.warn('[DPG Components] Failed to retrieve data:', e);
                }
                return null;
            }
        }

        removeData(key) {
            try {
                sessionStorage.removeItem(`dpg_${key}`);
                return true;
            } catch (e) {
                if (this.debug) {
                    console.warn('[DPG Components] Failed to remove data:', e);
                }
                return false;
            }
        }

        clearAllData() {
            try {
                Object.keys(sessionStorage).forEach(key => {
                    if (key.startsWith('dpg_')) {
                        sessionStorage.removeItem(key);
                    }
                });
                return true;
            } catch (e) {
                if (this.debug) {
                    console.warn('[DPG Components] Failed to clear data:', e);
                }
                return false;
            }
        }

        // Responsive Utilities
        isMobile() {
            return window.innerWidth <= 768;
        }

        isTablet() {
            return window.innerWidth > 768 && window.innerWidth <= 1024;
        }

        isDesktop() {
            return window.innerWidth > 1024;
        }

        getBreakpoint() {
            if (this.isMobile()) return 'mobile';
            if (this.isTablet()) return 'tablet';
            return 'desktop';
        }

        onResize(callback) {
            const debouncedCallback = this.debounce(callback, 250);
            $(window).on('resize.dpg-components', debouncedCallback);
        }

        // Enhanced Accessibility Helpers
        announceToScreenReader(message) {
            const $announcement = $(`<div class="dpg-sr-only" aria-live="polite" role="status">${message}</div>`);
            $('body').append($announcement);
            setTimeout(() => $announcement.remove(), 2000);
        }

        makeAccessible($element) {
            // Add ARIA attributes for better accessibility
            if ($element.hasClass('dpg-tab')) {
                $element.attr({
                    'role': 'tab',
                    'aria-selected': $element.hasClass('active').toString(),
                    'tabindex': $element.hasClass('active') ? '0' : '-1'
                });
            }

            if ($element.hasClass('dpg-dropdown')) {
                const $toggle = $element.find('.dpg-dropdown-toggle');
                const $menu = $element.find('.dpg-dropdown-menu');
                
                $toggle.attr({
                    'aria-haspopup': 'true',
                    'aria-expanded': $element.hasClass('open').toString()
                });

                if ($menu.length && !$menu.attr('id')) {
                    const menuId = 'dropdown-menu-' + Date.now();
                    $menu.attr('id', menuId);
                    $toggle.attr('aria-controls', menuId);
                }
            }

            if ($element.hasClass('dpg-modal')) {
                $element.attr({
                    'role': 'dialog',
                    'aria-modal': 'true'
                });
                
                const $title = $element.find('.dpg-modal-title');
                if ($title.length && !$title.attr('id')) {
                    const titleId = 'modal-title-' + Date.now();
                    $title.attr('id', titleId);
                    $element.attr('aria-labelledby', titleId);
                }
            }

            if ($element.hasClass('dpg-alert')) {
                $element.attr({
                    'role': 'alert',
                    'aria-live': 'polite'
                });
            }
        }

        // Performance Monitoring
        measurePerformance(name, fn) {
            if (!this.debug) return fn();
            
            const start = performance.now();
            const result = fn();
            const end = performance.now();
            
       
            return result;
        }

        // Event System
        on(event, callback) {
            $(document).on(`dpg:${event}`, callback);
        }

        off(event, callback) {
            $(document).off(`dpg:${event}`, callback);
        }

        trigger(event, data) {
            $(document).trigger(`dpg:${event}`, data);
        }

        // Component State Management
        getComponentState(component) {
            const states = {
                modals: this.activeModals.map($modal => $modal.attr('id')),
                dropdowns: this.activeDropdowns.map($dropdown => $dropdown.attr('id')),
                tooltips: this.tooltipInstances.length,
                collapsedSections: Array.from(this.collapsedSections)
            };
            
            return component ? states[component] : states;
        }

        // Cleanup and Destroy
        destroy() {
            if (this.debug) {
           
            }

            // Close all active components
            this.closeAllDropdowns();
            this.activeModals.forEach($modal => this.closeModal($modal));
            
            // Remove all tooltips
            this.tooltipInstances.forEach(({ tooltip }) => tooltip.remove());
            this.tooltipInstances = [];
            
            // Remove event listeners
            $(document).off('.dpg-components');
            $(window).off('.dpg-components');
            
            // Clear data storage
            this.clearAllData();
            
            // Reset state
            this.activeModals = [];
            this.activeDropdowns = [];
            this.collapsedSections.clear();

            if (this.debug) {
              
            }
        }

        // Public API
        getVersion() {
            return '2.0.0';
        }

        getActiveComponents() {
            return {
                modals: this.activeModals.length,
                dropdowns: this.activeDropdowns.length,
                tooltips: this.tooltipInstances.length,
                collapsedSections: this.collapsedSections.size
            };
        }

        getConfig() {
            return this.config;
        }

        // Validation Integration
        validateForm($form) {
            let isValid = true;
            const errors = [];

            $form.find('input[required], textarea[required], select[required]').each((i, el) => {
                const $field = $(el);
                if (!$field[0].checkValidity || !$field[0].checkValidity()) {
                    isValid = false;
                    errors.push({
                        field: $field,
                        message: $field[0].validationMessage || 'This field is required'
                    });
                }
            });

            if (!isValid) {
                // Highlight errors
                errors.forEach(({ field, message }) => {
                    this.showFieldError(field, message);
                });

                // Focus first error
                if (errors.length > 0) {
                    errors[0].field.focus();
                }
            }

            return { isValid, errors };
        }

        showFieldError($field, message) {
            $field.addClass('dpg-field-error');
            
            let $error = $field.siblings('.dpg-field-error-message');
            if (!$error.length) {
                $error = $('<div class="dpg-field-error-message" role="alert"></div>');
                $field.after($error);
            }
            
            $error.text(message);
            $field.attr('aria-describedby', $error.attr('id') || ($error.attr('id', 'error-' + Date.now()), $error.attr('id')));
        }

        clearFieldError($field) {
            $field.removeClass('dpg-field-error');
            $field.siblings('.dpg-field-error-message').remove();
            $field.removeAttr('aria-describedby');
        }
    }

    // Auto-initialize when DOM is ready
    $(document).ready(function() {
        // Initialize components
        window.dpgComponents = new DPGComponents();
        
        // Make components available globally
        if (typeof window.dpgForm !== 'undefined') {
            window.dpgForm.components = window.dpgComponents;
        }

        // Responsive behavior
        window.dpgComponents.onResize(() => {
            // Reposition active dropdowns
            window.dpgComponents.activeDropdowns.forEach($dropdown => {
                window.dpgComponents.positionDropdown($dropdown);
            });

            // Reposition tooltips
            window.dpgComponents.tooltipInstances.forEach(({ trigger, tooltip }) => {
                window.dpgComponents.positionTooltip(trigger, tooltip);
            });
        });

        // Integration with form validation system
        $(document).on('dpg:validation-error', (e, field, message) => {
            window.dpgComponents.showFieldError($(field), message);
        });

        $(document).on('dpg:validation-success', (e, field) => {
            window.dpgComponents.clearFieldError($(field));
        });

        // Auto-enhance new forms
        $(document).on('DOMNodeInserted', '.dpg-form', function() {
            window.dpgComponents.enhanceForm($(this));
        });

        if (window.dpgComponents.debug) {
       
        }
    });

    // Cleanup on page unload
    $(window).on('beforeunload', function() {
        if (window.dpgComponents && window.dpgComponents.destroy) {
            window.dpgComponents.destroy();
        }
    });

    // Export for use in other modules
    if (typeof module !== 'undefined' && module.exports) {
        module.exports = DPGComponents;
    } else if (typeof window !== 'undefined') {
        window.DPGComponents = DPGComponents;
    }

})(jQuery);