/**
 * UNIFIED SEO Analysis for DPG Templates - WordPress Compliant
 * Handles both page templates and custom HTML with intelligent fallbacks
 * 
 * @file This file defines the DPGSEOAnalyzer class for WordPress SEO analysis
 * @author DPG Team
 * @since 1.0.0
 */

/* global jQuery, wp, window, document, dpgForm */

( function( $, wp ) {
	'use strict';

	const { __, sprintf } = wp && wp.i18n ? wp.i18n : {
		__: function( text ) {
			return text;
		},
		sprintf: function( format ) {
			var args = Array.prototype.slice.call( arguments, 1 );
			var i = 0;
			return format.replace( /%[sd]/g, function() {
				return args[ i++ ] || '';
			} );
		}
	};

	/**
	 * DPG SEO Analyzer Class
	 * 
	 * @constructor
	 * @param {Object} options Configuration options
	 */
	function DPGSEOAnalyzer( options ) {
		this.options = {
			container: '.dpg-seo-analysis',
			focusKeyphraseField: '#dpg_focus_keyphrase',
			titleField: '#dpg_meta_title',
			descriptionField: '#dpg_meta_description',
			contentField: '#dpg_template_html',
			itemsField: '#dpg_items',
			sourcePageField: '#dpg_source_page_id',
			templateSourceField: '#dpg_template_source',
            scoreField:          '#dpg_seo_score',
			debounceDelay: 1500,
			enabled: true
		};

		// Merge with provided options
		if ( options && typeof options === 'object' ) {
			var key;
			for ( key in options ) {
				if ( options.hasOwnProperty( key ) ) {
					this.options[ key ] = options[ key ];
				}
			}
		}

		// Get configuration from dpgForm (set by PHP localization)
		var config = window.dpgForm && window.dpgForm.seoAnalyzer ? window.dpgForm.seoAnalyzer : {};
		if ( config.options ) {
			for ( key in config.options ) {
				if ( config.options.hasOwnProperty( key ) ) {
					this.options[ key ] = config.options[ key ];
				}
			}
		}

		this.settings = config.settings || {
			minContentLength: 10,
			maxContentLength: 10000,
			titleLength: { min: 30, max: 60 },
			descriptionLength: { min: 120, max: 160 }
		};

		this.debug = window.dpgForm && window.dpgForm.debug === true;
		this.sampleItem = 'Example Item';
		this.debounceTimer = null;
		this.analysisInProgress = false;
		
		// Cache for REST API availability and page content
		this.restAPIAvailable = null;
		this.restAPICheck = null;
		this.pageContentCache = new Map();
		
		if ( this.options.enabled ) {
			this.init();
		} else {
			this.showDisabledMessage();
		}
	}

	/**
	 * Initialize the analyzer
	 */
	DPGSEOAnalyzer.prototype.init = function() {
		if ( ! $( this.options.container ).length ) {
			if ( this.debug ) {

			}
			return;
		}

		this.bindEvents();
		this.setupTabs();
		this.loadSampleItem();
		
		if ( this.options.autoRun !== false ) {
			this.runInitialAnalysis();
		}

		if ( this.debug ) {
	
		}
	};

	/**
	 * Bind event handlers
	 */
	DPGSEOAnalyzer.prototype.bindEvents = function() {
		var self = this;
		var fields = [
			this.options.focusKeyphraseField,
			this.options.titleField,
			this.options.descriptionField,
			this.options.contentField,
			this.options.itemsField,
			this.options.sourcePageField,
			this.options.templateSourceField
		];

		fields.forEach( function( sel ) {
			$( document ).on( 'input.dpgseo keyup.dpgseo paste.dpgseo change.dpgseo', sel, function() {
				self.debounceAnalysis();
			} );
		} );

		$( document ).on( 'click.dpgseo', '.dpg-seo-tab', function( e ) {
			e.preventDefault();
			self.switchTab( $( e.currentTarget ).data( 'tab' ) );
		} );

		$( document ).on( 'click.dpgseo', '.dpg-seo-check-expandable', function( e ) {
			e.preventDefault();
			self.toggleCheckDetails( $( e.currentTarget ) );
		} );

		$( document ).on( 'change.dpgseo', this.options.templateSourceField, function() {
			// Clear page content cache when source changes
			self.pageContentCache.clear();
			setTimeout( function() {
				self.debounceAnalysis();
			}, 100 );
		} );

		if ( this.debug ) {

		}
	};

	/**
	 * Load sample item from items list
	 */
	DPGSEOAnalyzer.prototype.loadSampleItem = function() {
		var raw = $( this.options.itemsField ).val() || '';
		var lines = raw.split( '\n' ).map( function( l ) {
			return l.trim();
		} ).filter( Boolean );
		
		if ( lines.length ) {
			var first = lines[ 0 ];
			this.sampleItem = first.includes( '|' )
				? first.split( '|', 2 )[ 0 ].trim()
				: first;
		}
		
		if ( ! this.sampleItem || this.sampleItem.length < 2 ) {
			this.sampleItem = 'Example Item';
		}

		if ( this.debug ) {
		
		}
	};

	/**
	 * Debounce analysis execution
	 */
	DPGSEOAnalyzer.prototype.debounceAnalysis = function() {
		var self = this;
		
		if ( ! this.options.enabled ) {
			return;
		}
		
		clearTimeout( this.debounceTimer );
		this.debounceTimer = setTimeout( function() {
			self.runAnalysis();
		}, this.options.debounceDelay );
	};

	/**
	 * Run initial analysis with delay
	 */
	DPGSEOAnalyzer.prototype.runInitialAnalysis = function() {
		var self = this;
		
		if ( ! this.options.enabled ) {
			return;
		}
		
		setTimeout( function() {
			self.runAnalysis();
		}, 500 );
	};

	/**
	 * Main analysis execution
	 */
	DPGSEOAnalyzer.prototype.runAnalysis = function() {
		var self = this;
		var $container = $( this.options.container );
		
		if ( this.analysisInProgress || ! this.options.enabled ) {
			return;
		}

		if ( ! $container.length ) {
			return;
		}

		this.analysisInProgress = true;
		
		try {
			this.showLoadingState();
			
			var data = this.collectAnalysisData();
			
			if ( ! this.hasMinimumData( data ) ) {
				this.showEmptyState();
				return;
			}
			
			this.performUnifiedAnalysis( data ).then( function( results ) {
				self.updateAnalysisDisplay( results );
				
				if ( self.debug ) {
		
				}
			} ).catch( function( err ) {
				self.showError( err );
				if ( self.debug ) {
					console.error( '[DPG SEO] Analysis error:', err );
				}
			} ).finally( function() {
				self.analysisInProgress = false;
			} );
			
		} catch ( err ) {
			this.showError( err );
			if ( this.debug ) {
				console.error( '[DPG SEO] Analysis error:', err );
			}
			this.analysisInProgress = false;
		}
	};

	/**
	 * Unified analysis system
	 * 
	 * @param {Object} data Analysis data
	 * @return {Promise} Analysis results
	 */

DPGSEOAnalyzer.prototype.performUnifiedAnalysis = function( data ) {
    var self = this;
    var templateSource = data.template_source;
    var templateType = data.template_type || 'service-area';
    
    if ( templateSource === 'page' && data.source_page_id ) {
        // WordPress-compliant page content retrieval with multiple fallbacks
        return this.getPageContentViaWordPress( data.source_page_id ).then( function( result ) {
            if ( result.success && result.content && result.content.length > 50 ) {
                // SUCCESS: Set content and run IDENTICAL analysis to HTML
                data.content = result.content;
                
                // Use page data if form fields are empty (WordPress standard)
                if ( ! data.title && result.title ) {
                    data.title = result.title;
                }
                if ( ! data.description && result.excerpt ) {
                    data.description = result.excerpt;
                }
                
                // Run IDENTICAL comprehensive analysis to custom HTML templates
                return self.clientSideAnalysis( data );
                
            } else {
                // WordPress-compliant fallback with better scoring
                return self.createWordPressCompliantPartialAnalysis( data, result.error || 'Content unavailable' );
            }
        }).catch( function( error ) {
            return self.createWordPressCompliantPartialAnalysis( data, error.message );
        });
    }
    
    // Custom template: Always use full client-side analysis
    return Promise.resolve( this.clientSideAnalysis( data ) );
};

DPGSEOAnalyzer.prototype.getNonce = function() {
    var nonce = '';
    
    // Method 1: Try dpgCreateForm (Create form)
    if ( window.dpgCreateForm && window.dpgCreateForm.nonces ) {
        nonce = window.dpgCreateForm.nonces.page_data || window.dpgCreateForm.nonces.form || '';
    }
    
    // Method 2: Try dpgEditForm (Edit form)
    if ( !nonce && window.dpgEditForm && window.dpgEditForm.nonces ) {
        nonce = window.dpgEditForm.nonces.page_data || window.dpgEditForm.nonces.form || '';
    }
    
    // Method 3: Try global dpgForm
    if ( !nonce && window.dpgForm && window.dpgForm.nonces ) {
        nonce = window.dpgForm.nonces.page_data || window.dpgForm.nonces.form || window.dpgForm.nonce || '';
    }
    
    // Method 4: Fallback to form fields
    if ( !nonce ) {
        var nonceField = document.querySelector( 'input[name="_wpnonce"]' );
        if ( nonceField ) {
            nonce = nonceField.value;
        }
    }
    
    // Method 5: Another fallback
    if ( !nonce ) {
        var dpgNonceField = document.querySelector( 'input[name="dpg_nonce"]' );
        if ( dpgNonceField ) {
            nonce = dpgNonceField.value;
        }
    }
    
    return nonce;
};

// 2. ADD: Enhanced AJAX URL retrieval
DPGSEOAnalyzer.prototype.getAjaxUrl = function() {
    if ( window.dpgCreateForm && window.dpgCreateForm.ajaxUrl ) {
        return window.dpgCreateForm.ajaxUrl;
    }
    if ( window.dpgEditForm && window.dpgEditForm.ajaxUrl ) {
        return window.dpgEditForm.ajaxUrl;
    }
    if ( window.dpgForm && window.dpgForm.ajaxUrl ) {
        return window.dpgForm.ajaxUrl;
    }
    return '/wp-admin/admin-ajax.php';
};

// 3. ADD: WordPress REST API support (missing from free version)
DPGSEOAnalyzer.prototype.fetchPageViaWordPressRest = function (pageId) {
    const self = this;
    const cacheKey = 'wp_rest_' + pageId;

    if (this.pageContentCache.has(cacheKey)) {
        return Promise.resolve(this.pageContentCache.get(cacheKey));
    }

    const url = '/wp-json/wp/v2/pages/' + pageId + '?context=edit&_embed';

    return new Promise(function (resolve, reject) {
        jQuery.getJSON(url)
              .done(function (json) {
                  const html = json?.content?.rendered ?? '';
                  if (html.length > 50) {
                      const result = {
                          success : true,
                          content : html,
                          title   : json?.title?.rendered ?? '',
                          excerpt : self.stripHtml(json?.excerpt?.rendered ?? ''),
                          method  : 'wordpress_rest'
                      };
                      self.pageContentCache.set(cacheKey, result);
                      resolve(result);
                  } else {
                      reject(new Error('REST returned empty content'));
                  }
              })
              .fail(function (xhr, status, error) {
                  reject(new Error('REST request failed: ' + status + ' – ' + error));
              });
    });
};

DPGSEOAnalyzer.prototype.getPageContentViaWordPress = function (pageId) {
    const self = this;
    const cacheKey = 'wp_page_' + pageId;

    if (this.pageContentCache.has(cacheKey)) {
        return Promise.resolve(this.pageContentCache.get(cacheKey));
    }

    // Priority 1: Embedded JSON in the edit screen
    return this.fetchPageViaWordPressEmbedded(pageId)
        .catch(() => ({ success: false }))

    // Priority 2: Fallback to admin-ajax.php
        .then(function (res) {
            if (res.success && res.content && res.content.length > 50) {
                self.pageContentCache.set(cacheKey, res);
                return res;
            }
            return self.fetchPageViaWordPressAjax(pageId)
                       .catch(() => ({ success: false }));
        })

    // Priority 3: Last resort - WP REST API
        .then(function (res) {
            if (res.success && res.content && res.content.length > 50) {
                self.pageContentCache.set(cacheKey, res);
                return res;
            }
            return self.fetchPageViaWordPressRest(pageId);
        })

    // Final bookkeeping
        .then(function (res) {
            self.pageContentCache.set(cacheKey, res);
            return res;
        })
        .catch(function (err) {
            const fail = { success: false, error: err.message, method: 'all_failed' };
            self.pageContentCache.set(cacheKey, fail);
            return fail;
        });
};








/**
 * WORDPRESS-COMPLIANT: Enhanced embedded data fetching
 */
DPGSEOAnalyzer.prototype.fetchPageViaWordPressEmbedded = function( pageId ) {
    var self = this;
    var dataId = 'dpg-page-data-' + pageId;
    var pageDataElement = document.getElementById( dataId );

    if ( self.debug ) {

    }

    if ( pageDataElement ) {
        try {
            var pageData = JSON.parse( pageDataElement.textContent );

            if ( self.debug ) {
       
            }

            // Validate content quality for FULL analysis
            if ( ! pageData.content || pageData.content.length < 50 ) {
                throw new Error( 'WordPress embedded content insufficient for full analysis (length: ' + (pageData.content ? pageData.content.length : 0) + ')' );
            }

            return Promise.resolve( {
                success: true,
                content: pageData.content,
                title: pageData.title || '',
                excerpt: pageData.excerpt || '',
                method: 'wordpress_embedded',
                debug: pageData
            } );
        } catch ( e ) {
            if ( self.debug ) {
             
            }
            return Promise.reject( new Error( 'WordPress embedded parsing failed: ' + e.message ) );
        }
    }

    return Promise.reject( new Error( 'WordPress embedded data element not found' ) );
};

/**
 * WORDPRESS-COMPLIANT: AJAX page data fetching using WordPress standards
 */
DPGSEOAnalyzer.prototype.fetchPageViaWordPressAjax = function( pageId ) {
    var self = this;

    return new Promise( function( resolve, reject ) {
        
        if ( typeof jQuery === 'undefined' ) {
            reject( new Error( 'WordPress jQuery not available' ) );
            return;
        }

        var nonce = self.getNonce();
        var ajaxUrl = self.getAjaxUrl();

        jQuery.ajax( {
            url: ajaxUrl,
            type: 'GET',
            data: {
                action: 'dpg_get_page_data',
                id: pageId,
                _wpnonce: nonce,        // Primary nonce field
                nonce: nonce,           // Backup nonce field
                security: nonce,        // Some plugins expect this field name
                _ajax_nonce: nonce      // Another common nonce field name
            },
            timeout: 15000,
            success: function( response ) {
                try {
                    var data;
                    if ( typeof response === 'object' ) {
                        data = response;
                    } else {
                        data = JSON.parse( response );
                    }

                    if ( data && data.success && data.data ) {
                        resolve( {
                            success: true,
                            content: data.data.content || '',
                            title: data.data.title || '',
                            excerpt: data.data.excerpt || '',
                            method: 'wordpress_ajax_universal',
                            debug: data.data
                        } );
                    } else {
                        if ( data && !data.success && data.data ) {
                            throw new Error( 'WordPress AJAX error: ' + data.data );
                        }
                        throw new Error( 'Invalid response format: ' + JSON.stringify( data ) );
                    }

                } catch ( e ) {
                    reject( new Error( 'WordPress AJAX response parsing failed: ' + e.message ) );
                }
            },
            error: function( xhr, status, error ) {
                var errorMsg = 'WordPress AJAX failed: ' + status;
                if ( xhr.status ) {
                    errorMsg += ' (HTTP ' + xhr.status + ')';
                }
                if ( xhr.responseText ) {
                    try {
                        var errorData = JSON.parse( xhr.responseText );
                        if ( errorData && errorData.data ) {
                            errorMsg += ' - ' + errorData.data;
                        }
                    } catch ( e ) {
                        errorMsg += ' - ' + xhr.responseText.substring( 0, 100 );
                    }
                }
                reject( new Error( errorMsg ) );
            }
        } );
    } );
};

/**
 * WORDPRESS-COMPLIANT: Enhanced partial analysis when content unavailable
 * This provides better scoring than the current 75% cap
 */

DPGSEOAnalyzer.prototype.createWordPressCompliantPartialAnalysis = function ( data, error ) {
    var checks = {};

    /* ---------------------------------------------------------------------
     * 1.  Content-dependent checks evaluated against an empty string
     *     – they give us:
     *       • “No content found”
     *       • “Keyphrase not found …”
     *       • “Keyphrase not in first paragraph”
     * ------------------------------------------------------------------- */
    var kp = data.focus_keyphrase.toLowerCase();
    checks.content_length            = this.checkContentLength( '',               data.sample_item );
    checks.keyphrase_frequency       = this.checkKeyphraseFrequency( '',  kp,     data.sample_item );
    checks.keyphrase_first_paragraph = this.checkKeyphraseFirstParagraph( '', kp, data.sample_item );

    /* ---------------------------------------------------------------------
     * 2.  Basic checks that don’t need the page HTML
     * ------------------------------------------------------------------- */
    checks.focus_keyphrase            = this.checkFocusKeyphrase( data.focus_keyphrase );
    checks.title_length               = this.checkTitleLength( data.title, data.sample_item );
    checks.title_keyphrase            = this.checkTitleKeyphrase( data.title, kp, data.sample_item );
    checks.meta_description_length    = this.checkDescriptionLength( data.description, data.sample_item );
    checks.meta_description_keyphrase = this.checkDescriptionKeyphrase( data.description, kp, data.sample_item );

    /* ---------------------------------------------------------------------
     * 3.  Explainer block – why the deep checks are missing
     * ------------------------------------------------------------------- */
    checks.content_guidance = {
        status   : 'info',
        score    : 0,
        critical : false,
        message  : __( 'Content analysis temporarily unavailable – basic SEO checks completed', 'dpg' ),
        how_to_fix : this.getWordPressCompliantGuidance( error, data.source_page_id )
    };

    /* ---------------------------------------------------------------------
     * 4.  Score: use the normal formula, then hard-cap to 15 %
     * ------------------------------------------------------------------- */
    var score = this.calculateRealisticScore(
        checks,
        Object.assign( {}, data, { content : '' } )   // pretend there is no HTML
    );
    if ( score > 15 ) score = 15;

    return {
        checks          : checks,
        score           : score,
        focus_keyphrase : data.focus_keyphrase,
        sample_item     : data.sample_item,
        template_source : data.template_source,
        warning         : 'wordpress_compliant_partial'
    };
};


/**
 * WORDPRESS-COMPLIANT: Better guidance for WordPress users
 */
DPGSEOAnalyzer.prototype.getWordPressCompliantGuidance = function( error, pageId ) {
    var guidance = [
        __( 'WordPress-compliant solutions to enable full SEO analysis:', 'dpg' )
    ];
    
    // WordPress-specific guidance based on error type
    if ( error && error.includes( 'AJAX' ) ) {
        guidance.push( __( '• Check WordPress AJAX is working correctly', 'dpg' ) );
        guidance.push( __( '• Disable conflicting caching plugins temporarily', 'dpg' ) );
        guidance.push( __( '• Verify WordPress admin-ajax.php is accessible', 'dpg' ) );
    }
    
    if ( error && error.includes( 'insufficient' ) ) {
        guidance.push( __( '• Ensure the selected page has substantial published content', 'dpg' ) );
        guidance.push( __( '• Check the page is not empty or in draft status', 'dpg' ) );
        guidance.push( __( '• Verify page content is properly saved in WordPress', 'dpg' ) );
    }
    
    if ( error && error.includes( 'parsing' ) ) {
        guidance.push( __( '• Check for WordPress theme conflicts', 'dpg' ) );
        guidance.push( __( '• Temporarily disable other SEO plugins', 'dpg' ) );
        guidance.push( __( '• Clear all WordPress caches and try again', 'dpg' ) );
    }
    
    // WordPress-standard solutions (always show these)
    guidance.push( __( '• Switch to "Custom HTML" for guaranteed full analysis', 'dpg' ) );
    guidance.push( __( '• Save WordPress permalinks: Settings → Permalinks → Save', 'dpg' ) );
    guidance.push( __( '• Test WordPress REST API: yoursite.com/wp-json/wp/v2/pages', 'dpg' ) );
    
    if ( pageId ) {
        guidance.push( __( '• Test page directly: Edit page → Preview to verify content loads', 'dpg' ) );
    }
    
    guidance.push( __( '• Contact support with WordPress debug info if issue persists', 'dpg' ) );
    
    return guidance.join( '\n' );
};

/**
 * WORDPRESS-COMPLIANT: Realistic scoring that rewards good basic SEO
 * This replaces the 75% cap with more generous scoring
 */
DPGSEOAnalyzer.prototype.calculateWordPressCompliantScore = function( checks, data ) {
    // WordPress-compliant: Focus on achievable checks with proper weighting
    var availableChecks = [
        'focus_keyphrase',
        'title_length', 
        'title_keyphrase',
        'meta_description_length',
        'meta_description_keyphrase'
    ];
    
    var totalScore = 0;
    var totalWeight = 0;
    var weights = {
        focus_keyphrase: 15,        // Having a focus keyphrase
        title_length: 25,           // Title optimization (most important for partial)
        title_keyphrase: 30,        // Keyphrase in title (most important)
        meta_description_length: 15, // Description length
        meta_description_keyphrase: 15  // Keyphrase in description
    };
    
    availableChecks.forEach( function( checkName ) {
        var check = checks[ checkName ];
        var weight = weights[ checkName ];
        if ( check && typeof check.score === 'number' && weight > 0 ) {
            totalScore += weight * ( check.score / 100 );
            totalWeight += weight;
        }
    });
    
    // Calculate base score from available checks
    var baseScore = totalWeight > 0 ? ( totalScore / totalWeight ) * 100 : 0;
    
    // WordPress-compliant bonuses for good basic SEO
    if ( data.focus_keyphrase && data.title ) {
        baseScore = Math.min( baseScore + 15, 100 ); // Bonus for having basics
    }
    
    if ( data.focus_keyphrase && data.title && data.description ) {
        baseScore = Math.min( baseScore + 10, 100 ); // Bonus for complete basic SEO
    }
    
    // WordPress-compliant: More generous scoring (85% max instead of 75%)
    // This reflects that partial analysis can still indicate good SEO
    return Math.min( Math.max( baseScore, 10 ), 85 ); // Max 85% when content analysis unavailable
};

/**
 * ENSURE your existing clientSideAnalysis method performs ALL these checks:
 * This is the method that should run for BOTH HTML and page templates when content is available
 */
DPGSEOAnalyzer.prototype.clientSideAnalysis = function( data ) {
    var checks = {};
    var kp = data.focus_keyphrase.toLowerCase();
    var templateType = data.template_type || 'service-area';

    // CORE CHECKS (same for all template types but with enhanced processing)
    checks.focus_keyphrase = this.checkFocusKeyphrase( data.focus_keyphrase );
    checks.content_length = this.checkContentLength( data.content, data.sample_item, templateType );
    checks.keyphrase_frequency = this.checkKeyphraseFrequency( data.content, kp, data.sample_item, templateType );
    checks.keyphrase_first_paragraph = this.checkKeyphraseFirstParagraph( data.content, kp, data.sample_item, templateType );
    checks.title_length = this.checkTitleLength( data.title, data.sample_item, templateType );
    checks.title_keyphrase = this.checkTitleKeyphrase( data.title, kp, data.sample_item, templateType );
    checks.meta_description_length = this.checkDescriptionLength( data.description, data.sample_item, templateType );
    checks.meta_description_keyphrase = this.checkDescriptionKeyphrase( data.description, kp, data.sample_item, templateType );

    // STRUCTURE CHECKS (only when we have content)
    if ( data.content && data.content.trim().length > 0 ) {
        checks.keyphrase_in_h1 = this.checkKeyphraseInHeading( data.content, kp, 'h1', true, data.sample_item, templateType );
        checks.keyphrase_in_h2 = this.checkKeyphraseInHeading( data.content, kp, 'h2', false, data.sample_item, templateType );
        checks.keyphrase_in_h3 = this.checkKeyphraseInHeading( data.content, kp, 'h3', false, data.sample_item, templateType );
        checks.keyphrase_in_alt = this.checkKeyphraseInAlt( data.content, kp, data.sample_item, templateType );
        checks.h1_seo_title_match = this.checkH1SeoTitleMatch( data.content, data.title, data.sample_item, templateType );
        checks.h2_structure = this.checkH2Structure( data.content, data.sample_item, templateType );
        checks.headings_structure = this.checkHeadingsStructure( data.content, data.sample_item, templateType );
        checks.internal_links = this.checkInternalLinks( data.content, data.sample_item, templateType );
    }

    var score = this.calculateRealisticScore( checks, data );

    return {
        checks: checks,
        score: score,
        focus_keyphrase: data.focus_keyphrase,
        sample_item: data.sample_item,
        template_source: data.template_source,
        template_type: templateType
    };
};




DPGSEOAnalyzer.prototype.updateAnalysisDisplay = function (results) {
    // Update score badge
    const scoreEl = document.getElementById('seo-score-number');
    if (scoreEl) {
        scoreEl.textContent = results.score + '%';
        scoreEl.className   = 'dpg-seo-score-number';

        if      (results.score >= 70) scoreEl.classList.add('dpg-seo-good');
        else if (results.score >= 40) scoreEl.classList.add('dpg-seo-ok');
        else                          scoreEl.classList.add('dpg-seo-poor');
    }

    // Update hidden score field
    const scoreField = document.querySelector(this.options.scoreField);
    if (scoreField) scoreField.value = results.score;

    // Categorize checks with better logic
    const problems     = [];
    const improvements = [];
    const goods        = [];

    for (const key in results.checks) {
        if (!results.checks.hasOwnProperty(key)) continue;
        const chk = results.checks[key];

        const item = {
            message   : chk.message,
            how_to_fix: chk.how_to_fix || ''
        };

        // Better categorization logic
        if (chk.status === 'error' || (chk.critical && chk.score === 0)) {
            problems.push(item);                                   // Problems
        } else if (chk.status === 'good' && chk.score >= 85) {    // 85%+ is good enough
            goods.push(item);                                      // Good
        } else if (chk.status === 'info' && chk.score === 0) {
            // Info items (like content guidance) don't go anywhere - they're just informational
            continue;
        } else {
            improvements.push(item);                               // Improvements
        }
    }

    // Update the three columns
    this.updateSEOSection('seo-problems',     'seo-problems-list',     problems);
    this.updateSEOSection('seo-improvements', 'seo-improvements-list', improvements);
    this.updateSEOSection('seo-good',         'seo-good-list',         goods);

    // Hide helper banner once populated
    const helpEl = document.getElementById('seo-help');
    if (helpEl && (problems.length || improvements.length || goods.length)) {
        helpEl.style.display = 'none';
    }
};

// Enhanced page dropdown change handler with better nonce handling
jQuery( document ).on( 'change', '#dpg_source_page_id', function () {
    const id = this.value;
    if ( ! id ) return;

    var nonce = '';
    var ajaxUrl = '/wp-admin/admin-ajax.php';
    
    // Use the analyzer's methods if available
    if ( window.dpgSEOAnalyzer ) {
        if ( window.dpgSEOAnalyzer.getNonce ) {
            nonce = window.dpgSEOAnalyzer.getNonce();
        }
        if ( window.dpgSEOAnalyzer.getAjaxUrl ) {
            ajaxUrl = window.dpgSEOAnalyzer.getAjaxUrl();
        }
    } else {
        // Fallback nonce retrieval
        if ( window.dpgCreateForm && window.dpgCreateForm.nonces ) {
            nonce = window.dpgCreateForm.nonces.page_data || window.dpgCreateForm.nonces.form || '';
        } else if ( window.dpgEditForm && window.dpgEditForm.nonces ) {
            nonce = window.dpgEditForm.nonces.page_data || window.dpgEditForm.nonces.form || '';
        } else if ( window.dpgForm && window.dpgForm.nonces ) {
            nonce = window.dpgForm.nonces.page_data || window.dpgForm.nonces.form || '';
        }
        
        // Fallback AJAX URL
        if ( window.dpgCreateForm && window.dpgCreateForm.ajaxUrl ) {
            ajaxUrl = window.dpgCreateForm.ajaxUrl;
        } else if ( window.dpgEditForm && window.dpgEditForm.ajaxUrl ) {
            ajaxUrl = window.dpgEditForm.ajaxUrl;
        } else if ( window.dpgForm && window.dpgForm.ajaxUrl ) {
            ajaxUrl = window.dpgForm.ajaxUrl;
        }
    }

    jQuery.get( 
        ajaxUrl, 
        { 
            action: 'dpg_get_page_data', 
            id: id,
            _wpnonce: nonce,
            nonce: nonce,
            security: nonce
        }
    )
    .done( function ( response ) {
        if ( window.dpgSEOAnalyzer ) {
            window.dpgSEOAnalyzer.pageContentCache.delete( 'wp_page_' + id );
            window.dpgSEOAnalyzer.runAnalysis();
        }
    } )
    .fail( function ( xhr, status, error ) {
        if ( window.dpgSEOAnalyzer ) {
            window.dpgSEOAnalyzer.runAnalysis();
        }
    } );
} );


/**
 * Show enhanced guidance instead of just warnings
 */
DPGSEOAnalyzer.prototype.showEnhancedPageContentGuidance = function() {
    var helpEl = document.getElementById( 'seo-help' );
    if ( helpEl ) {
        helpEl.style.display = 'block';
        helpEl.innerHTML = 
            '<div class="dpg-seo-guidance">' +
            '<h4>💡 ' + __( 'SEO Analysis Status', 'dpg' ) + '</h4>' +
            '<p>' + __( 'Basic SEO checks are working! For complete content analysis:', 'dpg' ) + '</p>' +
            '<div class="dpg-seo-solutions">' +
            '<h5>' + __( 'Quick Solutions:', 'dpg' ) + '</h5>' +
            '<ul>' +
            '<li><strong>' + __( 'Switch to Custom HTML:', 'dpg' ) + '</strong> ' + __( 'Full analysis available instantly', 'dpg' ) + '</li>' +
            '<li><strong>' + __( 'Enable REST API:', 'dpg' ) + '</strong> ' + __( 'Contact your hosting provider if blocked', 'dpg' ) + '</li>' +
            '<li><strong>' + __( 'Check Page Status:', 'dpg' ) + '</strong> ' + __( 'Ensure the page is published and accessible', 'dpg' ) + '</li>' +
            '</ul>' +
            '</div>' +
            '<div class="dpg-seo-current">' +
            '<h5>' + __( 'Currently Analyzing:', 'dpg' ) + '</h5>' +
            '<ul>' +
            '<li>✅ ' + __( 'Focus keyphrase usage', 'dpg' ) + '</li>' +
            '<li>✅ ' + __( 'Title optimization', 'dpg' ) + '</li>' +
            '<li>✅ ' + __( 'Meta description quality', 'dpg' ) + '</li>' +
            '<li>⏸️ ' + __( 'Content analysis (temporarily unavailable)', 'dpg' ) + '</li>' +
            '</ul>' +
            '</div>' +
            '</div>';
    }
};





	/**
	 * Collect analysis data from form fields
	 * 
	 * @return {Object} Analysis data
	 */
DPGSEOAnalyzer.prototype.collectAnalysisData = function () {
    // Load the example item first
    this.loadSampleItem();

    // Properly detect template SOURCE (not type)
    var templateSourceVal = $(this.options.templateSourceField).length
        ? $(this.options.templateSourceField).val()
        : '';

    var sourcePageId = $(this.options.sourcePageField).val();
    var templateSource = templateSourceVal || (sourcePageId ? 'page' : 'custom');

    // Get template TYPE separately - this is service-area vs keyword
    var templateType = $('#dpg_type').length ? $('#dpg_type').val() : 'service-area';

    // Content comes from either the textarea or will be fetched later
    var htmlContent = (templateSource === 'custom')
        ? $(this.options.contentField).val().trim()
        : '';

    // Bundle & return with proper template type tracking
    var data = {
        focus_keyphrase : $(this.options.focusKeyphraseField).val().trim(),
        title           : $(this.options.titleField).val().trim(),
        description     : $(this.options.descriptionField).val().trim(),
        content         : htmlContent,
        items           : $(this.options.itemsField).val().trim(),
        sample_item     : this.sampleItem,
        template_source : templateSource,
        template_type   : templateType,
        source_page_id  : sourcePageId,
        template_data   : this.getTemplateData()
    };

    return data;
};

	/**
	 * Get template data object
	 * 
	 * @return {Object} Template data
	 */
	DPGSEOAnalyzer.prototype.getTemplateData = function() {
		return {
			hard_meta: {
				title: $( this.options.titleField ).val().trim(),
				description: $( this.options.descriptionField ).val().trim(),
				keywords: $( '#dpg_meta_keywords' ).val() || '',
				og_title: $( '#dpg_og_title' ).val() || '',
				og_description: $( '#dpg_og_description' ).val() || '',
				og_image: $( '#dpg_og_image' ).val() || ''
			},
			template_html: $( this.options.contentField ).val() || '',
			template_source: $( this.options.templateSourceField ).val() || 'custom',
			source_page_id: $( this.options.sourcePageField ).val() || '',
			items: this.parseItems( $( this.options.itemsField ).val() || '' )
		};
	};

	/**
	 * Parse items list
	 * 
	 * @param {string} text Items text
	 * @return {Array} Parsed items
	 */
	DPGSEOAnalyzer.prototype.parseItems = function( text ) {
		return ( text || '' )
			.split( '\n' )
			.map( function( l ) {
				return l.trim();
			} )
			.filter( Boolean )
			.map( function( line ) {
				if ( line.includes( '|' ) ) {
					var parts = line.split( '|', 2 ).map( function( s ) {
						return s.trim();
					} );
					return { 
						name: parts[ 0 ], 
						score: parseFloat( parts[ 1 ] ) || null 
					};
				}
				return line;
			} );
	};



	/**
	 * Check focus keyphrase
	 * 
	 * @param {string} keyphrase Focus keyphrase
	 * @return {Object} Check result
	 */
	DPGSEOAnalyzer.prototype.checkFocusKeyphrase = function( keyphrase ) {
		if ( ! keyphrase ) {
			return {
				status: 'warning',
				score: 0,
				critical: false,
				message: __( 'No focus keyphrase set.', 'dpg' ),
				how_to_fix: __( 'Set a focus keyphrase to optimize your content.', 'dpg' )
			};
		}

		return {
			status: 'good',
			score: 100,
			critical: false,
			message: sprintf( __( 'Focus keyphrase: "%s"', 'dpg' ), keyphrase ),
			how_to_fix: ''
		};
	};

	/**
	 * Check content length
	 * 
	 * @param {string} content Content
	 * @param {string} sample Sample item
	 * @return {Object} Check result
	 */
DPGSEOAnalyzer.prototype.checkContentLength = function( content, sample, templateType ) {
    var text = this.stripHtml( this.processContent( content, sample, templateType ) );
    var words = this.countWords( text );

    if ( words === 0 ) {
        return {
            status: 'warning',
            score: 0,
            critical: false,
            message: __( 'No content found.', 'dpg' ),
            how_to_fix: __( 'Add meaningful content. Aim for 300+ words.', 'dpg' )
        };
    }

    if ( words < 50 ) {
        return {
            status: 'warning',
            score: 25,
            critical: false,
            message: sprintf( __( 'Content is too short (%d words).', 'dpg' ), words ),
            how_to_fix: __( 'Add more content. Aim for 300+ words.', 'dpg' )
        };
    }

    if ( words < 150 ) {
        return {
            status: 'warning',
            score: 50,
            critical: false,
            message: sprintf( __( 'Content could be longer (%d words).', 'dpg' ), words ),
            how_to_fix: __( 'Consider adding more content for better SEO.', 'dpg' )
        };
    }

    if ( words < 300 ) {
        return {
            status: 'warning',
            score: 75,
            critical: false,
            message: sprintf( __( 'Good content length (%d words).', 'dpg' ), words ),
            how_to_fix: __( 'Content length is good, but 300+ words is ideal.', 'dpg' )
        };
    }

    return {
        status: 'good',
        score: 100,
        critical: false,
        message: sprintf( __( 'Excellent content length (%d words).', 'dpg' ), words ),
        how_to_fix: ''
    };
};

	/**
	 * Check keyphrase frequency
	 * 
	 * @param {string} content Content
	 * @param {string} keyphrase Keyphrase
	 * @param {string} sample Sample item
	 * @return {Object} Check result
	 */
DPGSEOAnalyzer.prototype.checkKeyphraseFrequency = function( content, keyphrase, sample, templateType ) {
    if ( ! keyphrase ) {
        return this.checkFocusKeyphrase( '' );
    }

    var text = this.stripHtml( this.processContent( content, sample, templateType ) ).toLowerCase();
    var processedKeyphrase = this.processKeyphrase( keyphrase, sample, templateType ).toLowerCase();
    var count = this.countOccurrences( text, processedKeyphrase );
    var totalWords = this.countWords( text );

    if ( count === 0 ) {
        return {
            status: 'warning',
            score: 0,
            critical: false,
            message: sprintf( __( 'Keyphrase "%s" not found in content.', 'dpg' ), processedKeyphrase ),
            how_to_fix: sprintf( __( 'Include "%s" in your content 2-4 times.', 'dpg' ), processedKeyphrase )
        };
    }

    if ( count === 1 ) {
        return {
            status: 'warning',
            score: 60,
            critical: false,
            message: sprintf( __( 'Keyphrase appears %d time.', 'dpg' ), count ),
            how_to_fix: sprintf( __( 'Use "%s" 2-4 times for better optimization.', 'dpg' ), processedKeyphrase )
        };
    }

    if ( count <= 4 ) {
        return {
            status: 'good',
            score: 100,
            critical: false,
            message: sprintf( __( 'Good keyphrase frequency (%d times).', 'dpg' ), count ),
            how_to_fix: ''
        };
    }

    var density = totalWords > 0 ? Math.round( ( count / totalWords ) * 100 ) : 0;
    if ( density > 3 ) {
        return {
            status: 'warning',
            score: 70,
            critical: false,
            message: sprintf( __( 'Keyphrase may be overused (%d times, %d%% density).', 'dpg' ), count, density ),
            how_to_fix: __( 'Reduce keyphrase density to below 3%.', 'dpg' )
        };
    }

    return {
        status: 'good',
        score: 95,
        critical: false,
        message: sprintf( __( 'Good keyphrase frequency (%d times).', 'dpg' ), count ),
        how_to_fix: ''
    };
};

	/**
	 * Check keyphrase in first paragraph
	 * 
	 * @param {string} content Content
	 * @param {string} keyphrase Keyphrase
	 * @param {string} sample Sample item
	 * @return {Object} Check result
	 */
DPGSEOAnalyzer.prototype.checkKeyphraseFirstParagraph = function( content, keyphrase, sample, templateType ) {
    // No keyphrase → bail out
    if ( ! keyphrase ) {
        return this.checkFocusKeyphrase( '' );
    }

    // If your keyphrase contains "{item}", check for the literal placeholder
    var lowerKeyphrase = keyphrase.toLowerCase();
    if ( lowerKeyphrase.indexOf( '{item}' ) !== -1 ) {
        var rawHtml = ( content || '' ).toLowerCase();
        if ( rawHtml.indexOf( '{item}' ) !== -1 ) {
            return {
                status: 'good',
                score: 100,
                critical: false,
                message: __( 'Keyphrase placeholder found early in content.', 'dpg' ),
                how_to_fix: ''
            };
        }
    }

    // Normal flow: replace {item} with sample, strip tags, then check first 200 chars
    var text = this.stripHtml( this.processContent( content, sample, templateType ) );
    var firstParagraph = text.substring( 0, 200 ).toLowerCase();
    var processedKeyphrase = this.processKeyphrase( keyphrase, sample, templateType ).toLowerCase();

    if ( firstParagraph.includes( processedKeyphrase ) ) {
        return {
            status: 'good',
            score: 100,
            critical: false,
            message: __( 'Keyphrase appears early in content.', 'dpg' ),
            how_to_fix: ''
        };
    }

    return {
        status: 'warning',
        score: 0,
        critical: false,
        message: __( 'Keyphrase not found in first paragraph.', 'dpg' ),
        how_to_fix: __( 'Include your keyphrase early in the content.', 'dpg' )
    };
};

	/**
	 * Check title length
	 * 
	 * @param {string} title Title
	 * @param {string} sample Sample item
	 * @return {Object} Check result
	 */
DPGSEOAnalyzer.prototype.checkTitleLength = function( title, sample, templateType ) {
    var processedTitle = this.processContent( title, sample, templateType );
    var length = processedTitle.length;

    if ( length === 0 ) {
        return {
            status: 'warning',
            score: 0,
            critical: false,
            message: __( 'No SEO title set.', 'dpg' ),
            how_to_fix: __( 'Add an SEO title (30-60 characters recommended).', 'dpg' )
        };
    }

    if ( length < 30 ) {
        return {
            status: 'warning',
            score: 60,
            critical: false,
            message: sprintf( __( 'Title too short (%d characters).', 'dpg' ), length ),
            how_to_fix: __( 'Make it 30-60 characters.', 'dpg' )
        };
    }

    if ( length > 60 ) {
        return {
            status: 'warning',
            score: 80,
            critical: false,
            message: sprintf( __( 'Title too long (%d characters).', 'dpg' ), length ),
            how_to_fix: __( 'Keep it under 60 characters.', 'dpg' )
        };
    }

    return {
        status: 'good',
        score: 100,
        critical: false,
        message: sprintf( __( 'Title length perfect (%d characters).', 'dpg' ), length ),
        how_to_fix: ''
    };
};


	/**
	 * Check title keyphrase
	 * 
	 * @param {string} title Title
	 * @param {string} keyphrase Keyphrase
	 * @param {string} sample Sample item
	 * @return {Object} Check result
	 */
DPGSEOAnalyzer.prototype.checkTitleKeyphrase = function( title, keyphrase, sample, templateType ) {
    // No keyphrase → bail
    if ( ! keyphrase ) {
        return this.checkFocusKeyphrase( '' );
    }

    // If your keyphrase literally contains "{item}", look for that placeholder in the raw title
    var lowerKp = keyphrase.toLowerCase();
    if ( lowerKp.indexOf( '{item}' ) !== -1 ) {
        var rawTitle = ( title || '' ).toLowerCase();
        if ( rawTitle.indexOf( '{item}' ) !== -1 ) {
            return {
                status: 'good',
                score: 100,
                critical: false,
                message: __( 'Keyphrase placeholder found in title.', 'dpg' ),
                how_to_fix: ''
            };
        }
    }

    // Process with templateType support
    var processedTitle = this.processContent( title, sample, templateType ).toLowerCase();
    var processedKeyphrase = this.processKeyphrase( keyphrase, sample, templateType ).toLowerCase();

    if ( processedTitle.includes( processedKeyphrase ) ) {
        return {
            status: 'good',
            score: 100,
            critical: false,
            message: __( 'Keyphrase found in title.', 'dpg' ),
            how_to_fix: ''
        };
    }

    return {
        status: 'warning',
        score: 0,
        critical: false,
        message: __( 'Keyphrase not in title.', 'dpg' ),
        how_to_fix: sprintf( __( 'Include "%s" in your title.', 'dpg' ), processedKeyphrase )
    };
};

	/**
	 * Check description length
	 * 
	 * @param {string} description Description
	 * @param {string} sample Sample item
	 * @return {Object} Check result
	 */
DPGSEOAnalyzer.prototype.checkDescriptionLength = function( description, sample, templateType ) {
    var processedDesc = this.processContent( description, sample, templateType );
    var length = processedDesc.length;

    if ( length === 0 ) {
        return {
            status: 'warning',
            score: 0,
            critical: false,
            message: __( 'No meta description set.', 'dpg' ),
            how_to_fix: __( 'Add a meta description (120-160 characters).', 'dpg' )
        };
    }

    if ( length < this.settings.descriptionLength.min ) {
        return {
            status: 'warning',
            score: 70,
            critical: false,
            message: sprintf( __( 'Description too short (%d characters).', 'dpg' ), length ),
            how_to_fix: sprintf( __( 'Make it %d-%d characters.', 'dpg' ), this.settings.descriptionLength.min, this.settings.descriptionLength.max )
        };
    }

    if ( length > this.settings.descriptionLength.max ) {
        return {
            status: 'warning',
            score: 85,
            critical: false,
            message: sprintf( __( 'Description too long (%d characters).', 'dpg' ), length ),
            how_to_fix: sprintf( __( 'Keep it under %d characters.', 'dpg' ), this.settings.descriptionLength.max )
        };
    }

    return {
        status: 'good',
        score: 100,
        critical: false,
        message: sprintf( __( 'Description length perfect (%d characters).', 'dpg' ), length ),
        how_to_fix: ''
    };
};

	/**
	 * Check description keyphrase
	 * 
	 * @param {string} description Description
	 * @param {string} keyphrase Keyphrase
	 * @param {string} sample Sample item
	 * @return {Object} Check result
	 */
DPGSEOAnalyzer.prototype.checkDescriptionKeyphrase = function( description, keyphrase, sample, templateType ) {
    if ( ! keyphrase ) {
        return this.checkFocusKeyphrase( '' );
    }

    var processedDesc = this.processContent( description, sample, templateType ).toLowerCase();
    var processedKeyphrase = this.processKeyphrase( keyphrase, sample, templateType ).toLowerCase();

    if ( processedDesc.includes( processedKeyphrase ) ) {
        return {
            status: 'good',
            score: 100,
            critical: false,
            message: __( 'Keyphrase found in description.', 'dpg' ),
            how_to_fix: ''
        };
    }

    return {
        status: 'warning',
        score: 0,
        critical: false,
        message: __( 'Keyphrase not in description.', 'dpg' ),
        how_to_fix: sprintf( __( 'Include "%s" in your meta description.', 'dpg' ), processedKeyphrase )
    };
};

	/**
	 * Check keyphrase in heading
	 * 
	 * @param {string} content Content
	 * @param {string} keyphrase Keyphrase
	 * @param {string} tag Heading tag
	 * @param {boolean} exact Exact match required
	 * @return {Object} Check result
	 */
DPGSEOAnalyzer.prototype.checkKeyphraseInHeading = function( content, keyphrase, tag, exact, sample, templateType ) {
    if ( ! keyphrase ) {
        return this.checkFocusKeyphrase( '' );
    }
    
    var lowerKeyphrase = keyphrase.toLowerCase();
    var found = false;
    
    // First check for literal {item} placeholder in both keyphrase and content
    if ( lowerKeyphrase.indexOf( '{item}' ) !== -1 ) {
        var rawContent = ( content || '' ).toLowerCase();
        var headingRegex = new RegExp( '<' + tag + '[^>]*>(.*?)</' + tag + '>', 'gi' );
        var match;
        
        while ( ( match = headingRegex.exec( rawContent ) ) ) {
            var headingText = match[ 1 ].trim();
            
            // Check if heading contains {item} placeholder AND the rest of the keyphrase
            if ( headingText.includes( '{item}' ) ) {
                // Extract the non-{item} parts of the keyphrase for comparison
                var keyphraseWithoutItem = lowerKeyphrase.replace( '{item}', '' ).trim();
                var headingWithoutItem = headingText.replace( '{item}', '' ).trim();
                
                // If keyphrase is just "{item}", any heading with {item} is a match
                if ( keyphraseWithoutItem === '' ) {
                    found = true;
                    break;
                }
                
                // Check if the non-{item} parts match
                if ( exact ) {
                    // For exact match, compare the structure
                    var normalizedKeyphrase = lowerKeyphrase.replace( /\s+/g, ' ' ).trim();
                    var normalizedHeading = headingText.replace( /\s+/g, ' ' ).trim();
                    if ( normalizedHeading === normalizedKeyphrase ) {
                        found = true;
                        break;
                    }
                } else {
                    // For partial match, check if heading contains the keyphrase pattern
                    if ( headingText.includes( keyphraseWithoutItem ) || 
                         headingText.replace( '{item}', sample ).toLowerCase().includes( lowerKeyphrase.replace( '{item}', sample ) ) ) {
                        found = true;
                        break;
                    }
                }
            }
        }
        
        if ( found ) {
            return {
                status: 'good',
                score: 100,
                critical: false,
                message: sprintf( __( 'Keyphrase found in %s.', 'dpg' ), tag.toUpperCase() ),
                how_to_fix: ''
            };
        }
    }
    
    // Fallback to processed content comparison
    var html = this.processContent( content, sample, templateType ).toLowerCase();
    var target = this.processKeyphrase( keyphrase, sample, templateType ).toLowerCase();
    var regex = new RegExp( '<' + tag + '[^>]*>(.*?)</' + tag + '>', 'gi' );
    var match;
    
    while ( ( match = regex.exec( html ) ) ) {
        var txt = this.stripHtml( match[ 1 ] ).trim().toLowerCase();
        
        if ( ( exact && txt === target ) || ( ! exact && txt.includes( target ) ) ) { 
            found = true; 
            break; 
        }
    }
    
    if ( tag === 'h3' ) {
        return found ? {
            status: 'good',
            score: 100,
            critical: false,
            message: __( 'Keyphrase found in H3 heading.', 'dpg' ),
            how_to_fix: ''
        } : {
            status: 'warning',
            score: 0,
            critical: false,
            message: __( 'Keyphrase not found in any H3 heading.', 'dpg' ),
            how_to_fix: __( 'Include the keyphrase in at least one H3 heading.', 'dpg' )
        };
    }
    
    return found ? {
        status: 'good',
        score: 100,
        critical: false,
        message: sprintf( __( 'Keyphrase found in %s.', 'dpg' ), tag.toUpperCase() ),
        how_to_fix: ''
    } : {
        status: 'warning',
        score: 0,
        critical: false,
        message: sprintf( __( 'Keyphrase not found in any %s.', 'dpg' ), tag.toUpperCase() ),
        how_to_fix: sprintf( __( 'Add the keyphrase to a %s heading.', 'dpg' ), tag.toUpperCase() )
    };
};





	/**
	 * Check keyphrase in alt attributes
	 * 
	 * @param {string} content Content
	 * @param {string} keyphrase Keyphrase
	 * @return {Object} Check result
	 */
DPGSEOAnalyzer.prototype.checkKeyphraseInAlt = function( content, keyphrase, sample, templateType ) {
    if ( ! keyphrase ) {
        return this.checkFocusKeyphrase( '' );
    }
    
    var html = this.processContent( content, sample, templateType ).toLowerCase();
    var kp = this.processKeyphrase( keyphrase, sample, templateType ).toLowerCase();
    var imgs = html.match( /<img[^>]*alt=["']([^"']+)["'][^>]*>/gi ) || [];
    var ok = imgs.some( function( tag ) {
        return tag.includes( 'alt="' ) && tag.toLowerCase().includes( kp );
    } );
    
    return ok ? {
        status: 'good',
        score: 100,
        critical: false,
        message: __( 'Keyphrase appears in an image alt attribute.', 'dpg' ),
        how_to_fix: ''
    } : {
        status: 'warning',
        score: 0,
        critical: false,
        message: __( 'Keyphrase missing from image alt attributes.', 'dpg' ),
        how_to_fix: __( 'Add the keyphrase to at least one relevant image alt tag.', 'dpg' )
    };
};


	/**
	 * Check H1 SEO title match
	 * 
	 * @param {string} content Content
	 * @param {string} title Title
	 * @param {string} sample Sample item
	 * @return {Object} Check result
	 */
DPGSEOAnalyzer.prototype.checkH1SeoTitleMatch = function( content, title, sample, templateType ) {
    var html = this.processContent( content, sample, templateType );
    var h1Match = /<h1[^>]*>(.*?)<\/h1>/i.exec( html );

    if ( ! h1Match ) {
        return {
            status: 'warning',
            score: 0,
            critical: false,
            message: __( 'No H1 heading found.', 'dpg' ),
            how_to_fix: __( 'Add an H1 heading to your content.', 'dpg' )
        };
    }

    var h1Text = this.stripHtml( h1Match[ 1 ] ).trim().toLowerCase();
    var titleText = this.processContent( title, sample, templateType ).trim().toLowerCase();

    if ( ! titleText ) {
        return {
            status: 'warning',
            score: 50,
            critical: false,
            message: __( 'H1 found but no SEO title set.', 'dpg' ),
            how_to_fix: __( 'Set an SEO title that matches your H1.', 'dpg' )
        };
    }

    // Exact match gets full score
    if ( h1Text === titleText ) {
        return {
            status: 'good',
            score: 100,
            critical: false,
            message: __( 'H1 matches SEO title perfectly.', 'dpg' ),
            how_to_fix: ''
        };
    }

    // Check if both H1 and title contain the focus keyphrase
    var focusKeyphrase = $( this.options.focusKeyphraseField ).val().trim().toLowerCase();
    if ( focusKeyphrase ) {
        var processedKeyphrase = this.processKeyphrase( focusKeyphrase, sample, templateType ).toLowerCase();
        
        var h1HasKeyphrase = h1Text.includes( processedKeyphrase );
        var titleHasKeyphrase = titleText.includes( processedKeyphrase );
        
        // If both contain the keyphrase, that's good enough
        if ( h1HasKeyphrase && titleHasKeyphrase ) {
            return {
                status: 'good',
                score: 90,
                critical: false,
                message: __( 'H1 and SEO title both contain the focus keyphrase.', 'dpg' ),
                how_to_fix: ''
            };
        }
        
        // If only H1 has keyphrase, that's still decent
        if ( h1HasKeyphrase && ! titleHasKeyphrase ) {
            return {
                status: 'warning',
                score: 70,
                critical: false,
                message: __( 'H1 contains focus keyphrase but SEO title does not.', 'dpg' ),
                how_to_fix: __( 'Include the focus keyphrase in your SEO title.', 'dpg' )
            };
        }
        
        // If only title has keyphrase
        if ( ! h1HasKeyphrase && titleHasKeyphrase ) {
            return {
                status: 'warning',
                score: 60,
                critical: false,
                message: __( 'SEO title contains focus keyphrase but H1 does not.', 'dpg' ),
                how_to_fix: __( 'Include the focus keyphrase in your H1 heading.', 'dpg' )
            };
        }
    }

    // Default case - they differ but no keyphrase context
    return {
        status: 'warning',
        score: 50,
        critical: false,
        message: __( 'H1 differs from SEO title.', 'dpg' ),
        how_to_fix: __( 'Consider aligning your H1 with your SEO title, or ensure both contain your focus keyphrase.', 'dpg' )
    };
};

	/**
	 * Check H2 structure
	 * 
	 * @param {string} content Content
	 * @param {string} sample Sample item
	 * @return {Object} Check result
	 */
DPGSEOAnalyzer.prototype.checkH2Structure = function( content, sample, templateType ) {
    var html = this.processContent( content, sample, templateType );
    var h2s = html.match( /<h2[^>]*>.*?<\/h2>/gi ) || [];
    var focusKeyphrase = $( this.options.focusKeyphraseField ).val().trim().toLowerCase();

    if ( h2s.length === 0 ) {
        return {
            status: 'warning',
            score: 0,
            critical: false,
            message: __( 'No H2 subheadings found.', 'dpg' ),
            how_to_fix: __( 'Add H2 subheadings to structure your content.', 'dpg' )
        };
    }

    // Check for focus keyphrase in H2s instead of exact title match
    if ( focusKeyphrase ) {
        var processedKeyphrase = this.processKeyphrase( focusKeyphrase, sample, templateType ).toLowerCase();
        var foundKeyphraseInH2 = false;
        
        // First check for literal {item} placeholder if keyphrase contains it
        if ( focusKeyphrase.indexOf( '{item}' ) !== -1 ) {
            var rawHtml = ( content || '' ).toLowerCase();
            var h2Regex = /<h2[^>]*>(.*?)<\/h2>/gi;
            var match;
            
            while ( ( match = h2Regex.exec( rawHtml ) ) ) {
                var h2Text = match[ 1 ].trim();
                if ( h2Text.includes( '{item}' ) ) {
                    foundKeyphraseInH2 = true;
                    break;
                }
            }
        }
        
        // If no placeholder found, check processed content
        if ( ! foundKeyphraseInH2 ) {
            var self = this;
            for ( var i = 0; i < h2s.length; i++ ) {
                var h2Text = self.stripHtml( h2s[ i ] ).trim().toLowerCase();
                if ( h2Text.includes( processedKeyphrase ) ) {
                    foundKeyphraseInH2 = true;
                    break;
                }
            }
        }
        
        if ( foundKeyphraseInH2 ) {
            return {
                status: 'good',
                score: 100,
                critical: false,
                message: sprintf( __( 'Focus keyphrase found in H2 subheading (%d H2s total).', 'dpg' ), h2s.length ),
                how_to_fix: ''
            };
        } else {
            return {
                status: 'warning',
                score: 60,
                critical: false,
                message: sprintf( __( 'Found %d H2 subheading(s) but none contain the focus keyphrase.', 'dpg' ), h2s.length ),
                how_to_fix: sprintf( __( 'Include "%s" in at least one H2 subheading.', 'dpg' ), processedKeyphrase )
            };
        }
    }

    // If no focus keyphrase is set, having H2s is still good for structure
    return {
        status: 'good',
        score: 85,
        critical: false,
        message: sprintf( __( 'Found %d H2 subheading(s) - good content structure.', 'dpg' ), h2s.length ),
        how_to_fix: __( 'Consider including your focus keyphrase in one of the H2 headings.', 'dpg' )
    };
};

	/**
	 * Check headings structure
	 * 
	 * @param {string} content Content
	 * @param {string} sample Sample item
	 * @return {Object} Check result
	 */
DPGSEOAnalyzer.prototype.checkHeadingsStructure = function( content, sample, templateType ) {
    var html = this.processContent( content, sample, templateType );
    var headings = html.match( /<h[1-6][^>]*>.*?<\/h[1-6]>/gi ) || [];

    if ( headings.length === 0 ) {
        return {
            status: 'warning',
            score: 0,
            critical: false,
            message: __( 'No headings found.', 'dpg' ),
            how_to_fix: __( 'Add H1, H2, and H3 headings to structure your content.', 'dpg' )
        };
    }

    var hasH1 = /<h1[^>]*>/i.test( html );
    if ( ! hasH1 ) {
        return {
            status: 'warning',
            score: 50,
            critical: false,
            message: sprintf( __( 'Found %d headings but no H1.', 'dpg' ), headings.length ),
            how_to_fix: __( 'Add an H1 heading as your main title.', 'dpg' )
        };
    }

    return {
        status: 'good',
        score: 100,
        critical: false,
        message: sprintf( __( 'Good heading structure (%d headings).', 'dpg' ), headings.length ),
        how_to_fix: ''
    };
};

	/**
	 * Check internal links
	 * 
	 * @param {string} content Content
	 * @param {string} sample Sample item
	 * @return {Object} Check result
	 */
DPGSEOAnalyzer.prototype.checkInternalLinks = function( content, sample, templateType ) {
    var html = this.processContent( content, sample, templateType );
    var domain = window.location.hostname;
    
    var allLinks = html.match( /<a[^>]+href=["']([^"']+)["'][^>]*>/gi ) || [];
    
    if ( allLinks.length === 0 ) {
        return {
            status: 'warning',
            score: 0,
            critical: false,
            message: __( 'No links found.', 'dpg' ),
            how_to_fix: __( 'Add at least 1 internal link and 1 external link.', 'dpg' )
        };
    }

    var internalCount = 0;
    var externalCount = 0;
    
    allLinks.forEach( function( linkTag ) {
        var hrefMatch = linkTag.match( /href=["']([^"']+)["']/i );
        if ( hrefMatch ) {
            var href = hrefMatch[ 1 ];
            if ( href.startsWith( '/' ) || href.includes( domain ) ) {
                internalCount++;
            } else if ( href.startsWith( 'http' ) ) {
                externalCount++;
            }
        }
    } );

    var hasInternal = internalCount > 0;
    var hasExternal = externalCount > 0;

    if ( hasInternal && hasExternal ) {
        return {
            status: 'good',
            score: 100,
            critical: false,
            message: sprintf( __( 'Found %d internal and %d external links.', 'dpg' ), internalCount, externalCount ),
            how_to_fix: ''
        };
    } else if ( hasInternal && ! hasExternal ) {
        return {
            status: 'warning',
            score: 60,
            critical: false,
            message: sprintf( __( 'Found %d internal links but no external links.', 'dpg' ), internalCount ),
            how_to_fix: __( 'Add at least 1 external link to relevant sources.', 'dpg' )
        };
    } else if ( ! hasInternal && hasExternal ) {
        return {
            status: 'warning',
            score: 60,
            critical: false,
            message: sprintf( __( 'Found %d external links but no internal links.', 'dpg' ), externalCount ),
            how_to_fix: __( 'Add at least 1 internal link to other pages.', 'dpg' )
        };
    } else {
        return {
            status: 'warning',
            score: 30,
            critical: false,
            message: sprintf( __( 'Found %d links but no clear internal or external links.', 'dpg' ), allLinks.length ),
            how_to_fix: __( 'Add at least 1 internal link and 1 external link.', 'dpg' )
        };
    }
};

	/**
	 * Process content with sample item replacement
	 * 
	 * @param {string} html HTML content
	 * @param {string} sample Sample item
	 * @return {string} Processed content
	 */
DPGSEOAnalyzer.prototype.processContent = function( html, sample, templateType ) {
    if (!html) return '';
    
    var processed = html.replace( /{item}/gi, sample );
    
    return processed;
};

	/**
	 * Process keyphrase with sample item replacement
	 * 
	 * @param {string} kp Keyphrase
	 * @param {string} sample Sample item
	 * @return {string} Processed keyphrase
	 */
DPGSEOAnalyzer.prototype.processKeyphrase = function( kp, sample, templateType ) {
    if (!kp) return '';
    
    var processed = kp;
    
    // For keyword templates, handle {item} replacement differently
    if (templateType === 'keyword') {
        // For keyword templates, {item} might represent the focus term itself
        processed = processed.replace( /{item}/gi, sample );
    } else {
        // For service-area templates, normal replacement
        processed = processed.replace( /{item}/gi, sample );
    }
    
    return processed;
};

	/**
	 * Get actual word count
	 * 
	 * @param {string} content Content
	 * @param {string} sample Sample item
	 * @return {number} Word count
	 */
	DPGSEOAnalyzer.prototype.getActualWordCount = function( content, sample ) {
		var processedContent = this.processContent( content || '', sample );
		return this.countWords( this.stripHtml( processedContent ) );
	};

	/**
	 * Calculate realistic score
	 * 
	 * @param {Object} checks Check results
	 * @param {Object} data Analysis data
	 * @return {number} Score
	 */
DPGSEOAnalyzer.prototype.calculateRealisticScore = function( checks, data ) {
    var wordCount = this.getActualWordCount( data.content, data.sample_item );
    var baseScore = this.calculateBasicScore( checks );

    // Less aggressive word count penalties
    var scoreCap = 100;
    if ( wordCount < 10 ) {
        scoreCap = 20;        // Increased from 15
    } else if ( wordCount < 50 ) {
        scoreCap = 45;        // Increased from 35
    } else if ( wordCount < 100 ) {
        scoreCap = 65;        // Increased from 55
    } else if ( wordCount < 200 ) {
        scoreCap = 85;        // Increased from 75
    } else if ( wordCount < 300 ) {
        scoreCap = 95;        // Increased from 90
    }
    
    // Apply the cap
    baseScore = Math.min(baseScore, scoreCap);

    var minScore = data.focus_keyphrase ? 10 : 5;
    var finalScore = Math.max( minScore, Math.min( 100, Math.round( baseScore ) ) );
    
    return finalScore;
};

	/**
	 * Calculate basic score
	 * 
	 * @param {Object} checks Check results
	 * @return {number} Score
	 */
DPGSEOAnalyzer.prototype.calculateBasicScore = function( checks ) {
    var weights = {
        focus_keyphrase: 12,               // Reduced slightly
        content_length: 5,
        keyphrase_frequency: 10,
        title_keyphrase: 15,
        keyphrase_first_paragraph: 8,     // Reduced slightly
        title_length: 8,                  // Reduced slightly
        meta_description_keyphrase: 10,
        meta_description_length: 8,       // Reduced slightly
        h1_seo_title_match: 5,            // INCREASED from 3
        h2_structure: 4,                  // INCREASED from 3
        headings_structure: 6,            // INCREASED from 5
        internal_links: 5,
        keyphrase_in_h1: 6,               // INCREASED from 4
        keyphrase_in_h2: 4,               // INCREASED from 2
        keyphrase_in_h3: 3,               // INCREASED from 2
        keyphrase_in_alt: 3,              // INCREASED from 2
        content_warning: 0                // Special warning doesn't contribute to score
    };

    var totalScore = 0;
    var totalWeight = 0;

    var checkName;
    for ( checkName in weights ) {
        if ( weights.hasOwnProperty( checkName ) ) {
            var weight = weights[ checkName ];
            var check = checks[ checkName ];
            if ( check && typeof check.score === 'number' && weight > 0 ) {
                totalScore += weight * ( check.score / 100 );
                totalWeight += weight;
            }
        }
    }

    var finalScore = totalWeight > 0 ? ( totalScore / totalWeight ) * 100 : 0;
    
    return finalScore;
};

	/**
	 * Strip HTML tags
	 * 
	 * @param {string} html HTML content
	 * @return {string} Plain text
	 */
	DPGSEOAnalyzer.prototype.stripHtml = function( html ) {
		var div = document.createElement( 'div' );
		div.innerHTML = html || '';
		return div.textContent || div.innerText || '';
	};

	/**
	 * Count words in text
	 * 
	 * @param {string} text Text content
	 * @return {number} Word count
	 */
	DPGSEOAnalyzer.prototype.countWords = function( text ) {
		return ( text || '' ).trim().split( /\s+/ ).filter( Boolean ).length;
	};

	/**
	 * Count occurrences of phrase in text
	 * 
	 * @param {string} text Text content
	 * @param {string} phrase Phrase to count
	 * @return {number} Occurrence count
	 */
    DPGSEOAnalyzer.prototype.countOccurrences = function( text, phrase ) {
        if ( ! text || ! phrase ) {
            return 0;
        }
        
        var escapedPhrase = phrase.replace( /[.*+?^${}()|[\]\\]/g, '\\$&' );
        var regex = new RegExp( escapedPhrase, 'gi' );
        var matches = text.match( regex );
        
        return matches ? matches.length : 0;
    };

	/**
	 * Show loading state
	 */
DPGSEOAnalyzer.prototype.showLoadingState = function () {
    const scoreEl = document.getElementById('seo-score-number');
    if (scoreEl) {
        scoreEl.textContent = '...';
        scoreEl.className   = 'dpg-seo-score-number dpg-seo-loading';
    }

    /* reset hidden field while loading */
    const scoreField = document.querySelector(this.options.scoreField);
    if (scoreField) scoreField.value = '';
};

	/**
	 * Show empty state
	 */
DPGSEOAnalyzer.prototype.showEmptyState = function () {
    const scoreEl = document.getElementById('seo-score-number');
    if (scoreEl) {
        scoreEl.textContent = '0';
        scoreEl.className   = 'dpg-seo-score-number dpg-seo-poor';
    }

    /* reflect the empty state in the hidden field */
    const scoreField = document.querySelector(this.options.scoreField);
    if (scoreField) scoreField.value = 0;

    this.updateSEOSection('seo-problems',     'seo-problems-list',     []);
    this.updateSEOSection('seo-improvements', 'seo-improvements-list', []);
    this.updateSEOSection('seo-good',         'seo-good-list',         []);

    const helpEl = document.getElementById('seo-help');
    if (helpEl) {
        helpEl.style.display = 'block';
        helpEl.innerHTML = '<p>' + __( 'Add content and a focus keyphrase to see SEO analysis.', 'dpg' ) + '</p>';
    }
};

	/**
	 * Show disabled message
	 */
	DPGSEOAnalyzer.prototype.showDisabledMessage = function() {
		var $container = $( this.options.container );
		if ( $container.length ) {
			$container.html(
				'<div class="dpg-seo-help">' +
				'<p>' + __( 'SEO analysis is disabled for this form.', 'dpg' ) + '</p>' +
				'</div>'
			);
		}
	};

	/**
	 * Check if minimum data is available
	 * 
	 * @param {Object} data Analysis data
	 * @return {boolean} Has minimum data
	 */
	DPGSEOAnalyzer.prototype.hasMinimumData = function( data ) {
		var hasContent = data.content.trim().length > 0 || 
						( data.template_source === 'page' && data.source_page_id );
		var hasKeyphrase = data.focus_keyphrase.trim().length > 0;
		
		return hasContent || hasKeyphrase;
	};

	/**
	 * Show error message
	 * 
	 * @param {Error} err Error object
	 */
	DPGSEOAnalyzer.prototype.showError = function( err ) {
		var $container = $( this.options.container );
		var $notice = $container.find( '.dpg-seo-error' );
		if ( ! $notice.length ) {
			$notice = $( '<div>', {
				'class': 'notice notice-error is-dismissible dpg-seo-error',
				style: 'margin: 10px 0;'
			} ).prependTo( $container );
		}
		
		var errorMessage = err.message || err.toString() || __( 'Unknown error occurred', 'dpg' );
		$notice.html(
			'<p>' + sprintf( __( 'SEO analysis failed: %s', 'dpg' ), errorMessage ) + '</p>'
		);

		setTimeout( function() {
			$notice.fadeOut();
		}, 5000 );
	};



	/**
	 * Update SEO section
	 * 
	 * @param {string} sectionId Section ID
	 * @param {string} listId List ID
	 * @param {Array} items Items to display
	 */
	DPGSEOAnalyzer.prototype.updateSEOSection = function( sectionId, listId, items ) {
		var section = document.getElementById( sectionId );
		var list = document.getElementById( listId );
		
		if ( ! section || ! list ) {
			return;
		}

		if ( ! items.length ) {
			section.style.display = 'none';
			return;
		}

		section.style.display = 'block';
		list.innerHTML = items.map( function( item ) {
			return '<div class="dpg-seo-item">' +
				'<div class="dpg-seo-item-message">' + item.message + '</div>' +
				( item.how_to_fix ? '<div class="dpg-seo-item-fix">' + item.how_to_fix + '</div>' : '' ) +
				'</div>';
		} ).join( '' );
	};

	/**
	 * Setup tabs
	 */
	DPGSEOAnalyzer.prototype.setupTabs = function() {
		var $tabs = $( '.dpg-seo-tab' );
		if ( $tabs.length ) {
			$tabs.first().addClass( 'active' );
		}
	};

	/**
	 * Switch tab
	 * 
	 * @param {string} tabName Tab name
	 */
	DPGSEOAnalyzer.prototype.switchTab = function( tabName ) {
		$( '.dpg-seo-tab' ).removeClass( 'active' );
		$( '.dpg-seo-tab[data-tab="' + tabName + '"]' ).addClass( 'active' );
		
		$( '.dpg-seo-tab-content' ).hide();
		$( '#dpg-seo-tab-' + tabName ).show();
	};

	/**
	 * Toggle check details
	 * 
	 * @param {jQuery} $element Element to toggle
	 */
	DPGSEOAnalyzer.prototype.toggleCheckDetails = function( $element ) {
		var $details = $element.next( '.dpg-seo-check-details' );
		if ( $details.length ) {
			$details.slideToggle();
			$element.toggleClass( 'expanded' );
		}
	};

	/**
	 * Update focus keyphrase
	 * 
	 * @param {string} keyphrase New keyphrase
	 */
	DPGSEOAnalyzer.prototype.updateFocusKeyphrase = function( keyphrase ) {
		var $display = $( '.dpg-current-keyphrase' );
		if ( $display.length ) {
			$display.text( keyphrase || __( '(none set)', 'dpg' ) );
		}
		
		this.debounceAnalysis();
	};

	/**
	 * Destroy analyzer
	 */
	DPGSEOAnalyzer.prototype.destroy = function() {
		$( document ).off( '.dpgseo' );
		clearTimeout( this.debounceTimer );
		this.analysisInProgress = false;
		this.pageContentCache.clear();
		
		if ( this.debug ) {
	
		}
	};

	// =========================
	// GLOBAL FUNCTIONS
	// =========================

	/**
	 * Global function to update SEO analysis
	 */
	window.dpgUpdateSEOAnalysis = function() {
		if ( window.dpgSEOAnalyzer && window.dpgSEOAnalyzer.runAnalysis ) {
			window.dpgSEOAnalyzer.runAnalysis();
		}
	};

	/**
	 * Global function to update SEO display
	 * 
	 * @param {Object} data Display data
	 */
	window.dpgUpdateSEODisplay = function( data ) {
		if ( window.dpgSEOAnalyzer && window.dpgSEOAnalyzer.updateAnalysisDisplay ) {
			window.dpgSEOAnalyzer.updateAnalysisDisplay( data );
		}
	};

jQuery( document ).on( 'change', '#dpg_source_page_id', function () {

	// 1 – which page did the user pick?
	const id = this.value;
	if ( ! id ) return;            // user cleared the dropdown

	// 2 – fetch fresh data for that page
	jQuery.get( dpgForm.ajaxUrl, { action: 'dpg_get_page_data', id } )
		.done( function ( markup ) {

			// 3 – inject / replace the <script id="dpg-page-data-{id}"> tag
			jQuery( '#dpg-page-data-' + id ).remove();
			jQuery( 'body' ).append( markup );

			// 4 – kick the analyser
			if ( window.dpgSEOAnalyzer ) {


				window.dpgSEOAnalyzer.pageContentCache.delete( 'wp_page_' + id );
				/* (alternatively you could do
				   window.dpgSEOAnalyzer.pageContentCache.clear(); ) */

				window.dpgSEOAnalyzer.runAnalysis();
			}
		} );
} );

	/**
	 * Global function to update SEO section
	 * 
	 * @param {string} sectionId Section ID
	 * @param {string} listId List ID
	 * @param {Array} items Items
	 */
	window.dpgUpdateSEOSection = function( sectionId, listId, items ) {
		if ( window.dpgSEOAnalyzer && window.dpgSEOAnalyzer.updateSEOSection ) {
			window.dpgSEOAnalyzer.updateSEOSection( sectionId, listId, items );
		}
	};

	// =========================
	// INITIALIZATION
	// =========================

	$( document ).ready( function() {
		if ( window.dpgForm ) {
			initializeSEOAnalyzer();
		} else {
			setTimeout( function() {
				initializeSEOAnalyzer();
			}, 100 );
		}

		/**
		 * Initialize SEO analyzer
		 */
		function initializeSEOAnalyzer() {
			try {
				var config = window.dpgForm && window.dpgForm.seoAnalyzer ? window.dpgForm.seoAnalyzer : {};
				var options = config.options || {};

				if ( options.enabled !== false && $( '.dpg-seo-analysis' ).length ) {
					window.dpgSEOAnalyzer = new DPGSEOAnalyzer( options );
					
					if ( window.dpgForm && window.dpgForm.debug ) {
					
					}
				}
			} catch ( error ) {
				if ( window.dpgForm && window.dpgForm.debug ) {
					console.error( '[DPG SEO] Failed to initialize Unified SEO Analyzer:', error );
				}
			}
		}
	} );

	$( window ).on( 'beforeunload', function() {
		if ( window.dpgSEOAnalyzer && window.dpgSEOAnalyzer.destroy ) {
			window.dpgSEOAnalyzer.destroy();
		}
	} );

jQuery( document ).on( 'submit',
	'#dpg-create-template-form, #dpg-edit-template-form',
	function () {
		if ( window.dpgSEOAnalyzer && window.dpgSEOAnalyzer.analysisInProgress ) {
			/* Finish any pending analysis synchronously so the
			   hidden <input id="dpg_seo_score"> contains the
			   final 0-100 score when PHP receives $_POST.       */
			window.dpgSEOAnalyzer.runAnalysis();
		}
});

} )( jQuery, window.wp || {} );