<?php
// File: wp-content/plugins/dynamic-seo-pages/includes/forms/class-dpg-form-edit.php

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

// Ensure WP_Filesystem is available
if ( ! function_exists( 'WP_Filesystem' ) ) {
    require_once ABSPATH . 'wp-admin/includes/file.php';
}

require_once DPG_PLUGIN_DIR . 'includes/forms/class-dpg-form-base.php';

/**
 * Edit Template Form Class
 * Handles editing existing templates with SEO analysis
 */
class DPG_Form_Edit extends DPG_Form_Base {

    /** @var DPG_SEO_Analyzer */
    private $seo_analyzer;

    /** @var array */
    private $template_data;

    /** @var string */
    private $ctx_id;

    /** @var array */
    protected $config;

    /**
     * Constructor.
     *
     * @param string $ctx_id  Context ID (filename) of template to edit.
     * @param array  $config  Configuration from DPG_Admin.
     */
public function __construct( $ctx_id, $config = [] ) {
    $this->config = $config;
    parent::__construct( 'edit-template', '', 'dpg_update_template' );
    
    // Sanitize context ID
    $this->ctx_id = sanitize_text_field( $ctx_id );
    
    // Load template data
    $this->load_template_data();

    // Register WordPress action handlers
    add_action( 'admin_post_dpg_update_template', [ $this, 'handle_submission' ] );
    
    // Register AJAX handlers with proper security
    $this->register_ajax_handlers();
}


private function get_current_source_page_id() {
        // For edit form, first check POST data (if form was submitted)
        if ( isset( $_POST['dpg_source_page_id'] ) && is_numeric( $_POST['dpg_source_page_id'] ) ) {
            return intval( $_POST['dpg_source_page_id'] );
        }
        
        // Otherwise get from loaded template data
        if ( ! empty( $this->template_data['source_page_id'] ) ) {
            return intval( $this->template_data['source_page_id'] );
        }
        
        return 0;
    }

/**
 * Load existing template JSON into $this->template_data.
 */
private function load_template_data() {
    global $wp_filesystem;
    WP_Filesystem();

    $json_file = trailingslashit( DPG_DATA_DIR ) . $this->ctx_id . '.json';

    if ( ! $wp_filesystem->exists( $json_file ) ) {
        wp_die( __( 'Template not found.', 'dpg' ) );
    }

    $raw = $wp_filesystem->get_contents( $json_file );
    
    // WordPress compliant debug logging - only when WP_DEBUG_LOG is also enabled
    if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
        // Use WordPress error_log function and limit sensitive data exposure
        error_log( 
            sprintf( 
                '[DPG] Loading template data for ID: %s (length: %d bytes)', 
                sanitize_text_field( $this->ctx_id ), 
                strlen( $raw ) 
            ) 
        );
    }
    
    // Handle potential encoding issues
    if ( $raw === false ) {
        wp_die( __( 'Could not read template file.', 'dpg' ) );
    }
    
    // Remove BOM and trim whitespace
    $raw = trim( $raw );
    if ( substr( $raw, 0, 3 ) === "\xEF\xBB\xBF" ) {
        $raw = substr( $raw, 3 );
    }
    
    // Try to decode JSON
    $data = json_decode( $raw, true );
    
    // Check for JSON errors - WordPress compliant error handling
    if ( json_last_error() !== JSON_ERROR_NONE ) {
        if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
            error_log( 
                sprintf( 
                    '[DPG] JSON decode error for template %s: %s', 
                    sanitize_text_field( $this->ctx_id ), 
                    json_last_error_msg() 
                ) 
            );
            // Don't log raw content - potential security issue
        }
        
        /* translators: %s: JSON error message */
        wp_die( 
            sprintf( 
                __( 'Invalid template data. JSON decode error: %s', 'dpg' ), 
                esc_html( json_last_error_msg() ) 
            ) 
        );
    }
    
    // Validate that we have an array
    if ( ! is_array( $data ) ) {
        if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
            error_log( 
                sprintf( 
                    '[DPG] Invalid data type for template %s. Expected array, got: %s', 
                    sanitize_text_field( $this->ctx_id ), 
                    gettype( $data ) 
                ) 
            );
        }
        
        /* translators: %s: data type received */
        wp_die( 
            sprintf( 
                __( 'Invalid template data format. Expected array, got: %s', 'dpg' ), 
                esc_html( gettype( $data ) ) 
            ) 
        );
    }
    
    // Validate required fields and provide defaults
    $data = wp_parse_args( $data, [
        'name'            => sanitize_text_field( $this->ctx_id ),
        'type'            => 'keyword',
        'template_html'   => '',
        'template_source' => 'custom',
        'source_page_id'  => 0,
        'items'           => [],
        'hard_meta'       => [],
        'focus_keyphrase' => '',
        'include_header'  => true,
        'include_footer'  => true,
        'created_date'    => '',
        'last_modified'   => '',
        'score'           => ''
    ] );
    
    // Special handling for template_html based on template_source
    if ( 'page' === $data['template_source'] && ! empty( $data['source_page_id'] ) ) {
        // If using page source, get the page content and ensure {item} placeholder exists
        $page = get_post( absint( $data['source_page_id'] ) );
        if ( $page && 'publish' === $page->post_status ) {
            $page_content = $page->post_content;
            
            // If the page content doesn't contain {item}, add it as a comment for validation
            if ( false === strpos( $page_content, '{item}' ) ) {
                $page_content = "<!-- Dynamic content: {item} -->\n" . $page_content;
            }
            
            $data['template_html'] = $page_content;
        }
    } else {
        // For custom templates, ensure {item} placeholder exists
        if ( ! empty( $data['template_html'] ) && false === strpos( $data['template_html'], '{item}' ) ) {
            // Add {item} placeholder as a comment if it's missing
            $data['template_html'] = "<!-- Dynamic content: {item} -->\n" . $data['template_html'];
        }
    }
    
    // Ensure hard_meta is properly formatted with sanitization
    if ( ! is_array( $data['hard_meta'] ) ) {
        $data['hard_meta'] = [];
    }
    
    $data['hard_meta'] = wp_parse_args( $data['hard_meta'], [
        'title'              => '',
        'description'        => '',
        'keywords'           => '',
        'og_title'           => '',
        'og_description'     => '',
        'og_image'           => '',
        'twitter_title'      => '',
        'twitter_description' => '',
        'twitter_image'      => ''
    ] );
    
    // Sanitize hard_meta values
    foreach ( $data['hard_meta'] as $key => $value ) {
        if ( in_array( $key, [ 'og_image', 'twitter_image' ], true ) ) {
            $data['hard_meta'][ $key ] = esc_url_raw( $value );
        } elseif ( in_array( $key, [ 'description', 'og_description', 'twitter_description', 'keywords' ], true ) ) {
            $data['hard_meta'][ $key ] = sanitize_textarea_field( $value );
        } else {
            $data['hard_meta'][ $key ] = sanitize_text_field( $value );
        }
    }
    
    // Ensure items is an array and sanitize
    if ( ! is_array( $data['items'] ) ) {
        $data['items'] = [];
    }
    
    // Sanitize items array
    foreach ( $data['items'] as $index => $item ) {
        if ( is_array( $item ) ) {
            $data['items'][ $index ] = [
                'name'  => sanitize_text_field( $item['name'] ?? '' ),
                'score' => isset( $item['score'] ) ? floatval( $item['score'] ) : null
            ];
        } else {
            $data['items'][ $index ] = sanitize_text_field( $item );
        }
    }
    
    // Final WordPress compliant debug log
    if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
        error_log( 
            sprintf( 
                '[DPG] Template loaded successfully - ID: %s, Name: %s, Items: %d, Source: %s, Has placeholder: %s', 
                sanitize_text_field( $this->ctx_id ),
                sanitize_text_field( $data['name'] ),
                count( $data['items'] ),
                sanitize_text_field( $data['template_source'] ),
                ( false !== strpos( $data['template_html'], '{item}' ) ) ? 'Yes' : 'No'
            ) 
        );
    }

    $this->template_data = $data;
}


    /**
     * Enqueue this form's extra assets.
     */
public function enqueue_assets() {
    /* 0) Shared bundles from the base class */
    parent::enqueue_assets();   // registers + enqueues dpg-components-js

    /* 1) CodeMirror for the HTML editor */
    wp_enqueue_code_editor( [ 'type' => 'text/html' ] );
    wp_enqueue_script( 'wp-theme-plugin-editor' );
    wp_enqueue_style( 'wp-codemirror' );

    /* 2) Edit-form specific styling (kept unchanged) */
    wp_enqueue_style(
        'dpg-form-components',
        plugin_dir_url( DPG_PLUGIN_FILE ) . 'includes/forms/assets/css/components.css',
        [ 'dpg-admin-forms' ],
        DPG_VERSION
    );

    /* 3) SEO analyser – dependency chain fixed here */
    wp_enqueue_script(
        'dpg-unified-seo-analyzer',
        plugin_dir_url( DPG_PLUGIN_FILE ) . 'includes/forms/assets/js/seo-analyzer.js',
        [
            'jquery',
            'wp-i18n',
            'dpg-components-js',      // ← critical addition
        ],
        DPG_VERSION,
        true
    );

    /* 4) FIXED: Enhanced localised config & nonces for Edit Form */
    wp_localize_script( 'dpg-unified-seo-analyzer', 'dpgEditForm', [
        'ajaxUrl' => admin_url( 'admin-ajax.php' ),
        'nonces'  => [
            'form'      => wp_create_nonce( 'dpg_form_nonce' ),
            'page_data' => wp_create_nonce( 'dpg_form_nonce' ), // FIXED: Use same nonce action
            'ajax'      => wp_create_nonce( 'dpg_form_nonce' ), // FIXED: Additional nonce field
        ],
        'strings' => [
            'loading'        => __( 'Loading...', 'dpg' ),
            'error'          => __( 'Error loading data.', 'dpg' ),
            'security_error' => __( 'Security verification failed.', 'dpg' ),
        ],
        'debug' => defined( 'WP_DEBUG' ) && WP_DEBUG, // FIXED: Add debug flag
    ] );
    
    /* 5) FIXED: Also localize for global access (same as Create form) */
    wp_localize_script( 'dpg-components-js', 'dpgCreateForm', [
        'ajaxUrl' => admin_url( 'admin-ajax.php' ),
        'nonces'  => [
            'form'      => wp_create_nonce( 'dpg_form_nonce' ),
            'page_data' => wp_create_nonce( 'dpg_form_nonce' ),
            'ajax'      => wp_create_nonce( 'dpg_form_nonce' ),
        ],
        'debug' => defined( 'WP_DEBUG' ) && WP_DEBUG,
    ] );
    
    /* 6) ADDITIONAL: Global dpgForm object for compatibility */
    wp_localize_script( 'dpg-components-js', 'dpgForm', [
        'ajaxUrl' => admin_url( 'admin-ajax.php' ),
        'nonces'  => [
            'form'      => wp_create_nonce( 'dpg_form_nonce' ),
            'page_data' => wp_create_nonce( 'dpg_form_nonce' ),
            'ajax'      => wp_create_nonce( 'dpg_form_nonce' ),
        ],
        'debug' => defined( 'WP_DEBUG' ) && WP_DEBUG,
    ] );
}

public function embed_page_data_for_seo( $page_id ) {
    if ( ! $page_id ) {
        if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
            error_log( '[DPG] embed_page_data_for_seo: No page ID provided' );
        }
        return;
    }
    
    $fetched_post = get_post( $page_id );
    if ( ! $fetched_post || is_wp_error( $fetched_post ) ) {
        if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
            error_log( "[DPG] embed_page_data_for_seo: Page {$page_id} not found or error occurred" );
        }
        return;
    }
    
    // Allow draft/private pages for logged-in users with edit capability
    $allowed_statuses = [ 'publish' ];
    if ( current_user_can( 'edit_post', $fetched_post->ID ) ) {
        $allowed_statuses = [ 'publish', 'draft', 'private' ];
    }
    
    if ( ! in_array( $fetched_post->post_status, $allowed_statuses, true ) ) {
        if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
            error_log( "[DPG] embed_page_data_for_seo: Page {$page_id} has invalid status: {$fetched_post->post_status}" );
        }
        return;
    }
    
    // CORRECT FIX: Properly store and restore the global $post
    global $post;
    $original_post = $post; // Store the original global post
    
    // Safely setup post data
    $post = $fetched_post;
    setup_postdata( $post );
    
    // FIXED: Safely access post properties with null checks
    $raw_content = $post->post_content ?? '';
    $post_title = $post->post_title ?? '';
    $post_excerpt = $post->post_excerpt ?? '';
    $post_id = $post->ID ?? 0;
    $post_status = $post->post_status ?? 'unknown';
    
    // Get processed content (this handles page builders, shortcodes, etc.)
    $processed_content = '';
    $title = '';
    
    try {
        $processed_content = apply_filters( 'the_content', $raw_content );
        $title = apply_filters( 'the_title', $post_title, $post_id );
    } catch ( Exception $e ) {
        if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
            error_log( "[DPG] Error processing content for page {$page_id}: " . $e->getMessage() );
        }
        // Fallback to raw content
        $processed_content = $raw_content;
        $title = $post_title;
    }
    
    // Clean up - restore original post
    wp_reset_postdata();
    $post = $original_post;
    
    // Ensure we have the {item} placeholder for template analysis
    if ( false === strpos( $processed_content, '{item}' ) ) {
        // Add {item} placeholder as a comment so analysis works
        $processed_content = "<!-- Dynamic content: {item} -->\n" . $processed_content;
    }
    
    // Create comprehensive page data for JavaScript fallback
    $page_data = array(
        'content' => $processed_content,
        'raw_content' => $raw_content,
        'title' => $title,
        'excerpt' => $post_excerpt ?: wp_trim_words( wp_strip_all_tags( $processed_content ), 25 ),
        'id' => $post_id,
        'status' => $post_status,
        'content_length' => strlen( $processed_content ),
        'raw_content_length' => strlen( $raw_content ),
        'has_item_placeholder' => ( false !== strpos( $processed_content, '{item}' ) ),
        'has_shortcodes' => has_shortcode( $raw_content, '' ),
        'method' => 'server_embedded_processed_edit_correct_fix',
        'permalink' => get_permalink( $post_id ),
    );
    
    // Embed as JSON in a hidden script tag
    printf(
        '<script type="application/json" id="dpg-page-data-%d">%s</script>',
        $page_id,
        wp_json_encode( $page_data, JSON_UNESCAPED_UNICODE )
    );
    
    // DEBUG: Output for troubleshooting
    if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
        printf(
            '<!-- DPG EDIT DEBUG CORRECT: Embedded page %d, raw: %d chars, processed: %d chars, has {item}: %s, has shortcodes: %s -->',
            $page_id,
            strlen( $raw_content ),
            strlen( $processed_content ),
            $page_data['has_item_placeholder'] ? 'Yes' : 'No',
            $page_data['has_shortcodes'] ? 'Yes' : 'No'
        );
    }
}

/**
 * WordPress-compliant AJAX handler for page data
 */
public function ajax_get_page_data() {
    // Nonce verification
    if ( ! check_ajax_referer( 'dpg_form_nonce', 'nonce', false ) ) {
        wp_send_json_error( __( 'Security verification failed.', 'dpg' ), 403 );
    }

    // Capability check
    if ( ! current_user_can( 'manage_options' ) ) {
        wp_send_json_error( __( 'Permission denied.', 'dpg' ), 403 );
    }
    
    // Validate and sanitize input
    $page_id = isset( $_GET['id'] ) ? absint( $_GET['id'] ) : 0;
    if ( ! $page_id ) {
        wp_send_json_error( __( 'Invalid page ID.', 'dpg' ), 400 );
    }
    
    // Use the comprehensive function from your main plugin file
    if ( function_exists( 'dpg_ajax_get_comprehensive_page_data' ) ) {
        dpg_ajax_get_comprehensive_page_data();
    } else {
        wp_send_json_error( __( 'AJAX handler not available.', 'dpg' ), 500 );
    }
}

public function register_ajax_handlers() {
    add_action( 'wp_ajax_dpg_get_page_data', [ $this, 'ajax_get_page_data' ] );
    add_action( 'wp_ajax_nopriv_dpg_get_page_data', [ $this, 'ajax_get_page_data' ] );
}

    /**
     * Tell the base we need SEO features.
     *
     * @return bool
     */
    protected function has_seo_features() {
        return true;
    }

/**
     * Render the edit template form
     * REPLACE your existing render() method with this complete version
     */
    public function render() {
        $this->enqueue_assets();

        $template_name = $this->template_data['name'] ?? $this->ctx_id;

        echo '<div class="dpg-edit-template-wrapper">';

        // Form Header
        $this->render_form_header(
            sprintf( __( 'Edit Template: %s', 'dpg' ), esc_html( $template_name ) ),
            sprintf(
                /* translators: 1: template ID, 2: template type */
                __( 'Template ID: %1$s | Type: %2$s', 'dpg' ),
                esc_html( $this->ctx_id ),
                esc_html( ucfirst( str_replace( '-', ' ', $this->template_data['type'] ?? '' ) ) )
            )
        );

        // Begin form with proper class for JavaScript validation
        $this->start_form( 'dpg-edit-template-form dpg-form' );
        
        // WordPress nonce fields (both for compatibility)
        wp_nonce_field( $this->action, 'dpg_nonce' );
        wp_nonce_field( 'dpg_form_nonce', '_wpnonce' );

        // WordPress security fields
        printf( '<input type="hidden" name="dpg_user_id" value="%d">', esc_attr( get_current_user_id() ) );
        printf( '<input type="hidden" name="dpg_timestamp" value="%d">', esc_attr( time() ) );
        printf( '<input type="hidden" name="dpg_ctx_id" value="%s">', esc_attr( $this->ctx_id ) );

        // Display WordPress notices
        if ( ! empty( $this->errors ) ) {
            $this->render_notice(
                wp_kses_post( implode( '<br>', array_map( 'esc_html', $this->errors ) ) ),
                'error'
            );
        }

        // ROW 1: Template Information (Full Width)
        $this->render_section(
            __( 'Template Information', 'dpg' ),
            [ $this, 'render_template_info' ],
            __( 'Basic information about this template.', 'dpg' )
        );

        // ROW 2: Basic Configuration + SEO Settings (Two Columns)
        $this->render_section(
            __( 'Template Configuration & SEO', 'dpg' ),
            [ $this, 'render_row_one_fields' ],
            __( 'Configure basic settings and SEO optimization.', 'dpg' )
        );

        // ROW 3: Template Content (Full Width)
        $this->render_section(
            __( 'Template Content', 'dpg' ),
            [ $this, 'render_content_fields' ],
            __( 'Edit the HTML content. Use <code>{item}</code> for dynamic content.', 'dpg' )
        );

        // ROW 4: SEO Analysis (Full Width)
        $this->render_section(
            __( 'SEO Analysis', 'dpg' ),
            [ $this, 'render_seo_analysis' ],
        );

        // ROW 5: Items List (Full Width)
        $this->render_section(
            __( 'Items List', 'dpg' ),
            [ $this, 'render_items_fields' ],
            __( 'Edit the items list.', 'dpg' )
        );

        // ROW 6: Social Media (Open by default)
        $this->render_section(
            __( 'Social Media', 'dpg' ),
            [ $this, 'render_social_fields' ],
            __( 'Configure sharing metadata.', 'dpg' ),
            false // Not collapsible - always open
        );

        // ROW 7: Template Options (Open by default)
        $this->render_section(
            __( 'Template Options', 'dpg' ),
            [ $this, 'render_options_fields' ],
            __( 'Additional generation settings.', 'dpg' ),
            false // Not collapsible - always open
        );

        // ENHANCED: Embed page data for SEO analysis fallback with debugging
        $source_page_id = $this->get_current_source_page_id();
        if ( $source_page_id ) {
            $this->debug_page_content( $source_page_id ); // Debug output
            $this->embed_page_data_for_seo( $source_page_id ); // Embed page data
        }

        // WordPress form close
        $this->end_form( __( 'Update Template', 'dpg' ) );
        echo '</div>'; // .dpg-edit-template-wrapper
    }

    /**
     * ADD this debug helper method to your DPG_Form_Edit class
     * (Remove after testing is complete)
     */
private function debug_page_content( $page_id ) {
    // Only show debug info if WordPress debug is enabled AND user has proper capabilities
    if ( ! defined( 'WP_DEBUG' ) || ! WP_DEBUG || ! current_user_can( 'manage_options' ) ) {
        return;
    }
    
    if ( ! $page_id ) {
        printf( '<!-- DEBUG: No page ID provided -->' );
        return;
    }
    
    $post = get_post( $page_id );
    if ( ! $post || is_wp_error( $post ) ) {
        printf( '<!-- DEBUG: Page %d not found or error occurred -->', absint( $page_id ) );
        return;
    }
    
    // FIXED: Safely access post properties with null checks
    $post_title = $post->post_title ?? 'No Title';
    $post_status = $post->post_status ?? 'unknown';
    $raw_content = $post->post_content ?? '';
    
    // Safely process content
    $processed_content = '';
    try {
        $processed_content = apply_filters( 'the_content', $raw_content );
    } catch ( Exception $e ) {
        $processed_content = $raw_content;
        error_log( "[DPG] Error in debug_page_content for page {$page_id}: " . $e->getMessage() );
    }
    
    printf( 
        '<!-- DEBUG PAGE ANALYSIS FIXED:
        Page ID: %d
        Title: %s
        Status: %s
        Raw content length: %d
        Processed content length: %d
        Raw has {item}: %s
        Processed has {item}: %s
        First 200 chars of processed: %s
        -->',
        absint( $page_id ),
        esc_html( $post_title ),
        esc_html( $post_status ),
        strlen( $raw_content ),
        strlen( $processed_content ),
        ( false !== strpos( $raw_content, '{item}' ) ) ? 'Yes' : 'No',
        ( false !== strpos( $processed_content, '{item}' ) ) ? 'Yes' : 'No',
        esc_html( substr( wp_strip_all_tags( $processed_content ), 0, 200 ) )
    );
}


    /**
     * Render Template Information Section
     */
    public function render_template_info() {
        $type = ( substr( $this->ctx_id, -5 ) === '-area' ) ? 'service-area' : 'keyword';
        $slug_base = ( $type === 'service-area' ) ? substr( $this->ctx_id, 0, -5 ) : '';

        echo '<div class="dpg-template-info">';
        echo '<div class="dpg-info-grid">';

        echo '<div class="dpg-info-item">';
        echo '<strong>' . esc_html__( 'Template Type:', 'dpg' ) . '</strong> ';
        echo esc_html( $type === 'service-area' ? __( 'Service Area', 'dpg' ) : __( 'Keyword', 'dpg' ) );
        echo '</div>';

        if ( $type === 'service-area' ) {
            echo '<div class="dpg-info-item">';
            echo '<strong>' . esc_html__( 'Base Slug:', 'dpg' ) . '</strong> ';
            echo '<code>' . esc_html( $slug_base ) . '</code>';
            echo '</div>';
        }

        if ( ! empty( $this->template_data['created_date'] ) ) {
            echo '<div class="dpg-info-item">';
            echo '<strong>' . esc_html__( 'Created:', 'dpg' ) . '</strong> ';
            echo esc_html( $this->template_data['created_date'] );
            echo '</div>';
        }

        if ( ! empty( $this->template_data['last_modified'] ) ) {
            echo '<div class="dpg-info-item">';
            echo '<strong>' . esc_html__( 'Last Modified:', 'dpg' ) . '</strong> ';
            echo esc_html( $this->template_data['last_modified'] );
            echo '</div>';
        }

        echo '</div>';
        echo '</div>';
    }

    /**
     * Render Row 1: Basic Configuration + SEO Settings (Two Columns) - Same as Create Form
     */
    public function render_row_one_fields() {
        $hard_meta = $this->template_data['hard_meta'] ?? [];
        $focus_keyphrase = $this->template_data['focus_keyphrase'] ?? '';
        
        // Auto-derive focus keyphrase if empty
        if ( empty( $focus_keyphrase ) ) {
            $keywords = $hard_meta['keywords'] ?? '';
            if ( $keywords ) {
                $focus_keyphrase = trim( str_replace( '{item}', '', $keywords ), ', ' );
            }
            if ( ! $focus_keyphrase ) {
                $title = $hard_meta['title'] ?? '';
                if ( $title ) {
                    $focus_keyphrase = trim( str_replace( '{item}', '', $title ) );
                }
            }
        }

        echo '<div class="dpg-row-layout dpg-row-layout--two-col">';
        
        // LEFT COLUMN: Basic Configuration + Items
        echo '<div class="dpg-column">';
            echo '<h4 class="dpg-column-title">' . esc_html__( 'Configure basic settings for your template.', 'dpg' ) . '</h4>';
            
            // Template Name - FIRST as requested (80% width)
            $this->render_field( [
                'type'        => 'text',
                'id'          => 'dpg_name',
                'name'        => 'dpg_name',
                'label'       => __( 'Template Name', 'dpg' ),
                'required'    => true,
                'value'       => $this->template_data['name'] ?? '',
                'placeholder' => __( 'e.g., Best Web Designers', 'dpg' ),
                'description' => __( 'A descriptive name for this template.', 'dpg' ),
                'attributes'  => [ 'class' => 'dpg-input--80', 'readonly' => true ],
            ] );

            // Template Type - SECOND as requested (80% width)
            $type = $this->template_data['type'] ?? 'service-area';
            $this->render_field( [
                'type'        => 'select',
                'id'          => 'dpg_type',
                'name'        => 'dpg_type',
                'label'       => __( 'Template Type', 'dpg' ),
                'required'    => true,
                'options'     => [
                    'service-area' => __( 'Service or Product Area', 'dpg' ),
                    'keyword'      => __( 'Keyword Based',       'dpg' ),
                ],
                'value'       => $type,
                'description' => __( 'URLs like "plumbing-chicago" or "best-plumber".', 'dpg' ),
                'attributes'  => [ 'class' => 'dpg-input--80', 'disabled' => true ],
            ] );

            // URL Slug - THIRD as requested (80% width) - Only show for service-area
            if ( 'service-area' === $type ) {
                $slug_base = ( substr( $this->ctx_id, -5 ) === '-area' ) ? substr( $this->ctx_id, 0, -5 ) : '';
                echo '<div id="dpg-slug-field">';
                    $this->render_field( [
                        'type'        => 'text',
                        'id'          => 'dpg_slug',
                        'name'        => 'dpg_slug',
                        'label'       => __( 'URL Slug', 'dpg' ),
                        'required'    => true,
                        'value'       => $slug_base,
                        'placeholder' => __( 'e.g., web-designer', 'dpg' ),
                        'description' => __( 'Used in URLs like "web-designer-chicago".', 'dpg' ),
                        'attributes'  => [ 'class' => 'dpg-input--80', 'readonly' => true ],
                    ] );
                echo '</div>';
            }

            // Items List (show current count)
            $items = $this->template_data['items'] ?? [];
            $items_count = count( $items );
            echo '<h5 class="dpg-subsection-title">' . 
                sprintf( 
                    esc_html__( 'Items List (%d items)', 'dpg' ), 
                    $items_count 
                ) . 
            '</h5>';
            echo '<p class="dpg-subsection-description">' . esc_html__( 'Managed in the Items List section below.', 'dpg' ) . '</p>';
            
            // Show sample of current items
            if ( ! empty( $items ) ) {
                echo '<div style="background: var(--dpg-gray-50); padding: 12px; border-radius: 6px; border: 1px solid var(--dpg-gray-200);">';
                echo '<strong>' . esc_html__( 'Current Items Preview:', 'dpg' ) . '</strong><br>';
                $preview_items = array_slice( $items, 0, 3 );
                foreach ( $preview_items as $item ) {
                    $item_name = is_array( $item ) ? $item['name'] : $item;
                    echo '<code style="display: block; margin: 2px 0;">' . esc_html( $item_name ) . '</code>';
                }
                if ( $items_count > 3 ) {
                    echo '<small style="color: var(--dpg-gray-700);">' . 
                        sprintf( esc_html__( '... and %d more', 'dpg' ), $items_count - 3 ) . 
                    '</small>';
                }
                echo '</div>';
            }
        echo '</div>';
        
        // RIGHT COLUMN: SEO Settings
        echo '<div class="dpg-column">';
            echo '<h4 class="dpg-column-title">' . esc_html__( 'Optimize your template for search engines.', 'dpg' ) . '</h4>';
            
            // Focus Keyphrase (80% width)
            $this->render_field( [
                'type'        => 'text',
                'id'          => 'dpg_focus_keyphrase',
                'name'        => 'dpg_focus_keyphrase',
                'label'       => __( 'Focus Keyphrase', 'dpg' ),
                'value'       => $focus_keyphrase,
                'placeholder' => __( 'e.g., web designer', 'dpg' ),
                'description' => __( 'Main phrase for SEO analysis.', 'dpg' ),
                'attributes'  => [ 'class' => 'dpg-input--80' ],
            ] );

            // SEO Title (80% width)
            $this->render_field( [
                'type'        => 'text',
                'id'          => 'dpg_meta_title',
                'name'        => 'dpg_meta_title',
                'label'       => __( 'SEO Title', 'dpg' ),
                'value'       => $hard_meta['title'] ?? '',
                'placeholder' => __( 'Best {item} – Professional Services', 'dpg' ),
                'description' => __( 'Appears in search results.', 'dpg' ),
                'attributes'  => [
                    'maxlength' => '60',
                    'class' => 'dpg-input--80'
                ],
            ] );
            echo '<div class="dpg-field-help">';
                $title_length = mb_strlen( $hard_meta['title'] ?? '' );
                echo '<span id="title-counter">' . max( 0, 60 - $title_length ) . '</span> ' . esc_html__( 'characters remaining', 'dpg' );
            echo '</div>';

            // Meta Description
            $this->render_field( [
                'type'        => 'textarea',
                'id'          => 'dpg_meta_description',
                'name'        => 'dpg_meta_description',
                'label'       => __( 'Meta Description', 'dpg' ),
                'value'       => $hard_meta['description'] ?? '',
                'placeholder' => __( 'Brief summary with {item}.', 'dpg' ),
                'description' => __( 'Appears in search results.', 'dpg' ),
                'rows'        => 3,
                'attributes'  => [
                    'maxlength' => '160'
                ],
            ] );
            echo '<div class="dpg-field-help">';
                $desc_length = mb_strlen( $hard_meta['description'] ?? '' );
                echo '<span id="description-counter">' . max( 0, 160 - $desc_length ) . '</span> ' . esc_html__( 'characters remaining', 'dpg' );
            echo '</div>';

            // Meta Keywords
            $this->render_field( [
                'type'        => 'textarea',
                'id'          => 'dpg_meta_keywords',
                'name'        => 'dpg_meta_keywords',
                'label'       => __( 'Meta Keywords', 'dpg' ),
                'value'       => $hard_meta['keywords'] ?? '',
                'placeholder' => __( '{item}, services, local', 'dpg' ),
                'description' => __( 'Comma-separated keywords.', 'dpg' ),
                'rows'        => 2,
            ] );
        echo '</div>';
        
        echo '</div>'; // End dpg-row-layout
    }

    /**
     * Render content fields - Same structure as Create Form
     */
public function render_content_fields() {
    /* --------------------------------------------------------------------
     *  a) existing settings + any fresh POST override
     * ------------------------------------------------------------------ */
    $template_source = isset( $_POST['dpg_template_source'] )
        ? sanitize_key( $_POST['dpg_template_source'] )
        : ( $this->template_data['template_source'] ?? 'custom' );

    $source_page_id  = isset( $_POST['dpg_source_page_id'] )
        ? absint( $_POST['dpg_source_page_id'] )
        : absint( $this->template_data['source_page_id'] ?? 0 );

    /* --------------------------------------------------------------------
     *  b) Template-source selector
     * ------------------------------------------------------------------ */
    echo '<div class="dpg-field-grid dpg-field-grid--mixed">';
        echo '<div>';
            $this->render_field( [
                'type'       => 'select',
                'id'         => 'dpg_template_source',
                'name'       => 'dpg_template_source',
                'label'      => __( 'Template Source', 'dpg' ),
                'required'   => true,
                'options'    => [
                    'custom' => __( 'Custom HTML',   'dpg' ),
                    'page'   => __( 'Existing Page', 'dpg' ),
                ],
                'value'      => $template_source,
                'attributes' => [ 'class' => 'dpg-input--medium' ],
                'description'=> __( 'Choose where the HTML should come from.', 'dpg' ),
            ] );
        echo '</div><div></div>';
    echo '</div>';

    /* --------------------------------------------------------------------
     *  c) Page dropdown with data-in-use flag
     * ------------------------------------------------------------------ */
    $show_page_sel = ( $template_source === 'page' ) ? 'block' : 'none';

    echo '<div id="dpg-page-selection"
               class="dpg-field-grid dpg-field-grid--mixed"
               style="display:' . esc_attr( $show_page_sel ) . ';">';

        echo '<div>';

            echo '<label for="dpg_source_page_id" class="dpg-field-label">'
                   . esc_html__( 'Select Page', 'dpg' ) .
                 '</label>';

            echo '<select id="dpg_source_page_id"
                           name="dpg_source_page_id"
                           class="dpg-select dpg-input--medium">';

                printf(
                    '<option value="">%s</option>',
                    esc_html__( 'Select a page…', 'dpg' )
                );

                $pages = get_pages( [
                    'post_status' => 'publish',
                    'number'      => 200,
                    'sort_column' => 'post_title',
                ] );

                foreach ( $pages as $p ) {

                    /* Skip our own page when checking "in use" so it isn't flagged */
                    $in_use = $this->get_page_assigned_template( $p->ID, $this->ctx_id ) ? 1 : 0;

                    printf(
                        '<option value="%1$d" data-in-use="%3$d"%4$s>%2$s</option>',
                        $p->ID,
                        esc_html( $p->post_title . " (ID: {$p->ID})" ),
                        $in_use,
                        selected( $p->ID, $source_page_id, false )
                    );
                }

            echo '</select>';

                /* ── NEW inline banner ─────────────────────────── */
                echo '<div id="dpg-page-in-use-alert"
                        class="dpg-notice dpg-notice-error dpg-page-alert"
                        style="display:none;">'
                . esc_html__(
                    'You can\'t use this page – it is already assigned to another template. ' .
                    'Delete that template, or re-assign it to a different page, before you can use this one.',
                    'dpg'
                )
                . '</div>';

                /* help text, now re-worded */
                echo '<p class="dpg-field-description">'
                . esc_html__(
                    'Select a different page, or free this page up by editing/deleting the template that currently uses it.',
                    'dpg'
                )
                . '</p>';

        echo '</div><div></div>';
    echo '</div>'; // #dpg-page-selection



    /* --------------------------------------------------------------------
     *  d) Custom-HTML textarea
     * ------------------------------------------------------------------ */
    $show_html = ( $template_source === 'custom' ) ? 'block' : 'none';

    echo '<div id="dpg-custom-template"
               class="dpg-field dpg-field-textarea dpg-field-required"
               style="display:' . esc_attr( $show_html ) . ';">';

        echo '<label for="dpg_template_html" class="dpg-field-label">'
             . esc_html__( 'Template HTML', 'dpg' ) .
             ' <span class="dpg-required">*</span></label>';

        echo '<div class="dpg-field-input">';
            $current_html = isset( $_POST['dpg_template_html'] )
                ? wp_unslash( $_POST['dpg_template_html'] )
                : ( $this->template_data['template_html'] ?? '' );

            echo '<textarea name="dpg_template_html" id="dpg_template_html"
                            rows="15"
                            class="dpg-input dpg-textarea dpg-code-editor">'
                 . esc_textarea( $current_html ) .
                 '</textarea>';

            echo '<p class="dpg-field-description">'
                   . wp_kses_post( __( 'Use <code>{item}</code> where the item name should be inserted.', 'dpg' ) ) .
                 '</p>';
        echo '</div>';

    echo '</div>';
}

/**
 * Render the SEO-analysis panel in the Edit screen.
 */
public function render_seo_analysis() {
	echo '<div class="dpg-seo-analysis">';

		/* ── header with round badge ────────────────────────── */
		echo '<div class="dpg-seo-header">';
			echo '<h3>' . esc_html__( 'SEO Analysis', 'dpg' ) . '</h3>';
			echo '<div class="dpg-seo-score">';
				echo '<div class="dpg-seo-score-circle">';
					echo '<div id="seo-score-number" class="dpg-seo-score-number">0</div>';
					echo '<div class="dpg-seo-score-label">' . esc_html__( 'SEO Score', 'dpg' ) . '</div>';
				echo '</div>';
			echo '</div>';
		echo '</div>';

		/* ── results: Problems / Improvements / Good results ─ */
		echo '<div class="dpg-seo-results">';

			echo '<div id="seo-problems" class="dpg-seo-section dpg-seo-problems" style="display:none;">';
				echo '<h4 class="dpg-seo-section-title">🔴 ' . esc_html__( 'Problems', 'dpg' ) . '</h4>';
				echo '<div id="seo-problems-list" class="dpg-seo-items"></div>';
			echo '</div>';

			echo '<div class="dpg-seo-two-column">';

				echo '<div class="dpg-seo-column">';
					echo '<div id="seo-improvements" class="dpg-seo-section dpg-seo-improvements" style="display:none;">';
						echo '<h4 class="dpg-seo-section-title">🟡 ' . esc_html__( 'Improvements', 'dpg' ) . '</h4>';
						echo '<div id="seo-improvements-list" class="dpg-seo-items"></div>';
					echo '</div>';
				echo '</div>';

				echo '<div class="dpg-seo-column">';
					echo '<div id="seo-good" class="dpg-seo-section dpg-seo-good" style="display:none;">';
						echo '<h4 class="dpg-seo-section-title">🟢 ' . esc_html__( 'Good Results', 'dpg' ) . '</h4>';
						echo '<div id="seo-good-list" class="dpg-seo-items"></div>';
					echo '</div>';
				echo '</div>';

			echo '</div>'; // .dpg-seo-two-column

			echo '<div id="seo-help" class="dpg-seo-help">';
				echo '<p>' . esc_html__( 'Add content to see SEO analysis.', 'dpg' ) . '</p>';
			echo '</div>';

		echo '</div>'; // .dpg-seo-results

		/* ── NEW: maintain score for PHP/DB layer ───────────── */
		echo '<input type="hidden" id="dpg_seo_score" name="dpg_seo_score" value="0">';

	echo '</div>'; // .dpg-seo-analysis
}


    /**
     * Render items fields - Full width section
     */
    public function render_items_fields() {
        $items = $this->template_data['items'] ?? [];
        $items_text = '';
        foreach ( $items as $item ) {
            if ( is_array( $item ) ) {
                $items_text .= sanitize_text_field( $item['name'] );
                if ( ! empty( $item['score'] ) ) {
                    $items_text .= '|' . floatval( $item['score'] );
                }
                $items_text .= "\n";
            } else {
                $items_text .= sanitize_text_field( $item ) . "\n";
            }
        }
        
        $this->render_field( [
            'type'        => 'textarea',
            'id'          => 'dpg_items',
            'name'        => 'dpg_items',
            'label'       => __( 'Items (one per line)', 'dpg' ),
            'required'    => true,
            'value'       => trim( $items_text ),
            'placeholder' => __( "New York City\nLos Angeles\nChicago\nHouston", 'dpg' ),
            'description' => __( 'Each line generates a page; add scores via "|".', 'dpg' ),
            'rows'        => 8,
        ] );
    }

    /**
     * Render social media fields - Same as Create Form (three columns in rows)
     */
    public function render_social_fields() {
        $hard_meta = $this->template_data['hard_meta'] ?? [];
        
        // First row: Social Title, Social Image URL, Social Description
        echo '<div class="dpg-field-grid dpg-field-grid--three-col">';
            echo '<div>';
                $this->render_field( [
                    'type'        => 'text',
                    'id'          => 'dpg_og_title',
                    'name'        => 'dpg_og_title',
                    'label'       => __( 'Social Title', 'dpg' ),
                    'value'       => $hard_meta['og_title'] ?? '',
                    'description' => __( 'Use {item} placeholder.', 'dpg' ),
                ] );
            echo '</div>';
            
            echo '<div>';
                $this->render_field( [
                    'type'        => 'url',
                    'id'          => 'dpg_og_image',
                    'name'        => 'dpg_og_image',
                    'label'       => __( 'Social Image URL', 'dpg' ),
                    'value'       => $hard_meta['og_image'] ?? '',
                    'description' => __( '1200×630px recommended.', 'dpg' ),
                ] );
            echo '</div>';
            
            echo '<div>';
                $this->render_field( [
                    'type'        => 'textarea',
                    'id'          => 'dpg_og_description',
                    'name'        => 'dpg_og_description',
                    'label'       => __( 'Social Description', 'dpg' ),
                    'value'       => $hard_meta['og_description'] ?? '',
                    'rows'        => 3,
                    'description' => __( 'Use {item} placeholder.', 'dpg' ),
                ] );
            echo '</div>';
        echo '</div>';

        // Second row: Twitter fields
        echo '<div class="dpg-field-grid dpg-field-grid--three-col">';
            echo '<div>';
                $this->render_field( [
                    'type'        => 'text',
                    'id'          => 'dpg_tw_title',
                    'name'        => 'dpg_tw_title',
                    'label'       => __( 'Twitter Title', 'dpg' ),
                    'value'       => $hard_meta['twitter_title'] ?? '',
                    'description' => __( 'Optional; falls back to Social Title.', 'dpg' ),
                ] );
            echo '</div>';
            
            echo '<div>';
                $this->render_field( [
                    'type'        => 'url',
                    'id'          => 'dpg_tw_image',
                    'name'        => 'dpg_tw_image',
                    'label'       => __( 'Twitter Image URL', 'dpg' ),
                    'value'       => $hard_meta['twitter_image'] ?? '',
                    'description' => __( 'Optional; falls back to Social Image.', 'dpg' ),
                ] );
            echo '</div>';
            
            echo '<div>';
                $this->render_field( [
                    'type'        => 'textarea',
                    'id'          => 'dpg_tw_description',
                    'name'        => 'dpg_tw_description',
                    'label'       => __( 'Twitter Description', 'dpg' ),
                    'value'       => $hard_meta['twitter_description'] ?? '',
                    'rows'        => 2,
                    'description' => __( 'Optional; falls back to Social Description.', 'dpg' ),
                ] );
            echo '</div>';
        echo '</div>';
    }

    /**
     * Render template options - Same as Create Form
     */
    public function render_options_fields() {
        echo '<div class="dpg-field-grid dpg-field-grid--two-col">';
            // Header
            echo '<div>';
                echo '<label class="dpg-checkbox-label">';
                    $header_checked = ! empty( $this->template_data['include_header'] ) ? ' checked' : '';
                    echo '<input type="checkbox" id="dpg_include_header" name="dpg_include_header" value="1" class="dpg-checkbox"' . $header_checked . '>';
                    echo '<span class="dpg-checkbox-text">' . esc_html__( 'Include Website Header', 'dpg' ) . '</span>';
                echo '</label>';
                echo '<p class="dpg-field-description">' . esc_html__( 'Include theme header on generated pages.', 'dpg' ) . '</p>';
            echo '</div>';
            // Footer
            echo '<div>';
                echo '<label class="dpg-checkbox-label">';
                    $footer_checked = ! empty( $this->template_data['include_footer'] ) ? ' checked' : '';
                    echo '<input type="checkbox" id="dpg_include_footer" name="dpg_include_footer" value="1" class="dpg-checkbox"' . $footer_checked . '>';
                    echo '<span class="dpg-checkbox-text">' . esc_html__( 'Include Website Footer', 'dpg' ) . '</span>';
                echo '</label>';
                echo '<p class="dpg-field-description">' . esc_html__( 'Include theme footer on generated pages.', 'dpg' ) . '</p>';
            echo '</div>';
        echo '</div>';
    }

/**
 * Save changes coming from the “Edit Template” form.
 * – Validates nonce / capability / timestamp
 * – Updates the per-context items table
 * – Persists the latest SEO score in the DB and the JSON file
 * – Flushes rewrite rules & redirects
 *
 * @param array $data Sanitised POST array created in handle_submission()
 */
/**
 * Save changes coming from the "Edit Template" form.
 * – Validates nonce / capability / timestamp
 * – Updates the per-context items table
 * – Persists the latest SEO score in the DB and the JSON file
 * – Flushes rewrite rules & redirects
 *
 * @param array $data Sanitised POST array created in handle_submission()
 */
public function process( $data = null ) {

    // ───────────────────────────────────────────────────────────────────────────
    //  Normalize incoming data
    // ───────────────────────────────────────────────────────────────────────────
    if ( ! is_array( $data ) ) {
        $data = $_POST;
    }

    /* ──────────────────────────────
     * 0. Enhanced Security / anti-CSRF
     * ──────────────────────────── */
    
    // Verify nonce - multiple nonce support for compatibility
    $nonce_verified = false;
    if ( isset( $_POST['dpg_nonce'] ) && wp_verify_nonce( $_POST['dpg_nonce'], $this->action ) ) {
        $nonce_verified = true;
    } elseif ( isset( $_POST['_wpnonce'] ) && wp_verify_nonce( $_POST['_wpnonce'], 'dpg_form_nonce' ) ) {
        $nonce_verified = true;
    }
    
    if ( ! $nonce_verified ) {
        wp_die( 
            esc_html__( 'Security verification failed. Please refresh the page and try again.', 'dpg' ),
            esc_html__( 'Security Error', 'dpg' ),
            [ 'response' => 403 ]
        );
    }

    // Capability check
    if ( ! current_user_can( 'manage_options' ) ) {
        wp_die( 
            esc_html__( 'You do not have permission to perform this action.', 'dpg' ),
            esc_html__( 'Permission Denied', 'dpg' ),
            [ 'response' => 403 ]
        );
    }

    // Enhanced user and timestamp validation
    $submitted_user = absint( $data['dpg_user_id'] ?? 0 );
    $submitted_time = absint( $data['dpg_timestamp'] ?? 0 );
    $current_user   = get_current_user_id();
    $current_time   = time();
    
    if ( $submitted_user !== $current_user ) {
        wp_die( 
            esc_html__( 'Security check failed: User verification error.', 'dpg' ),
            esc_html__( 'Security Error', 'dpg' ),
            [ 'response' => 403 ]
        );
    }
    
    if ( $current_time - $submitted_time > 300 ) { // 5 minutes
        wp_die( 
            esc_html__( 'Security check failed: Form has expired. Please refresh and try again.', 'dpg' ),
            esc_html__( 'Form Expired', 'dpg' ),
            [ 'response' => 403 ]
        );
    }

    /* ──────────────────────────────
     * 1. Load the current JSON with enhanced error handling
     * ──────────────────────────── */
    global $wp_filesystem;
    
    // Initialize WP_Filesystem
    if ( empty( $wp_filesystem ) ) {
        require_once ABSPATH . 'wp-admin/includes/file.php';
        WP_Filesystem();
    }

    $json_file = trailingslashit( DPG_DATA_DIR ) . $this->ctx_id . '.json';
    
    if ( ! $wp_filesystem->exists( $json_file ) ) {
        wp_die( 
            sprintf(
                /* translators: %s: template file name */
                esc_html__( 'Template file not found: %s', 'dpg' ),
                esc_html( basename( $json_file ) )
            ),
            esc_html__( 'Template Not Found', 'dpg' ),
            [ 'response' => 404 ]
        );
    }
    
    $existing_json = $wp_filesystem->get_contents( $json_file );
    if ( false === $existing_json ) {
        wp_die( 
            esc_html__( 'Could not read template file. Please check file permissions.', 'dpg' ),
            esc_html__( 'File Read Error', 'dpg' ),
            [ 'response' => 500 ]
        );
    }
    
    $existing = json_decode( $existing_json, true );
    if ( ! is_array( $existing ) ) {
        wp_die( 
            esc_html__( 'Invalid template data format. The template file may be corrupted.', 'dpg' ),
            esc_html__( 'Invalid Template Data', 'dpg' ),
            [ 'response' => 500 ]
        );
    }

    /* ──────────────────────────────
     * 2. Validate incoming data
     * ──────────────────────────── */
    $this->errors = $this->validate_data( $data );
    if ( $this->errors ) {
        $this->render(); // re-show the form
        return;
    }

    /* ──────────────────────────────
     * 3. Sanitise & normalise fields
     * ──────────────────────────── */
    $template_source = sanitize_key( $data['dpg_template_source'] ?? 'custom' );
    $page_id         = absint( $data['dpg_source_page_id'] ?? 0 );

    /* HTML source with enhanced validation */
    if ( 'custom' === $template_source ) {
        $html = wp_kses_post( $data['dpg_template_html'] ?? '' );
        // Ensure {item} placeholder exists for custom templates
        if ( ! empty( $html ) && false === strpos( $html, '{item}' ) ) {
            $html = "<!-- Dynamic content: {item} -->\n" . $html;
        }
    } elseif ( $page_id ) {
        $post = get_post( $page_id );
        if ( is_object( $post ) && 'publish' === $post->post_status ) {
            $html = $post->post_content;
            // Ensure {item} placeholder exists for page templates
            if ( false === strpos( $html, '{item}' ) ) {
                $html = "<!-- Dynamic content: {item} -->\n" . $html;
            }
        } else {
            wp_die( 
                esc_html__( 'Selected page is not available or not published.', 'dpg' ),
                esc_html__( 'Invalid Page Selection', 'dpg' ),
                [ 'response' => 400 ]
            );
        }
    } else {
        $html = '';
    }

    /* Items array (keep optional sitemap-priority pipe) with enhanced sanitization */
    $items_raw = explode( "\n", wp_unslash( $data['dpg_items'] ?? '' ) );
    $items     = [];

    foreach ( $items_raw as $line ) {
        $line = trim( $line );
        if ( '' === $line ) {
            continue;
        }
        if ( false !== strpos( $line, '|' ) ) {
            $parts = array_map( 'trim', explode( '|', $line, 2 ) );
            if ( count( $parts ) >= 2 ) {
                list( $label, $prio ) = $parts;
                $items[] = [
                    'name'  => sanitize_text_field( $label ),
                    'score' => is_numeric( $prio )
                        ? $this->clamp( floatval( $prio ), 0.1, 1.0 )
                        : null,
                ];
            } else {
                $items[] = sanitize_text_field( $parts[0] );
            }
        } else {
            $items[] = sanitize_text_field( $line );
        }
    }

    /* ★ Enhanced SEO score validation */
    $seo_score = isset( $data['dpg_seo_score'] )
        ? max( 0, min( 100, intval( $data['dpg_seo_score'] ) ) )
        : 0;

    /* ──────────────────────────────
     * 4. Persist to the database with error handling
     * ──────────────────────────── */
    if ( class_exists( 'DPG_DB' ) ) {
        try {
            /* 4-a find / create numeric context-ID in option dpg_contexts */
            $contexts = get_option( 'dpg_contexts', [] );
            $row      = wp_list_filter( $contexts, [ 'ctx_id' => $this->ctx_id ] );

            if ( $row ) {
                $numeric_id = absint( array_shift( $row )['id'] );
            } else {
                $numeric_id = empty( $contexts )
                    ? 1
                    : ( max( array_column( $contexts, 'id' ) ) + 1 );

                $contexts[] = [
                    'id'      => $numeric_id,
                    'ctx_id'  => $this->ctx_id,
                    'name'    => sanitize_text_field( $existing['name'] ?? $this->ctx_id ),
                    'created' => current_time( 'mysql' ),
                ];
                update_option( 'dpg_contexts', $contexts, false );
            }

            /* 4-b store items + score with error handling */
            $type_key   = ( substr( $this->ctx_id, -5 ) === '-area' ) ? 'area' : 'keyword';
            $plain_list = array_map(
                static function ( $it ) { 
                    return is_array( $it ) ? sanitize_text_field( $it['name'] ) : sanitize_text_field( $it ); 
                },
                $items
            );

            DPG_DB::create_table( $type_key, $numeric_id );
            DPG_DB::save_items( $type_key, $numeric_id, $plain_list );
            DPG_DB::update_template_column( $numeric_id, 'seo_score', $seo_score );

            /* ★ ALSO mirror into the global score table */
            if ( method_exists( 'DPG_DB', 'save_score' ) ) {
                DPG_DB::save_score( $this->ctx_id, $seo_score );
            }
        } catch ( Exception $e ) {
            // Log error but don't stop the process
            if ( defined( 'WP_DEBUG' ) && WP_DEBUG && defined( 'WP_DEBUG_LOG' ) && WP_DEBUG_LOG ) {
                error_log( '[DPG] Database error in process(): ' . $e->getMessage() );
            }
        }
    }

    /* ──────────────────────────────
     * 5. Merge & rewrite JSON file with enhanced validation
     * ──────────────────────────── */
    $json_data = array_merge( $existing, [
        'template_source' => $template_source,
        'source_page_id'  => $page_id,
        'template_html'   => $html,
        'items'           => array_values( $items ),
        'last_modified'   => current_time( 'mysql' ),
        'seo_score'       => $seo_score,
        'focus_keyphrase' => sanitize_text_field( $data['dpg_focus_keyphrase'] ?? '' ),
        'hard_meta'       => [
            'title'               => sanitize_text_field( $data['dpg_meta_title'] ?? '' ),
            'description'         => sanitize_textarea_field( $data['dpg_meta_description'] ?? '' ),
            'keywords'            => sanitize_textarea_field( $data['dpg_meta_keywords'] ?? '' ),
            'og_title'            => sanitize_text_field( $data['dpg_og_title'] ?? '' ),
            'og_description'      => sanitize_textarea_field( $data['dpg_og_description'] ?? '' ),
            'og_image'            => esc_url_raw( $data['dpg_og_image'] ?? '' ),
            'twitter_title'       => sanitize_text_field( $data['dpg_tw_title'] ?? '' ),
            'twitter_description' => sanitize_textarea_field( $data['dpg_tw_description'] ?? '' ),
            'twitter_image'       => esc_url_raw( $data['dpg_tw_image'] ?? '' ),
        ],
        'include_header'  => ! empty( $data['dpg_include_header'] ),
        'include_footer'  => ! empty( $data['dpg_include_footer'] ),
    ] );

    /* Optional sitemap-priority override with validation */
    if ( isset( $data['dpg_score'] ) && is_numeric( $data['dpg_score'] ) ) {
        $score = floatval( $data['dpg_score'] );
        if ( $score >= 0.1 && $score <= 1.0 ) {
            $json_data['score'] = $this->clamp( $score, 0.1, 1.0 );
        }
    }

    /* Write JSON file with enhanced error handling */
    $json_content = wp_json_encode( $json_data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE );
    if ( false === $json_content ) {
        wp_die( 
            esc_html__( 'Could not encode template data. Please check your input for invalid characters.', 'dpg' ),
            esc_html__( 'Data Encoding Error', 'dpg' ),
            [ 'response' => 500 ]
        );
    }

    $ok = $wp_filesystem->put_contents( $json_file, $json_content, FS_CHMOD_FILE );

    if ( false === $ok ) {
        wp_die( 
            esc_html__( 'Could not update template file. Please check file permissions and disk space.', 'dpg' ),
            esc_html__( 'File Write Error', 'dpg' ),
            [ 'response' => 500 ]
        );
    }

    /* ──────────────────────────────
     * 6. Finish – flush & redirect with success message
     * ──────────────────────────── */
    flush_rewrite_rules();
    
    // Enhanced redirect with success message
    $redirect_url = add_query_arg( 
        [
            'updated' => '1',
            'template' => urlencode( $this->ctx_id )
        ], 
        admin_url( 'admin.php?page=dpg-dashboard' ) 
    );
    
    wp_safe_redirect( $redirect_url );
    exit;
}


/**
 * Map POST → $data and hand over to process()
 */
public function handle_submission() {
    // Basic security checks only
    if ( ! current_user_can( 'manage_options' ) ) {
        wp_die( esc_html__( 'Permission denied', 'dpg' ) );
    }
    
    // Check nonce
    $nonce_ok = false;
    if ( isset( $_POST['dpg_nonce'] ) && wp_verify_nonce( $_POST['dpg_nonce'], 'dpg_update_template' ) ) {
        $nonce_ok = true;
    }
    if ( isset( $_POST['_wpnonce'] ) && wp_verify_nonce( $_POST['_wpnonce'], 'dpg_form_nonce' ) ) {
        $nonce_ok = true;
    }
    
    if ( ! $nonce_ok ) {
        wp_die( esc_html__( 'Security check failed.', 'dpg' ), 403 );
    }

    /* ──────────────────────────────
     * 1. Debug output (remove after testing)
     * ──────────────────────────── */
    if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
        error_log( 'DPG Edit Debug:' );
        error_log( 'Template source: ' . ( $_POST['dpg_template_source'] ?? 'EMPTY' ) );
        error_log( 'Source page ID: ' . ( $_POST['dpg_source_page_id'] ?? 'EMPTY' ) );
        error_log( 'Template HTML length: ' . strlen( $_POST['dpg_template_html'] ?? '' ) );
        error_log( 'Template HTML preview: ' . substr( $_POST['dpg_template_html'] ?? '', 0, 200 ) );
    }

    // Get form data
    $data = $this->get_form_data( [
        'dpg_ctx_id'          => '',
        'dpg_user_id'         => 0,
        'dpg_timestamp'       => 0,
        'dpg_template_source' => 'custom',
        'dpg_source_page_id'  => '',
        'dpg_template_html'   => '',
        'dpg_items'           => '',
        'dpg_score'           => '',
        'dpg_seo_score'       => 0,
        'dpg_focus_keyphrase' => '',
        'dpg_meta_title'      => '',
        'dpg_meta_description'=> '',
        'dpg_meta_keywords'   => '',
        'dpg_og_title'        => '',
        'dpg_og_description'  => '',
        'dpg_og_image'        => '',
        'dpg_tw_title'        => '',
        'dpg_tw_description'  => '',
        'dpg_tw_image'        => '',
        'dpg_include_header'  => '',
        'dpg_include_footer'  => '',
    ] );

    // FIXED: Validate form data before processing
    $this->errors = $this->validate_data( $data );
    if ( ! empty( $this->errors ) ) {
        return $this->render();
    }

    // Process the form
    return $this->process( $data );
}


    /**
     * Validate form data
     */
public function validate_data( $data ) {
    $errors = [];

    // ENHANCED: Use base class template validation for both custom HTML and page-based templates
    $template_errors = $this->validate_template_content( $data );
    if ( ! empty( $template_errors ) ) {
        $errors = array_merge( $errors, $template_errors );
    }

    // Items validation
    if ( empty( $data['dpg_items'] ) ) {
        $errors[] = __( 'Items list is required.', 'dpg' );
    } else {
        $items = array_filter( array_map( 'trim', explode( "\n", $data['dpg_items'] ) ) );
        if ( empty( $items ) ) {
            $errors[] = __( 'At least one item is required.', 'dpg' );
        }
        
        // Free version items limit (only for non-premium users)
        if ( count( $items ) > 20 ) {
            $errors[] = sprintf(
                __( 'Free version only allows up to 20 items. You have %d items. Please reduce the number.', 'dpg' ),
                count( $items )
            );
        }
    }

    // Score validation
    if ( ! empty( $data['dpg_score'] ) ) {
        $s = floatval( $data['dpg_score'] );
        if ( $s < 0.1 || $s > 1.0 ) {
            $errors[] = __( 'Page score must be between 0.1 and 1.0.', 'dpg' );
        }
    }

    // SEO Score validation
    if ( isset( $data['dpg_seo_score'] ) ) {
        $seo_score = intval( $data['dpg_seo_score'] );
        if ( $seo_score < 0 || $seo_score > 100 ) {
            $errors[] = __( 'SEO score must be between 0 and 100.', 'dpg' );
        }
    }

    // Validate URLs if provided
    if ( ! empty( $data['dpg_og_image'] ) && ! filter_var( $data['dpg_og_image'], FILTER_VALIDATE_URL ) ) {
        $errors[] = __( 'Social image URL is not valid.', 'dpg' );
    }
    if ( ! empty( $data['dpg_tw_image'] ) && ! filter_var( $data['dpg_tw_image'], FILTER_VALIDATE_URL ) ) {
        $errors[] = __( 'Twitter image URL is not valid.', 'dpg' );
    }

    return $errors;
}

public function debug_nonce_info() {
    if ( defined( 'WP_DEBUG' ) && WP_DEBUG && current_user_can( 'manage_options' ) ) {
        $nonce = wp_create_nonce( 'dpg_form_nonce' );
        echo "<!-- DEBUG EDIT FORM NONCE INFO: \n";
        echo "Generated nonce: $nonce\n";
        echo "User ID: " . get_current_user_id() . "\n";
        echo "Action: dpg_form_nonce\n";
        echo "Verification test: " . ( wp_verify_nonce( $nonce, 'dpg_form_nonce' ) ? 'PASS' : 'FAIL' ) . "\n";
        echo "Template: Edit Form\n";
        echo "Context ID: " . esc_html( $this->ctx_id ) . "\n";
        echo "-->";
    }
}

    /**
     * Helper: clamp a number between min and max
     */
    private function clamp( $num, $min, $max ) {
        return $num < $min ? $min : ( $num > $max ? $max : $num );
    }
}