<?php
/**
 * Plugin Name:  Dynamic SEO Pages
 * Plugin URI:   https://wordpress.org/plugins/dynamic-seo-pages
 * Description:  Generates SEO-optimised keyword-targeted landing pages from one template using a list of service areas or keywords – auto-creates friendly URLs, meta tags, Open Graph/Twitter cards and an XML sitemap.
 * Version:      1.3.7
 * Author:       Dynamic SEO Pro Agency
 * Author URI:   https://dynamicseopro.com
 * Text Domain:  dynamic-seo-pages
 * Requires at least: 5.0
 * Tested up to:      6.6
 * Requires PHP:      7.4
 * License:      GPL v2 or later
 * License URI:  https://www.gnu.org/licenses/gpl-2.0.html
 */

/* ────────────────────────────────────────────────────────────────────
 *  SAFETY
 * ────────────────────────────────────────────────────────────────── */
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/* ────────────────────────────────────────────────────────────────────
 *  CORE CONSTANTS  (declare **before** any class files are loaded)
 * ────────────────────────────────────────────────────────────────── */
define( 'DPG_PLUGIN_FILE', __FILE__ );
define( 'DPG_VERSION',      '1.3.8' );
define( 'DPG_PLUGIN_DIR',   plugin_dir_path( __FILE__ ) );
define( 'DPG_PLUGIN_URL',   plugin_dir_url(  __FILE__ ) );

/* ──── WordPress-Compliant Directory Structure ──────────────────── */
if ( ! defined( 'DPG_DATA_DIR' ) ) {
    $upload_dir = wp_upload_dir();
    define( 'DPG_DATA_DIR', $upload_dir['basedir'] . '/dpg-data/' );
}
if ( ! defined( 'DPG_TMPL_DIR' ) ) {
    $upload_dir = wp_upload_dir();
    define( 'DPG_TMPL_DIR', $upload_dir['basedir'] . '/dpg-templates/' );
}
if ( ! defined( 'DPG_CACHE_DIR' ) ) {
    $upload_dir = wp_upload_dir();
    define( 'DPG_CACHE_DIR', $upload_dir['basedir'] . '/dpg-cache/' );
}

define( 'DPG_TEXT_DOMAIN', 'dynamic-seo-pages' );
define( 'DPG_REQUIRED_CAPABILITY', 'manage_options' );
define( 'DPG_MIN_PHP_VERSION', '7.4' );
define( 'DPG_MIN_WP_VERSION',  '5.0' );

/* ────────────────────────────────────────────────────────────────────
 *  INCLUDE EARLY-LOADED CLASSES
 * ────────────────────────────────────────────────────────────────── */
require_once DPG_PLUGIN_DIR . 'includes/class-dpg-db.php';
require_once DPG_PLUGIN_DIR . 'includes/forms/class-dpg-form-settings.php';
require_once DPG_PLUGIN_DIR . 'includes/class-dpg-frontend.php';

// Bridge between the old global hook and the class method
if ( ! function_exists( 'dpg_add_rewrite_rules' ) ) {
    function dpg_add_rewrite_rules() {
        if ( class_exists( 'DPG_Frontend' ) && method_exists( 'DPG_Frontend', 'add_rewrite_rules' ) ) {
            DPG_Frontend::add_rewrite_rules();
        }
    }
}

// Now hook it the way WordPress expects
add_action( 'init', 'dpg_add_rewrite_rules', 1 );



/* ────────────────────────────────────────────────────────────────────
 *  WORDPRESS-COMPLIANT DIRECTORY MIGRATION
 * ────────────────────────────────────────────────────────────────── */
function dpg_migrate_to_compliant_directories() {
    // Old legacy locations (to migrate from)
    $legacy_locations = [
        'data' => WP_CONTENT_DIR . '/dpg-data/',
        'templates' => WP_CONTENT_DIR . '/dpg-templates/',
        'cache' => WP_CONTENT_DIR . '/dpg-cache/',
        'plugin_data' => plugin_dir_path( __FILE__ ) . 'data/', // Very old location
    ];
    
    // New WordPress-compliant locations
    $upload_dir = wp_upload_dir();
    $new_data_dir = $upload_dir['basedir'] . '/dpg-data/';
    
    global $wp_filesystem;
    if ( ! function_exists( 'WP_Filesystem' ) ) {
        require_once ABSPATH . 'wp-admin/includes/file.php';
    }
    
    if ( ! WP_Filesystem() ) {
        return false;
    }
    
    $migrated = false;
    
    foreach ( $legacy_locations as $type => $legacy_dir ) {
        // Only migrate if legacy exists and new doesn't
        if ( $wp_filesystem->exists( $legacy_dir ) && ! $wp_filesystem->exists( $new_data_dir ) ) {
            // Create new directory
            wp_mkdir_p( $new_data_dir );
            
            // Move JSON files from legacy to new location
            $files = glob( $legacy_dir . '*.json' );
            if ( $files ) {
                foreach ( $files as $file ) {
                    $filename = basename( $file );
                    $wp_filesystem->move( $file, $new_data_dir . $filename );
                }
            }
            
            // Remove empty legacy directory
            $wp_filesystem->rmdir( $legacy_dir );
            $migrated = true;
            
            if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                error_log( "DPG: Migrated {$type} from {$legacy_dir} to {$new_data_dir}" );
            }
        }
    }
    
    return $migrated;
}
add_action( 'plugins_loaded', 'dpg_migrate_to_compliant_directories', 1 );

// Check if pro version is active
add_action( 'admin_init', 'dpg_free_check_pro_version_conflict' );
function dpg_free_check_pro_version_conflict() {
    // Check if pro version is active
    if ( is_plugin_active( 'dynamic-seo-pro/dynamic-seo-pro.php' ) ) {
        // Deactivate this free version to prevent conflicts
        deactivate_plugins( plugin_basename( __FILE__ ) );
        
        // Add admin notice
        add_action( 'admin_notices', 'dpg_free_conflict_notice' );
        
        // Prevent further execution
        return;
    }
}

function dpg_free_conflict_notice() {
    ?>
    <div class="notice notice-success is-dismissible">
        <h3><?php esc_html_e( 'Upgraded to Pro Version!', 'dynamic-seo-pages' ); ?></h3>
        <p><strong><?php esc_html_e( 'The free version has been automatically deactivated because the Pro version is now active.', 'dynamic-seo-pages' ); ?></strong></p>
        <p><?php esc_html_e( 'Your Pro version includes all features plus:', 'dynamic-seo-pages' ); ?></p>
        <ul style="list-style: disc; margin-left: 20px;">
            <li><?php esc_html_e( 'Unlimited templates and keywords', 'dynamic-seo-pages' ); ?></li>
            <li><?php esc_html_e( 'Advanced SEO features', 'dynamic-seo-pages' ); ?></li>
            <li><?php esc_html_e( 'Priority support', 'dynamic-seo-pages' ); ?></li>
            <li><?php esc_html_e( 'Regular updates', 'dynamic-seo-pages' ); ?></li>
        </ul>
        
        <p>
            <a href="<?php echo esc_url( admin_url( 'admin.php?page=dpg-dashboard' ) ); ?>" class="button button-primary">
                <?php esc_html_e( 'Go to Pro Dashboard', 'dynamic-seo-pages' ); ?>
            </a>
            
            <a href="<?php echo esc_url( admin_url( 'plugins.php' ) ); ?>" class="button button-secondary">
                <?php esc_html_e( 'Manage Plugins', 'dynamic-seo-pages' ); ?>
            </a>
        </p>
    </div>
    <?php
}

// If pro version is active, stop loading this plugin
if ( is_plugin_active( 'dynamic-seo-pro/dynamic-seo-pro.php' ) ) {
    return;
}

/* ────────────────────────────────────────────────────────────────────
 *  EXPORT FUNCTIONALITY FOR MIGRATION TO PRO
 * ──────────────────────────────────────────────────────────────────── */

/**
 * Only register our export AJAX endpoint when we're on
 * admin.php?page=dpg-settings
 */
add_action( 'load-admin_page_dpg-settings', function() {
    add_action( 'wp_ajax_dpg_export_for_migration', 'dpg_export_templates_for_migration' );
} );

/**
 * AJAX handler to export templates & settings for migration
 */
function dpg_export_templates_for_migration() {
    // Security checks
    if ( ! current_user_can( 'manage_options' ) ) {
        wp_die( 'Permission denied', 403 );
    }
    if ( ! check_ajax_referer( 'dpg_export_migration', 'nonce', false ) ) {
        wp_die( 'Security check failed', 403 );
    }

    // Get all template data
    $templates  = [];
    $json_files = glob( DPG_DATA_DIR . '*.json' );

    if ( $json_files ) {
        foreach ( $json_files as $file ) {
            $ctx_id = basename( $file, '.json' );
            if ( dpg_is_master_template( $ctx_id ) ) {
                continue;
            }
            $content = file_get_contents( $file );
            if ( $content ) {
                $data = json_decode( $content, true );
                if ( $data ) {
                    $templates[] = [
                        'ctx_id' => $ctx_id,
                        'data'   => $data,
                    ];
                }
            }
        }
    }

    // Get plugin settings
    $settings = [
        'dpg_pages_active'   => get_option( 'dpg_pages_active',   true  ),
        'dpg_debug_mode'     => get_option( 'dpg_debug_mode',     false ),
        'dpg_auto_meta'      => get_option( 'dpg_auto_meta',      true  ),
        'dpg_sitemap_enabled'=> get_option( 'dpg_sitemap_enabled', true  ),
        'dpg_minify_html'    => get_option( 'dpg_minify_html',    false ),
        'dpg_compress_output'=> get_option( 'dpg_compress_output',false ),
        'dpg_custom_scripts' => get_option( 'dpg_custom_scripts', ''    ),
    ];

    // Build export payload
    $export_data = [
        'version'        => DPG_VERSION,
        'export_date'    => current_time( 'Y-m-d H:i:s' ),
        'site_url'       => get_site_url(),
        'templates'      => $templates,
        'settings'       => $settings,
        'template_count' => count( $templates ),
        'total_pages'    => dpg_get_generated_pages_count(),
    ];

    // Send download response
    nocache_headers();
    header( 'Content-Type: application/json; charset=utf-8' );
    header( 'Content-Disposition: attachment; filename="dpg-migration-export-' . date( 'Y-m-d' ) . '.json"' );

    echo wp_json_encode( $export_data, JSON_PRETTY_PRINT );
    exit;
}





/**
 * Enqueue admin assets properly - CORRECTED VERSION
 */
add_action( 'admin_enqueue_scripts', 'dpg_enqueue_admin_assets' );
function dpg_enqueue_admin_assets( $hook ) {
    // ✅ Only load on DPG pages
    if ( strpos( $hook, 'dpg' ) === false && $hook !== 'toplevel_page_dpg-dashboard' ) {
        return;
    }
    
    wp_enqueue_style(
        'dpg-admin-style',
        DPG_PLUGIN_URL . 'includes/forms/assets/css/admin-forms.css',
        [],
        DPG_VERSION
    );
    
    wp_enqueue_script(
        'dpg-admin-script',
        DPG_PLUGIN_URL . 'includes/forms/assets/js/admin.js',
        [ 'jquery' ],
        DPG_VERSION,
        true
    );
    
    wp_localize_script( 'dpg-admin-script', 'dpgMenuData', [
        'urls' => [
            'dashboard' => admin_url( 'admin.php?page=dpg-dashboard' ),
            'create'    => admin_url( 'admin.php?page=dpg-create' ),
            'settings'  => admin_url( 'admin.php?page=dpg-settings' ),
        ],
        'strings' => [
            'dashboard'      => __( 'Dashboard', 'dynamic-seo-pages'),
            'create_template' => __( 'Create Template', 'dynamic-seo-pages'),
            'settings'       => __( 'Settings', 'dynamic-seo-pages'),
        ],
        'nonce' => wp_create_nonce( 'dpg_export' ),
        'cache_nonce' => wp_create_nonce( 'dpg_clear_cache' ),
        'ajaxurl' => admin_url( 'admin-ajax.php' ),
        'debug' => defined( 'WP_DEBUG' ) && WP_DEBUG,
        'is_dpg_page' => true
    ]);
}

/**
 * WordPress-compliant hover menu function
 */
add_action( 'admin_footer', 'dpg_add_sidebar_hover_menu_minimal' );
function dpg_add_sidebar_hover_menu_minimal() {
    // ✅ Only show on DPG pages
    $screen = get_current_screen();
    if ( ! $screen || strpos( $screen->id, 'dpg' ) === false ) {
        return;
    }
    
    $menu_items = array(
        'dashboard' => array(
            'url' => admin_url( 'admin.php?page=dpg-dashboard' ),
            'label' => __( 'Dashboard', 'dynamic-seo-pages')
        ),
        'create' => array(
            'url' => admin_url( 'admin.php?page=dpg-create' ),
            'label' => __( 'Create Template', 'dynamic-seo-pages')
        ),
        'settings' => array(
            'url' => admin_url( 'admin.php?page=dpg-settings' ),
            'label' => __( 'Settings', 'dynamic-seo-pages')
        )
    );
    ?>
    <nav id="dpg-hover-menu" class="dpg-hover-menu">
      <ul>
        <?php foreach ( $menu_items as $key => $item ): ?>
          <li>
            <a href="<?php echo esc_url( $item['url'] ); ?>">
              <?php echo esc_html( $item['label'] ); ?>
            </a>
          </li>
        <?php endforeach; ?>
      </ul>
    </nav>
    <?php
}

/**
 * Check system requirements
 */
function dpg_check_requirements() {
    $errors = [];

    // Check PHP version
    if ( version_compare( PHP_VERSION, DPG_MIN_PHP_VERSION, '<' ) ) {
        $errors[] = sprintf(
            __( 'Dynamic SEO Pages requires PHP %s or higher. You are running PHP %s.', 'dynamic-seo-pages'),
            DPG_MIN_PHP_VERSION,
            PHP_VERSION
        );
    }

    // Check WordPress version
    global $wp_version;
    if ( version_compare( $wp_version, DPG_MIN_WP_VERSION, '<' ) ) {
        $errors[] = sprintf(
            __( 'Dynamic SEO Pages requires WordPress %s or higher. You are running WordPress %s.', 'dynamic-seo-pages'),
            DPG_MIN_WP_VERSION,
            $wp_version
        );
    }

    // Check required functions
    $required_functions = [ 'json_decode', 'json_encode', 'curl_init', 'file_get_contents' ];
    foreach ( $required_functions as $function ) {
        if ( ! function_exists( $function ) ) {
            $errors[] = sprintf(
                __( 'Dynamic SEO Pages requires the PHP function %s which is not available on your server.', 'dynamic-seo-pages'),
                $function
            );
        }
    }

    // Check file permissions for uploads directory (WordPress-compliant)
    $upload_dir = wp_upload_dir();
    if ( ! wp_is_writable( $upload_dir['basedir'] ) ) {
        $errors[] = __( 'Dynamic SEO Pages requires write permissions to the WordPress uploads directory.', 'dynamic-seo-pages');
    }

    return $errors;
}

/**
 * WordPress-compliant admin notice system
 */
add_action( 'admin_notices', 'dpg_display_admin_notices' );
function dpg_display_admin_notices() {
    // Only show on relevant pages to avoid overwhelming dashboard
    $screen = get_current_screen();
    if ( ! $screen || ( strpos( $screen->id, 'dpg' ) === false && $screen->id !== 'dashboard' && $screen->id !== 'plugins' ) ) {
        return;
    }
    
    // Show activation notice (one-time only)
    dpg_show_activation_notice();
    
    // Show requirement issues (critical only)
    dpg_show_requirement_notices();
    
    // Show template source notice (contextual only)
    dpg_show_template_source_notice();
    
    // Show admin messages from form submissions
    dpg_display_admin_messages();
}

/**
 * WordPress-compliant activation notice
 */
function dpg_show_activation_notice() {
    if ( ! get_transient( 'dpg_activation_notice' ) ) {
        return;
    }
    
    // Only show once, then auto-dismiss
    delete_transient( 'dpg_activation_notice' );
    
    echo '<div class="notice notice-success is-dismissible">';
    echo '<p><strong>' . esc_html__( 'Dynamic SEO Pages activated!', 'dynamic-seo-pages') . '</strong> ';
    echo sprintf(
        esc_html__( 'Get started by %s.', 'dynamic-seo-pages'),
        '<a href="' . esc_url( admin_url( 'admin.php?page=dpg-dashboard' ) ) . '">' . 
        esc_html__( 'creating your first template', 'dynamic-seo-pages') . '</a>'
    );
    echo '</p>';
    echo '</div>';
}

/**
 * WordPress-compliant requirement notices (critical issues only)
 */
function dpg_show_requirement_notices() {
    // Only show if user hasn't dismissed it recently
    $dismissed_until = get_user_meta( get_current_user_id(), 'dpg_requirements_dismissed', true );
    if ( $dismissed_until && $dismissed_until > time() ) {
        return;
    }
    
    $errors = dpg_check_requirements();
    if ( empty( $errors ) ) {
        return;
    }

    echo '<div class="notice notice-error is-dismissible" data-dismissible="dpg-requirements">';
    echo '<p><strong>' . esc_html__( 'Dynamic SEO Pages:', 'dynamic-seo-pages') . '</strong></p>';
    foreach ( $errors as $error ) {
        echo '<p>' . esc_html( $error ) . '</p>';
    }
    echo '<p>' . esc_html__( 'Please contact your hosting provider to resolve these issues.', 'dynamic-seo-pages') . '</p>';
    echo '</div>';
}

/**
 * WordPress-compliant template source notice (contextual only)
 */
function dpg_show_template_source_notice() {
    global $post, $pagenow;
    
    // Only show on edit-page screen
    if ( $pagenow !== 'post.php' || ! isset( $post ) || $post->post_type !== 'page' ) {
        return;
    }
    
    // Check if user already dismissed this for this page
    $dismissed_key = 'dpg_template_notice_' . $post->ID;
    if ( get_user_meta( get_current_user_id(), $dismissed_key, true ) ) {
        return;
    }
    
    $template_info = dpg_is_page_template_source( $post->ID, true );
    if ( empty( $template_info ) ) {
        return;
    }

    echo '<div class="notice notice-warning is-dismissible" data-dismissible="dpg-template-' . $post->ID . '">';
    echo '<p><strong>' . esc_html__( 'Dynamic SEO Notice:', 'dynamic-seo-pages') . '</strong> ';
    echo esc_html__( 'This page is being used as a template source for dynamic SEO pages.', 'dynamic-seo-pages') . '</p>';
    
    echo '<ul>';
    foreach ( array_slice( $template_info, 0, 2 ) as $info ) { // Limit to 2 for brevity
        echo '<li>';
        echo sprintf(
            esc_html__( 'Template: %1$s (%2$d pages)', 'dynamic-seo-pages'),
            '<strong>' . esc_html( $info['name'] ) . '</strong>',
            $info['items_count']
        );
        echo '</li>';
    }
    echo '</ul>';
    
    echo '<p>' . esc_html__( 'Changes you make here will affect all generated pages using this template.', 'dynamic-seo-pages') . '</p>';
    echo '</div>';
}

/**
 * WordPress-compliant success message system (for form submissions)
 */
function dpg_add_success_message( $message, $type = 'success' ) {
    $valid_types = [ 'success', 'error', 'warning', 'info' ];
    if ( ! in_array( $type, $valid_types, true ) ) {
        $type = 'success';
    }
    
    set_transient( 'dpg_admin_message', [
        'message' => $message,
        'type' => $type,
        'timestamp' => time()
    ], 30 ); // 30 seconds
}

/**
 * Display success/error messages from form submissions
 */
function dpg_display_admin_messages() {
    $message_data = get_transient( 'dpg_admin_message' );
    if ( ! $message_data ) {
        return;
    }
    
    // Auto-expire after 30 seconds
    if ( ( time() - $message_data['timestamp'] ) > 30 ) {
        delete_transient( 'dpg_admin_message' );
        return;
    }
    
    delete_transient( 'dpg_admin_message' );
    
    $class = 'notice notice-' . esc_attr( $message_data['type'] ) . ' is-dismissible';
    echo '<div class="' . $class . '">';
    echo '<p>' . wp_kses_post( $message_data['message'] ) . '</p>';
    echo '</div>';
}

/**
 * WordPress-compliant AJAX handler for dismissing notices
 */
add_action( 'wp_ajax_dpg_dismiss_notice', 'dpg_handle_dismiss_notice' );
function dpg_handle_dismiss_notice() {
    // Verify nonce and permissions
    if ( ! check_ajax_referer( 'dpg_dismiss_notice', 'nonce', false ) ) {
        wp_send_json_error( 'Invalid nonce' );
    }
    
    if ( ! current_user_can( 'manage_options' ) ) {
        wp_send_json_error( 'Insufficient permissions' );
    }
    
    $notice_id = sanitize_text_field( $_POST['notice_id'] ?? '' );
    if ( empty( $notice_id ) ) {
        wp_send_json_error( 'Invalid notice ID' );
    }
    
    $user_id = get_current_user_id();
    
    // Set dismissal based on notice type
    switch ( $notice_id ) {
        case 'dpg-requirements':
            // Dismiss requirements notice for 7 days
            update_user_meta( $user_id, 'dpg_requirements_dismissed', time() + WEEK_IN_SECONDS );
            break;
            
        default:
            // For template-specific notices
            if ( strpos( $notice_id, 'dpg-template-' ) === 0 ) {
                $page_id = str_replace( 'dpg-template-', '', $notice_id );
                update_user_meta( $user_id, 'dpg_template_notice_' . $page_id, true );
            }
            break;
    }
    
    wp_send_json_success( 'Notice dismissed' );
}

/**
 * Enqueue dismissible notice script (WordPress-compliant)
 */
add_action( 'admin_enqueue_scripts', 'dpg_enqueue_dismissible_notices_script' );
function dpg_enqueue_dismissible_notices_script( $hook ) {
    // Only load when we might have notices
    if ( ! current_user_can( 'manage_options' ) ) {
        return;
    }
    
    wp_enqueue_script( 'jquery' );
    
    // Inline script for dismissible notices (WordPress-compliant)
    $script = '
    jQuery(document).ready(function($) {
        $(document).on("click", ".notice[data-dismissible] .notice-dismiss", function() {
            var $notice = $(this).closest(".notice");
            var noticeId = $notice.data("dismissible");
            
            if (noticeId) {
                $.post(ajaxurl, {
                    action: "dpg_dismiss_notice",
                    notice_id: noticeId,
                    nonce: "' . wp_create_nonce( 'dpg_dismiss_notice' ) . '"
                });
            }
        });
    });';
    
    wp_add_inline_script( 'jquery', $script );
}

// Exit early if requirements not met
$requirement_errors = dpg_check_requirements();
if ( ! empty( $requirement_errors ) ) {
    return;
}

/**
 * WordPress-compliant directory creation with proper protection
 */
add_action( 'init', 'dpg_create_compliant_directories', 5 );
function dpg_create_compliant_directories() {
    $upload_dir = wp_upload_dir();
    
    $directories = [
        $upload_dir['basedir'] . '/dpg-data/' => __( 'data directory', 'dynamic-seo-pages'),
        $upload_dir['basedir'] . '/dpg-templates/' => __( 'templates directory', 'dynamic-seo-pages'),
        $upload_dir['basedir'] . '/dpg-cache/' => __( 'cache directory', 'dynamic-seo-pages'),
    ];

    $errors = [];
    
    foreach ( $directories as $dir => $description ) {
        if ( ! wp_mkdir_p( $dir ) ) {
            $errors[] = sprintf(
                __( 'Failed to create %s: %s', 'dynamic-seo-pages'),
                $description,
                $dir
            );
        } else {
            // Create index.php to prevent directory browsing (WordPress best practice)
            $index_content = "<?php\n// Silence is golden.\n";
            $index_file = $dir . 'index.php';
            
            if ( ! file_exists( $index_file ) ) {
                file_put_contents( $index_file, $index_content );
            }
            
            // Create .htaccess for additional protection (WordPress-compliant)
            $htaccess_content = "# WordPress-compliant directory protection\n";
            $htaccess_content .= "Options -Indexes\n";
            $htaccess_content .= "<Files *.json>\n";
            $htaccess_content .= "    Order allow,deny\n";
            $htaccess_content .= "    Deny from all\n";
            $htaccess_content .= "</Files>\n";
            
            $htaccess_file = $dir . '.htaccess';
            if ( ! file_exists( $htaccess_file ) ) {
                file_put_contents( $htaccess_file, $htaccess_content );
            }
        }
    }

    if ( ! empty( $errors ) && defined( 'WP_DEBUG' ) && WP_DEBUG ) {
        foreach ( $errors as $error ) {
            error_log( 'DPG: ' . $error );
        }
    }
    
    return empty( $errors );
}

/**
 * Load plugin textdomain for internationalization
 */
add_action( 'plugins_loaded', 'dpg_load_textdomain', 1 );
function dpg_load_textdomain() {
    $loaded = load_plugin_textdomain( 
        DPG_TEXT_DOMAIN, 
        false, 
        dirname( plugin_basename( __FILE__ ) ) . '/languages/' 
    );

    if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
        error_log( 'DPG: Textdomain ' . ( $loaded ? 'loaded' : 'failed to load' ) );
    }
}

/**
 * Initialize plugin options with defaults
 */
function dpg_init_options() {
    $default_options = [
        'dpg_pages_active' => true,
        'dpg_debug_mode' => false,
        'dpg_auto_meta' => true,
        'dpg_sitemap_enabled' => true,
        'dpg_minify_html' => false,
        'dpg_compress_output' => false,
        'dpg_custom_scripts' => '',
    ];

    // Remove old cache-related options that are no longer used
    $old_cache_options = [
        'dpg_cache_enabled',
        'dpg_cache_duration',
        'dpg_preload_enabled'
    ];

    foreach ( $old_cache_options as $old_option ) {
        delete_option( $old_option );
    }

    // Add new default options if they don't exist
    foreach ( $default_options as $option => $default_value ) {
        if ( get_option( $option ) === false ) {
            add_option( $option, $default_value );
        }
    }
}

/* =========================================================================
 * TEMPLATE CONTENT UTILITY FUNCTIONS
 * =========================================================================
 */

/**
 * Get template content based on source type (custom HTML vs page-based)
 *
 * @param array  $template_data Template configuration data
 * @param string $current_item  Current item to replace in template
 * @return string Processed template content
 */
function dpg_get_template_content( $template_data, $current_item ) {
    // Sanitize inputs
    $template_source = isset( $template_data['template_source'] )
        ? sanitize_text_field( $template_data['template_source'] )
        : 'custom';
    $current_item = sanitize_text_field( $current_item );

    /* ------------------------------------------------------------------
     * 1) PAGE-BASED TEMPLATE
     * ---------------------------------------------------------------- */
    if ( $template_source === 'page' && ! empty( $template_data['source_page_id'] ) ) {
        $page_id = absint( $template_data['source_page_id'] );
        $page    = get_post( $page_id );

        if ( $page && $page->post_status === 'publish' && $page->post_type === 'page' ) {
            // Apply content filters but prevent infinite loops
            remove_filter( 'the_content', 'dpg_prevent_infinite_loop' );
            $content = apply_filters( 'the_content', $page->post_content );
            add_filter( 'the_content', 'dpg_prevent_infinite_loop' );

            /* ——— CASE-INSENSITIVE PLACEHOLDER ——— */
            $content = str_ireplace( '{item}', esc_html( $current_item ), $content );
            /* ------------------------------------- */

            // Extra filters (meta replacement etc.)
            $content = apply_filters( 'dpg_template_content', $content, $template_data, $current_item );
            return $content;
        }
    }

    /* ------------------------------------------------------------------
     * 2) CUSTOM HTML TEMPLATE
     * ---------------------------------------------------------------- */
    $content = isset( $template_data['template_html'] )
        ? wp_kses_post( $template_data['template_html'] )
        : '';

    /* ——— CASE-INSENSITIVE PLACEHOLDER ——— */
    $content = str_ireplace( '{item}', esc_html( $current_item ), $content );
    /* ------------------------------------- */

    // Extra filters (meta replacement etc.)
    return apply_filters( 'dpg_template_content', $content, $template_data, $current_item );
}

/**
 * Prevent infinite loops in content processing
 */
function dpg_prevent_infinite_loop( $content ) {
    // Simple check to prevent processing DPG content recursively
    if ( strpos( $content, 'dpg-generated-content' ) !== false ) {
        return $content;
    }
    return $content;
}
add_filter( 'the_content', 'dpg_prevent_infinite_loop', 999 );

/**
 * Enhanced helper function to check if a page is being used as a template source
 *
 * @param int $page_id Page ID to check
 * @return bool|array True if page is used as template source, or array of template info
 */
function dpg_is_page_template_source( $page_id, $return_details = false ) {
    if ( ! defined( 'DPG_DATA_DIR' ) ) {
        return false;
    }

    $page_id = absint( $page_id );
    if ( ! $page_id ) {
        return false;
    }

    // Use caching to improve performance
    $cache_key = 'dpg_template_source_' . $page_id;
    $cached_result = wp_cache_get( $cache_key, 'dpg' );
    if ( false !== $cached_result ) {
        return $return_details ? $cached_result : ! empty( $cached_result );
    }

    global $wp_filesystem;
    if ( ! function_exists( 'WP_Filesystem' ) ) {
        require_once ABSPATH . 'wp-admin/includes/file.php';
    }
    
    if ( ! WP_Filesystem() ) {
        return false;
    }

    $json_files = glob( DPG_DATA_DIR . '*.json' );
    if ( ! $json_files ) {
        wp_cache_set( $cache_key, [], 'dpg', 300 );
        return false;
    }

    $template_info = [];
    foreach ( $json_files as $file ) {
        if ( ! $wp_filesystem->exists( $file ) ) {
            continue;
        }
        
        $content = $wp_filesystem->get_contents( $file );
        if ( false === $content ) {
            continue;
        }
        
        $data = json_decode( $content, true );
        if ( isset( $data['template_source'], $data['source_page_id'] ) &&
             $data['template_source'] === 'page' &&
             absint( $data['source_page_id'] ) === $page_id
        ) {
            $template_info[] = [
                'context_id' => basename( $file, '.json' ),
                'name' => $data['name'] ?? __( 'Unnamed Template', 'dynamic-seo-pages'),
                'created_date' => $data['created_date'] ?? '',
                'items_count' => is_array( $data['items'] ?? [] ) ? count( $data['items'] ) : 0,
            ];
        }
    }

    // Cache the result
    wp_cache_set( $cache_key, $template_info, 'dpg', 300 );

    return $return_details ? $template_info : ! empty( $template_info );
}

/**
 * Clear template source cache when needed
 */
function dpg_clear_template_source_cache( $page_id = null ) {
    if ( $page_id ) {
        wp_cache_delete( 'dpg_template_source_' . $page_id, 'dpg' );
    } else {
        // Clear all template source caches
        global $wp_object_cache;
        if ( isset( $wp_object_cache->cache['dpg'] ) ) {
            foreach ( array_keys( $wp_object_cache->cache['dpg'] ) as $key ) {
                if ( strpos( $key, 'dpg_template_source_' ) === 0 ) {
                    wp_cache_delete( $key, 'dpg' );
                }
            }
        }
    }
}

/**
 * Enhanced noindex handling for template source pages
 */
add_action( 'wp_head', 'dpg_add_noindex_to_template_sources', 1 );
function dpg_add_noindex_to_template_sources() {
    if ( is_page() && isset( $GLOBALS['post'] ) ) {
        $template_info = dpg_is_page_template_source( $GLOBALS['post']->ID, true );
        if ( ! empty( $template_info ) ) {
            echo '<meta name="robots" content="noindex, nofollow">' . "\n";
            echo '<!-- ' . esc_html__( 'This page is used as a DPG template source', 'dynamic-seo-pages') . ' -->' . "\n";
            
            // Add structured data to help with debugging
            if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                echo '<!-- DPG Template Info: ' . esc_html( wp_json_encode( $template_info ) ) . ' -->' . "\n";
            }
        }
    }
}

/**
 * Enhanced caching prevention for template source pages
 */
add_action( 'wp', 'dpg_prevent_template_source_caching', 1 );
function dpg_prevent_template_source_caching() {
    if ( is_page() && isset( $GLOBALS['post'] ) ) {
        if ( dpg_is_page_template_source( $GLOBALS['post']->ID ) ) {
            // Set no-cache headers
            if ( ! headers_sent() ) {
                header( 'Cache-Control: no-cache, no-store, must-revalidate, max-age=0' );
                header( 'Pragma: no-cache' );
                header( 'Expires: Thu, 01 Jan 1970 00:00:00 GMT' );
            }
            
            // WordPress cache functions
            if ( function_exists( 'wp_cache_set' ) ) {
                wp_cache_set( 'dpg_no_cache_' . $GLOBALS['post']->ID, true, 'dpg', 3600 );
            }
            
            // Popular caching plugin integration
            if ( function_exists( 'wp_cache_no_cache' ) ) {
                wp_cache_no_cache();
            }
            
            // WP Super Cache
            if ( defined( 'WPCACHEHOME' ) && function_exists( 'wp_cache_serve_cache_file' ) ) {
                define( 'DONOTCACHEPAGE', true );
            }
            
            // W3 Total Cache
            if ( defined( 'W3TC' ) ) {
                define( 'DONOTCACHEPAGE', true );
            }
            
            // WP Rocket
            if ( function_exists( 'rocket_clean_domain' ) ) {
                define( 'DONOTCACHEPAGE', true );
            }
        }
    }
}

/**
 * Clear template source cache when pages are updated
 */
add_action( 'save_post', 'dpg_clear_cache_on_page_update', 10, 2 );
function dpg_clear_cache_on_page_update( $post_id, $post ) {
    if ( $post && $post->post_type === 'page' ) {
        dpg_clear_template_source_cache( $post_id );
        
        // If this page is used as a template source, clear related caches
        if ( dpg_is_page_template_source( $post_id ) ) {
            // Clear all DPG caches
            dpg_clear_all_caches();
        }
    }
}

/**
 * Clear all DPG-related caches
 */
function dpg_clear_all_caches() {
    // Clear our internal caches
    dpg_clear_template_source_cache();
    
    // Clear WordPress object cache for DPG keys
    wp_cache_flush_group( 'dpg' );
    
    // Clear transients
    global $wpdb;
    $wpdb->query( "DELETE FROM {$wpdb->options} WHERE option_name LIKE '_transient_dpg_%' OR option_name LIKE '_transient_timeout_dpg_%'" );
    
    // Clear file-based cache if it exists
    if ( defined( 'DPG_CACHE_DIR' ) && is_dir( DPG_CACHE_DIR ) ) {
        $cache_files = glob( DPG_CACHE_DIR . '*.cache' );
        if ( $cache_files ) {
            foreach ( $cache_files as $file ) {
                if ( is_file( $file ) ) {
                    unlink( $file );
                }
            }
        }
    }
    
    do_action( 'dpg_clear_all_caches' );
}

/**
 * Get count of user-created templates only (exclude master templates)
 *
 * @return int Number of user templates
 */
function dpg_get_user_template_count() {
    $template_count = 0;
    $json_files = glob( DPG_DATA_DIR . '*.json' );
    
    if ( ! $json_files ) {
        return 0;
    }
    
    foreach ( $json_files as $file ) {
        $ctx_id = basename( $file, '.json' );
        
        // Skip master templates
        if ( dpg_is_master_template( $ctx_id ) ) {
            continue;
        }
        
        $template_count++;
    }
    
    return $template_count;
}

/**
 * Get count of generated pages from user templates only
 *
 * @return int Number of generated pages
 */
function dpg_get_generated_pages_count() {
    $total_pages = 0;
    $json_files = glob( DPG_DATA_DIR . '*.json' );
    
    if ( ! $json_files ) {
        return 0;
    }
    
    foreach ( $json_files as $file ) {
        $ctx_id = basename( $file, '.json' );
        
        // Skip master templates
        if ( dpg_is_master_template( $ctx_id ) ) {
            continue;
        }
        
        $data = json_decode( file_get_contents( $file ), true );
        if ( is_array( $data ) && isset( $data['items'] ) && is_array( $data['items'] ) ) {
            $total_pages += count( $data['items'] );
        }
    }
    
    return $total_pages;
}

/* =========================================================================
 * QUERY VARS REGISTRATION
 * =========================================================================
 */

/**
 * Register custom query variables with improved error handling
 *
 * @param array $vars Existing query vars
 * @return array Modified query vars
 */
add_filter( 'query_vars', 'dpg_register_query_vars', 1 );
function dpg_register_query_vars( $vars ) {
    $dpg_vars = [ 'dpg_ctx', 'dpg_item' ];
    
    foreach ( $dpg_vars as $var ) {
        if ( ! in_array( $var, $vars, true ) ) {
            $vars[] = $var;
        }
    }
    
    if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
        error_log( 'DPG: Query vars registered - ' . implode( ', ', $dpg_vars ) );
    }
    
    return $vars;
}

/**
 * Add query vars to WordPress public query vars with validation
 */
add_action( 'init', 'dpg_add_public_query_vars', 5 );
function dpg_add_public_query_vars() {
    global $wp;
    
    $dpg_vars = [ 'dpg_ctx', 'dpg_item' ];
    
    foreach ( $dpg_vars as $var ) {
        if ( ! in_array( $var, $wp->public_query_vars, true ) ) {
            $wp->add_query_var( $var );
        }
    }
    
    if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
        error_log( 'DPG: Public query vars added - ' . implode( ', ', $dpg_vars ) );
        error_log( 'DPG: Total public query vars: ' . count( $wp->public_query_vars ) );
    }
}

add_action( 'init', 'dpg_initialize_rewrite_system', 1 );
function dpg_initialize_rewrite_system() {
    // Ensure query vars are registered first
    dpg_add_public_query_vars();
    
    // Then add rewrite rules
    dpg_add_rewrite_rules();
    
    if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
        error_log( 'DPG: Rewrite system initialized' );
        
        // Debug current query vars
        global $wp_query;
        if ( isset( $wp_query ) ) {
            $ctx = get_query_var( 'dpg_ctx' );
            $item = get_query_var( 'dpg_item' );
            if ( $ctx || $item ) {
                error_log( 'DPG: Current page query vars - ctx: ' . $ctx . ', item: ' . $item );
            }
        }
    }
}

add_action( 'wp_loaded', 'dpg_maybe_flush_rewrite_rules' );
function dpg_maybe_flush_rewrite_rules() {
    $base_path = dpg_get_wp_base_path_compliant();
    $installation_signature = md5( home_url() . site_url() . ABSPATH );
    $flush_key = 'dpg_rules_flushed_' . $installation_signature . '_' . DPG_VERSION;
    
    // Check if we need to flush rules for this installation/version
    if ( ! get_option( $flush_key, false ) ) {
        if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
            error_log( 'DPG: Flushing rewrite rules for base path: "' . $base_path . '"' );
        }
        
        // Clear our caches first
        dpg_clear_rewrite_cache();
        
        // Add our rules
        dpg_add_rewrite_rules();
        
        // Flush WordPress rules
        flush_rewrite_rules( true );
        
        // Mark as flushed for this installation/version
        update_option( $flush_key, true, false );
        
        if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
            error_log( 'DPG: Rewrite rules flushed and cached for: ' . $flush_key );
        }
    }
}

/* =========================================================================
 * ENHANCED REWRITE RULES
 * =========================================================================
 */

/**
 * Utility – returns true when the template context ends in -area **or** -item
 */
function dpg_is_service_ctx( $ctx_id ) {
    return in_array( substr( $ctx_id, -5 ), [ '-area', '-item' ], true );
}



/**
 * WordPress-compliant base path detection
 */
function dpg_get_wp_base_path_compliant() {
    static $base_path = null;
    
    if ( null !== $base_path ) {
        return $base_path;
    }
    
    // Method 1: WordPress-standard using home_url()
    $home_url = home_url();
    $home_parsed = wp_parse_url( $home_url );
    $home_path = isset( $home_parsed['path'] ) ? $home_parsed['path'] : '';
    
    // Method 2: WordPress-standard using site_url() as fallback
    $site_url = site_url();
    $site_parsed = wp_parse_url( $site_url );
    $site_path = isset( $site_parsed['path'] ) ? $site_parsed['path'] : '';
    
    if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
        error_log( 'DPG: Debug path detection:' );
        error_log( 'DPG: home_url() = ' . $home_url );
        error_log( 'DPG: site_url() = ' . $site_url );
        error_log( 'DPG: home_path = ' . $home_path );
        error_log( 'DPG: site_path = ' . $site_path );
    }
    
    // Priority 1: Use home_path if it exists and is not root
    if ( $home_path && $home_path !== '/' ) {
        $base_path = trim( $home_path, '/' );
        
        if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
            error_log( 'DPG: Using home_path base: "' . $base_path . '"' );
        }
        return $base_path;
    }
    
    // Priority 2: Use site_path if it exists and is not root
    if ( $site_path && $site_path !== '/' ) {
        $base_path = trim( $site_path, '/' );
        
        if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
            error_log( 'DPG: Using site_path base: "' . $base_path . '"' );
        }
        return $base_path;
    }
    
    // Default: Root installation
    $base_path = '';
    
    if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
        error_log( 'DPG: Using default root base: "' . $base_path . '"' );
    }
    
    return $base_path;
}

/**
 * Enhanced check for master templates with caching
 *
 * @param string $ctx_id Context ID to check
 * @return bool True if master template
 */
function dpg_is_master_template( $ctx_id ) {
    $ctx_id = sanitize_file_name( $ctx_id );
    
    // Use caching for performance
    $cache_key = 'dpg_master_template_' . $ctx_id;
    $cached_result = wp_cache_get( $cache_key, 'dpg' );
    if ( false !== $cached_result ) {
        return $cached_result;
    }
    
    $json_file = DPG_DATA_DIR . $ctx_id . '.json';
    
    global $wp_filesystem;
    if ( ! function_exists( 'WP_Filesystem' ) ) {
        require_once ABSPATH . 'wp-admin/includes/file.php';
    }
    
    if ( ! WP_Filesystem() ) {
        wp_cache_set( $cache_key, false, 'dpg', 300 );
        return false;
    }
    
    if ( ! $wp_filesystem->exists( $json_file ) ) {
        wp_cache_set( $cache_key, false, 'dpg', 300 );
        return false;
    }
    
    $content = $wp_filesystem->get_contents( $json_file );
    if ( false === $content ) {
        wp_cache_set( $cache_key, false, 'dpg', 300 );
        return false;
    }
    
    $data = json_decode( $content, true );
    if ( ! is_array( $data ) ) {
        wp_cache_set( $cache_key, false, 'dpg', 300 );
        return false;
    }
    
    // If 'name' field exists and is not empty, it's a user template
    $is_master = ! ( isset( $data['name'] ) && ! empty( $data['name'] ) );
    
    wp_cache_set( $cache_key, $is_master, 'dpg', 300 );
    return $is_master;
}



/**
 * Clear rewrite rules cache when needed
 */
function dpg_clear_rewrite_cache() {
    // Clear all possible cached rule variations
    $base_path = dpg_get_wp_base_path_compliant();
    $installation_signature = md5( home_url() . site_url() . ABSPATH );
    
    $cache_keys = [
        'dpg_rewrite_rules',
        'dpg_rewrite_rules_' . md5( $base_path . DPG_VERSION ),
        'dpg_rewrite_rules_' . $installation_signature . '_' . DPG_VERSION,
        'dpg_rewrite_rules_fixed'
    ];
    
    foreach ( $cache_keys as $key ) {
        delete_transient( $key );
    }
    
    // Force flush WordPress rewrite rules
    flush_rewrite_rules( true );
    
    if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
        error_log( 'DPG: All rewrite rule caches cleared for base path: "' . $base_path . '"' );
        error_log( 'DPG: Installation signature: ' . $installation_signature );
    }
}

/* =========================================================================
 * AJAX HANDLERS FOR ADMIN
 * =========================================================================
 */

/**
 * Handle cache clearing AJAX request
 */
add_action( 'wp_ajax_dpg_clear_cache', 'dpg_handle_clear_cache_ajax' );
function dpg_handle_clear_cache_ajax() {
    // Verify nonce
    if ( ! check_ajax_referer( 'dpg_clear_cache', 'nonce', false ) ) {
        wp_send_json_error( __( 'Security verification failed.', 'dynamic-seo-pages'), 403 );
    }

    // Check permissions
    if ( ! current_user_can( DPG_REQUIRED_CAPABILITY ) ) {
        wp_send_json_error( __( 'Insufficient permissions.', 'dynamic-seo-pages'), 403 );
    }

    try {
        dpg_clear_all_caches();
        dpg_clear_rewrite_cache();
        
        wp_send_json_success( __( 'All caches cleared successfully.', 'dynamic-seo-pages') );
    } catch ( Exception $e ) {
        wp_send_json_error( __( 'Failed to clear cache.', 'dynamic-seo-pages'), 500 );
    }
}

/**
 * WordPress-compliant AJAX handler for page data
 */
add_action( 'wp_ajax_dpg_get_page_data', 'dpg_ajax_get_comprehensive_page_data' );
function dpg_ajax_get_comprehensive_page_data() {
    // WordPress security checks
    if ( ! current_user_can( 'manage_options' ) ) {
        wp_die( 'Permission denied', 403 );
    }
    
    $page_id = intval( $_GET['id'] ?? 0 );
    if ( ! $page_id ) {
        wp_die( 'Invalid page ID', 400 );
    }
    
    $post = get_post( $page_id );
    if ( ! $post ) {
        wp_die( 'Page not found', 404 );
    }
    
    // WordPress-compliant: Allow access to draft/private pages for users with edit capability
    $allowed_statuses = [ 'publish' ];
    if ( current_user_can( 'edit_post', $post->ID ) ) {
        $allowed_statuses = [ 'publish', 'draft', 'private' ];
    }
    
    if ( ! in_array( $post->post_status, $allowed_statuses, true ) ) {
        wp_die( 'Page not accessible', 403 );
    }
    
    // Get comprehensive page data for FULL SEO analysis
    $page_data = dpg_get_comprehensive_page_data_for_seo( $post );
    
    // WordPress-compliant: Return embedded script tag for JavaScript consumption
    printf(
        '<script type="application/json" id="dpg-page-data-%d">%s</script>',
        $page_id,
        wp_json_encode( $page_data, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES )
    );
    
    exit;
}

/**
 * WordPress-compliant comprehensive page data extraction
 */
function dpg_get_comprehensive_page_data_for_seo( $post ) {
    // WordPress-compliant: Apply content filters to render shortcodes, blocks, etc.
    $raw_content = $post->post_content;
    $processed_content = apply_filters( 'the_content', $raw_content );
    
    // WordPress-compliant: Enhanced content processing for better analysis
    if ( strlen( $processed_content ) - strlen( $raw_content ) < 50 ) {
        // Try manual processing for better content extraction
        $processed_content = do_shortcode( $raw_content );
        $processed_content = wpautop( $processed_content );
        
        // WordPress-compliant: Apply additional content filters
        $processed_content = wptexturize( $processed_content );
        $processed_content = convert_smilies( $processed_content );
        $processed_content = convert_chars( $processed_content );
    }
    
    // WordPress-compliant: Ensure {item} placeholder for template analysis
    if ( false === strpos( $processed_content, '{item}' ) ) {
        // Add {item} placeholder in multiple strategic locations for better SEO analysis
        $placeholder_comment = "<!-- Dynamic content placeholder for SEO analysis: {item} -->\n";
        
        // Add at the beginning
        $processed_content = $placeholder_comment . $processed_content;
        
        // WordPress-compliant: Try to add in the first paragraph for natural placement
        if ( preg_match( '/<p[^>]*>/', $processed_content, $matches, PREG_OFFSET_CAPTURE ) ) {
            $insert_pos = $matches[0][1] + strlen( $matches[0][0] );
            $processed_content = substr_replace( 
                $processed_content, 
                ' {item} ', 
                $insert_pos, 
                0 
            );
        }
        
        // WordPress-compliant: Add to first heading if available
        if ( preg_match( '/<h[1-6][^>]*>/i', $processed_content, $matches, PREG_OFFSET_CAPTURE ) ) {
            $insert_pos = $matches[0][1] + strlen( $matches[0][0] );
            $processed_content = substr_replace( 
                $processed_content, 
                ' {item} ', 
                $insert_pos, 
                0 
            );
        }
    }
    
    // WordPress-compliant: Extract comprehensive metadata
    $meta_title = get_post_meta( $post->ID, '_yoast_wpseo_title', true );
    if ( ! $meta_title ) {
        $meta_title = get_post_meta( $post->ID, '_aioseop_title', true );
    }
    if ( ! $meta_title ) {
        $meta_title = get_post_meta( $post->ID, '_genesis_title', true );
    }
    
    $meta_description = get_post_meta( $post->ID, '_yoast_wpseo_metadesc', true );
    if ( ! $meta_description ) {
        $meta_description = get_post_meta( $post->ID, '_aioseop_description', true );
    }
    if ( ! $meta_description ) {
        $meta_description = get_post_meta( $post->ID, '_genesis_description', true );
    }
    
    // WordPress-compliant: Get excerpt with multiple fallbacks
    $excerpt = $post->post_excerpt;
    if ( ! $excerpt ) {
        $excerpt = $meta_description;
    }
    if ( ! $excerpt ) {
        $excerpt = wp_trim_words( wp_strip_all_tags( $processed_content ), 25 );
    }
    
    // WordPress-compliant: Build comprehensive data array for FULL SEO analysis
    $page_data = array(
        // CORE CONTENT (essential for full analysis)
        'content' => $processed_content,
        'raw_content' => $raw_content,
        'title' => apply_filters( 'the_title', $post->post_title, $post->ID ),
        'excerpt' => $excerpt,
        
        // METADATA (for enhanced analysis)
        'meta_title' => $meta_title,
        'meta_description' => $meta_description,
        'id' => $post->ID,
        'status' => $post->post_status,
        'slug' => $post->post_name,
        'permalink' => get_permalink( $post->ID ),
        
        // ANALYSIS METRICS (for comprehensive scoring)
        'content_length' => strlen( $processed_content ),
        'raw_content_length' => strlen( $raw_content ),
        'has_item_placeholder' => ( false !== strpos( $processed_content, '{item}' ) ),
        'word_count' => str_word_count( wp_strip_all_tags( $processed_content ) ),
        'method' => 'ajax_comprehensive',
        'timestamp' => current_time( 'timestamp' ),
        
        // SEO ELEMENTS (for complete analysis)
        'featured_image' => get_the_post_thumbnail_url( $post->ID, 'large' ),
        'categories' => wp_get_post_categories( $post->ID, array( 'fields' => 'names' ) ),
        'tags' => wp_get_post_tags( $post->ID, array( 'fields' => 'names' ) ),
        
        // CONTENT ANALYSIS HELPERS (enables structure checks)
        'has_headings' => dpg_content_has_headings( $processed_content ),
        'has_images' => dpg_content_has_images( $processed_content ),
        'has_links' => dpg_content_has_links( $processed_content ),
        'heading_count' => dpg_count_headings( $processed_content ),
        'image_count' => dpg_count_images( $processed_content ),
        'link_count' => dpg_count_links( $processed_content ),
        'estimated_reading_time' => dpg_estimate_reading_time( $processed_content ),
    );
    
    // WordPress-compliant: Debug information for troubleshooting
    if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
        $page_data['debug'] = array(
            'filters_applied' => did_action( 'the_content' ),
            'shortcodes_processed' => ( strlen( $processed_content ) > strlen( $raw_content ) ),
            'content_difference' => strlen( $processed_content ) - strlen( $raw_content ),
            'extraction_method' => 'comprehensive_ajax',
            'placeholders_added' => substr_count( $processed_content, '{item}' ),
            'seo_plugin_detected' => dpg_detect_seo_plugin(),
        );
    }
    
    return $page_data;
}

/**
 * Helper functions for content analysis
 */
function dpg_content_has_headings( $content ) {
    return preg_match( '/<h[1-6][^>]*>/i', $content ) > 0;
}

function dpg_content_has_images( $content ) {
    return preg_match( '/<img[^>]*>/i', $content ) > 0;
}

function dpg_content_has_links( $content ) {
    return preg_match( '/<a[^>]*href/i', $content ) > 0;
}

function dpg_count_headings( $content ) {
    preg_match_all( '/<h[1-6][^>]*>/i', $content, $matches );
    return count( $matches[0] );
}

function dpg_count_images( $content ) {
    preg_match_all( '/<img[^>]*>/i', $content, $matches );
    return count( $matches[0] );
}

function dpg_count_links( $content ) {
    preg_match_all( '/<a[^>]*href/i', $content, $matches );
    return count( $matches[0] );
}

function dpg_estimate_reading_time( $content ) {
    $word_count = str_word_count( wp_strip_all_tags( $content ) );
    return max( 1, round( $word_count / 200 ) ); // 200 words per minute
}

function dpg_detect_seo_plugin() {
    $seo_plugins = array();
    
    if ( defined( 'WPSEO_VERSION' ) ) {
        $seo_plugins[] = 'Yoast SEO';
    }
    if ( defined( 'AIOSEOP_VERSION' ) ) {
        $seo_plugins[] = 'All in One SEO';
    }
    if ( defined( 'GENESIS_SEO_SETTINGS_FIELD' ) ) {
        $seo_plugins[] = 'Genesis SEO';
    }
    if ( class_exists( 'RankMath' ) ) {
        $seo_plugins[] = 'RankMath';
    }
    
    return implode( ', ', $seo_plugins );
}

/* =========================================================================
 * WORDPRESS-COMPLIANT BACKUP SYSTEM
 * =========================================================================
 */

/**
 * WordPress-compliant backup creation using database storage
 */
function dpg_create_compliant_backup() {
    $urls = dpg_get_all_dynamic_urls();
    
    if ( empty( $urls ) ) {
        return false;
    }
    
    // Store backup data in WordPress options table (compliant approach)
    $backup_data = [
        'urls' => $urls,
        'timestamp' => time(),
        'site_url' => get_site_url(),
        'plugin_version' => DPG_VERSION,
        'wp_version' => get_bloginfo( 'version' ),
        'total_urls' => count( $urls ),
    ];
    
    // Store in database (WordPress-compliant)
    update_option( 'dpg_current_backup', $backup_data, false );
    update_option( 'dpg_last_backup_time', time(), false );
    
    // Optional: Also create file backup in uploads directory
    $upload_dir = wp_upload_dir();
    $backup_dir = $upload_dir['basedir'] . '/dpg-backups/';
    
    if ( wp_mkdir_p( $backup_dir ) ) {
        $backup_file = $backup_dir . 'dpg-backup-' . date( 'Y-m-d' ) . '.txt';
        $backup_content = dpg_generate_backup_content( $urls, 'automatic' );
        
        // Use WordPress file functions
        $result = file_put_contents( $backup_file, $backup_content );
        
        if ( $result ) {
            update_option( 'dpg_backup_file_path', $backup_file, false );
        }
    }
    
    if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
        error_log( 'DPG: WordPress-compliant backup created for ' . count( $urls ) . ' URLs' );
    }
    
    return true;
}

/**
 * WordPress-compliant backup retrieval
 */
function dpg_get_backup_data() {
    $backup_data = get_option( 'dpg_current_backup', [] );
    
    if ( empty( $backup_data ) ) {
        return false;
    }
    
    // Validate backup data structure
    $required_fields = [ 'urls', 'timestamp', 'total_urls' ];
    foreach ( $required_fields as $field ) {
        if ( ! isset( $backup_data[ $field ] ) ) {
            return false;
        }
    }
    
    return $backup_data;
}

/**
 * WordPress-compliant backup restoration using template_redirect
 */
add_action( 'template_redirect', 'dpg_handle_backup_restore', 1 );
function dpg_handle_backup_restore() {
    // Only handle if plugin was deactivated and we have backup data
    if ( ! is_404() ) {
        return;
    }
    
    $backup_data = dpg_get_backup_data();
    if ( ! $backup_data ) {
        return;
    }
    
    $current_path = parse_url( $_SERVER['REQUEST_URI'], PHP_URL_PATH );
    $current_path = untrailingslashit( $current_path );
    
    // Check if current URL matches any backed up URL
    foreach ( $backup_data['urls'] as $backed_url ) {
        $backed_path = untrailingslashit( parse_url( $backed_url, PHP_URL_PATH ) );
        
        if ( $current_path === $backed_path ) {
            // WordPress-compliant redirect
            wp_safe_redirect( 
                add_query_arg( 'dpg_restored', '1', home_url( '/' ) ), 
                301 
            );
            exit;
        }
    }
}

/**
 * AJAX handler for manual backup export (WordPress-compliant)
 */
add_action( 'wp_ajax_dpg_export_backup', 'dpg_export_compliant_backup' );
function dpg_export_compliant_backup() {
    // WordPress security checks
    if ( ! current_user_can( 'manage_options' ) ) {
        wp_die( 'Permission denied', 403 );
    }
    
    if ( ! check_ajax_referer( 'dpg_export', 'nonce', false ) ) {
        wp_die( 'Security check failed', 403 );
    }
    
    $urls = dpg_get_all_dynamic_urls();
    
    if ( empty( $urls ) ) {
        wp_die( 'No URLs to export', 404 );
    }
    
    // Generate download content
    $content = dpg_generate_backup_content( $urls, 'manual' );
    
    // Set download headers (WordPress-compliant)
    nocache_headers();
    header( 'Content-Type: text/plain; charset=utf-8' );
    header( 'Content-Disposition: attachment; filename="dpg-backup-' . date( 'Y-m-d' ) . '.txt"' );
    header( 'Content-Length: ' . strlen( $content ) );
    
    echo $content;
    exit;
}

/**
 * WordPress-compliant scheduled backup
 */
add_action( 'dpg_daily_backup', 'dpg_run_compliant_backup' );
function dpg_run_compliant_backup() {
    $success = dpg_create_compliant_backup();
    
    if ( $success ) {
        if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
            error_log( 'DPG: Scheduled backup completed successfully' );
        }
    } else {
        if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
            error_log( 'DPG: Scheduled backup failed - no URLs found' );
        }
    }
}

/**
 * Generate standardized backup content
 */
function dpg_generate_backup_content( $urls, $backup_type = 'manual' ) {
    $content = "# Dynamic SEO Pages URL Backup\n";
    $content .= "# Generated: " . current_time( 'Y-m-d H:i:s' ) . "\n";
    $content .= "# Site: " . get_site_url() . "\n";
    $content .= "# Plugin Version: " . DPG_VERSION . "\n";
    $content .= "# WordPress Version: " . get_bloginfo( 'version' ) . "\n";
    $content .= "# Backup Type: " . ucfirst( $backup_type ) . "\n";
    $content .= "# Total URLs: " . count( $urls ) . "\n";
    $content .= "# ================================================\n\n";
    
    $content .= "INSTRUCTIONS:\n";
    $content .= "If your Dynamic SEO Pages plugin is deleted:\n";
    $content .= "1. Add 301 redirects to your hosting control panel\n";
    $content .= "2. Or use the WordPress redirect instructions below\n";
    $content .= "3. This will redirect all URLs to prevent 404 errors\n\n";
    
    $content .= "FULL URLS (" . count( $urls ) . " total):\n";
    foreach ( $urls as $url ) {
        $content .= home_url( $url ) . "\n";
    }
    
    $content .= "\n" . str_repeat( "=", 50 ) . "\n";
    $content .= "WORDPRESS REDIRECT CODE:\n";
    $content .= str_repeat( "=", 50 ) . "\n\n";
    
    $content .= "// Add this to your theme's functions.php file:\n";
    $content .= "add_action( 'template_redirect', 'dpg_emergency_redirects' );\n";
    $content .= "function dpg_emergency_redirects() {\n";
    $content .= "    \$legacy_urls = [\n";
    foreach ( $urls as $url ) {
        $content .= "        '" . esc_attr( $url ) . "',\n";
    }
    $content .= "    ];\n";
    $content .= "    \$current_path = parse_url( \$_SERVER['REQUEST_URI'], PHP_URL_PATH );\n";
    $content .= "    if ( in_array( \$current_path, \$legacy_urls, true ) ) {\n";
    $content .= "        wp_safe_redirect( home_url( '/' ), 301 );\n";
    $content .= "        exit;\n";
    $content .= "    }\n";
    $content .= "}\n\n";
    
    return $content;
}

function dpg_get_all_dynamic_urls() {
    $urls = [];
    
    // Use WordPress filesystem API consistently
    global $wp_filesystem;
    if ( ! function_exists( 'WP_Filesystem' ) ) {
        require_once ABSPATH . 'wp-admin/includes/file.php';
    }
    
    if ( ! WP_Filesystem() ) {
        return $urls;
    }
    
    $json_files = glob( DPG_DATA_DIR . '*.json' );
    if ( ! $json_files ) {
        return $urls;
    }
    
    foreach ( $json_files as $file ) {
        if ( ! $wp_filesystem->exists( $file ) ) {
            continue;
        }
        
        $ctx_id = sanitize_file_name( basename( $file, '.json' ) );
        if ( dpg_is_master_template( $ctx_id ) ) {
            continue;
        }
        
        $content = $wp_filesystem->get_contents( $file );
        if ( false === $content ) {
            continue;
        }
        
        $data = json_decode( $content, true );
        if ( ! is_array( $data ) || empty( $data['items'] ) ) {
            continue;
        }
        
        foreach ( $data['items'] as $item ) {
            $name = is_array( $item ) ? sanitize_text_field( $item['name'] ) : sanitize_text_field( $item );
            $slug = sanitize_title( $name );
            
            if ( substr( $ctx_id, -5 ) === '-area' ) {
                $base = substr( $ctx_id, 0, -5 );
                $urls[] = '/' . sanitize_title( $base ) . '-' . $slug . '/';
            } else {
                $urls[] = '/' . $slug . '/';
            }
        }
    }
    
    return array_unique( $urls );
}

/* =========================================================================
 * WORDPRESS-COMPLIANT DEACTIVATION SYSTEM (NO .HTACCESS MODIFICATION)
 * =========================================================================
 */

/**
 * WordPress-compliant deactivation handler
 * Store URLs in database instead of .htaccess modifications
 */
function dpg_deactivation_handler() {
    if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
        error_log( 'DPG: Plugin deactivation started' );
    }
    
    // Get all dynamic URLs
    $urls = dpg_get_all_dynamic_urls();
    
    if ( ! empty( $urls ) ) {
        // Store URLs in database for WordPress-compliant handling
        update_option( 'dpg_legacy_urls', $urls, false );
        update_option( 'dpg_deactivated_timestamp', time(), false );
        
        if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
            error_log( 'DPG: Stored ' . count( $urls ) . ' URLs for redirect handling' );
        }
    }
    
    // Clear scheduled events
    wp_clear_scheduled_hook( 'dpg_daily_backup' );
    
    // Clear rewrite rules (WordPress-compliant)
    flush_rewrite_rules();
    
    if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
        error_log( 'DPG: Plugin deactivation completed' );
    }
}

/**
 * WordPress-compliant redirect handling
 * Use template_redirect hook instead of .htaccess
 */
add_action( 'template_redirect', 'dpg_handle_legacy_url_redirects', 1 );
function dpg_handle_legacy_url_redirects() {
    // Only run if we have legacy URLs and plugin is deactivated
    $legacy_urls = get_option( 'dpg_legacy_urls', [] );
    if ( empty( $legacy_urls ) ) {
        return;
    }
    
    // Check if current URL was a DPG URL
    $current_path = parse_url( $_SERVER['REQUEST_URI'], PHP_URL_PATH );
    $current_path = untrailingslashit( $current_path );
    
    foreach ( $legacy_urls as $legacy_url ) {
        $legacy_path = untrailingslashit( parse_url( $legacy_url, PHP_URL_PATH ) );
        
        if ( $current_path === $legacy_path ) {
            // WordPress-compliant redirect
            wp_safe_redirect( 
                add_query_arg( 'dpg_moved', '1', home_url( '/' ) ), 
                301 
            );
            exit;
        }
    }
}

/**
 * Clean up legacy URLs when plugin is reactivated
 */
add_action( 'init', 'dpg_cleanup_legacy_urls_on_reactivation', 5 );
function dpg_cleanup_legacy_urls_on_reactivation() {
    // Only run once after reactivation
    $cleanup_key = 'dpg_reactivation_cleanup_' . DPG_VERSION;
    if ( get_option( $cleanup_key, false ) ) {
        return;
    }
    
    // Check if we have legacy URLs
    $legacy_urls = get_option( 'dpg_legacy_urls', [] );
    if ( ! empty( $legacy_urls ) ) {
        // Clear legacy URLs since plugin is active again
        delete_option( 'dpg_legacy_urls' );
        delete_option( 'dpg_deactivated_timestamp' );
        
        if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
            error_log( 'DPG: Cleaned up legacy URL redirects on reactivation' );
        }
    }
    
    // Mark cleanup as done
    update_option( $cleanup_key, true, false );
}

/**
 * Handle redirects notice
 */
add_action( 'template_redirect', 'dpg_handle_moved_notice' );
function dpg_handle_moved_notice() {
    // Show notice if someone hit a redirected URL
    if ( isset( $_GET['dpg_moved'] ) && $_GET['dpg_moved'] === '1' ) {
        // You can customize this - redirect to a specific page or show a message
        add_action( 'wp_footer', function() {
            echo '<script>console.log("DPG: This URL was redirected due to plugin deactivation");</script>';
        });
    }
}

/* =========================================================================
 * INCLUDE CLASS FILES
 * =========================================================================
 */

// Include classes with proper error handling
function dpg_include_classes() {
    $classes = [
        'class-dpg-db.php'           => 'DPG_DB',   
        'class-dpg-admin.php' => 'DPG_Admin',
        'class-dpg-frontend.php' => 'DPG_Frontend',
        'class-dpg-seo-protection.php' => 'DPG_SEO_Protection',
        'class-dpg-cron.php' => 'DPG_Cron',
    ];
    
    foreach ( $classes as $file => $class_name ) {
        $file_path = DPG_PLUGIN_DIR . 'includes/' . $file;
        if ( file_exists( $file_path ) ) {
            require_once $file_path;
            if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                error_log( "DPG: Loaded class file - {$file}" );
            }
        } else {
            if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                error_log( "DPG: Missing class file - {$file}" );
            }
        }
    }
}

dpg_include_classes();

/* =========================================================================
 * PLUGIN ACTIVATION/DEACTIVATION/UNINSTALL
 * ========================================================================= */

register_activation_hook( __FILE__, 'dpg_activation_handler' );
function dpg_activation_handler() {
    /* 1 ── Requirements -------------------------------------------------- */
    $errors = dpg_check_requirements();
    if ( ! empty( $errors ) ) {
        deactivate_plugins( plugin_basename( __FILE__ ) );
        wp_die(
            '<h1>' . esc_html__( 'Plugin Activation Failed', 'dynamic-seo-pages') . '</h1>' .
            '<p>'  . esc_html__( 'Dynamic SEO Pages could not be activated due to the following issues:', 'dynamic-seo-pages') . '</p>' .
            '<ul><li>' . implode( '</li><li>', array_map( 'esc_html', $errors ) ) . '</li></ul>' .
            '<p><a href="' . esc_url( admin_url( 'plugins.php' ) ) . '">' .
            esc_html__( 'Return to Plugins', 'dynamic-seo-pages') .
            '</a></p>'
        );
    }

    /* 2 ── Options & directories ---------------------------------------- */
    dpg_init_options();
    dpg_create_compliant_directories();

    /* 3 ── Install / update DB schema ----------------------------------- */
    if ( class_exists( 'DPG_DB' ) && method_exists( 'DPG_DB', 'install' ) ) {
        DPG_DB::install();
    }

    /* 4 ── Rewrite rules ------------------------------------------------- */
    dpg_add_rewrite_rules();
    flush_rewrite_rules();

    /* 5 ── Caches & misc ------------------------------------------------- */
    dpg_clear_all_caches();

    set_transient( 'dpg_activation_notice', true, 30 );

    if ( ! wp_next_scheduled( 'dpg_daily_backup' ) ) {
        wp_schedule_event( time(), 'daily', 'dpg_daily_backup' );
    }

    if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
        error_log( 'DPG: plugin activated (user ID ' . get_current_user_id() . ')' );
    }
}

register_deactivation_hook( __FILE__, 'dpg_deactivation_handler' );

/**
 * WordPress-compliant cleanup function for uninstall
 */
function dpg_remove_compliant_directories() {
    $upload_dir = wp_upload_dir();
    
    $directories = [
        $upload_dir['basedir'] . '/dpg-data/',
        $upload_dir['basedir'] . '/dpg-templates/',
        $upload_dir['basedir'] . '/dpg-cache/',
        $upload_dir['basedir'] . '/dpg-backups/',
    ];
    
    global $wp_filesystem;
    if ( ! function_exists( 'WP_Filesystem' ) ) {
        require_once ABSPATH . 'wp-admin/includes/file.php';
    }
    
    if ( ! WP_Filesystem() ) {
        return false;
    }
    
    foreach ( $directories as $dir ) {
        if ( $wp_filesystem->exists( $dir ) ) {
            $wp_filesystem->rmdir( $dir, true ); // Recursive delete
        }
    }
    
    return true;
}

/**
 * WordPress-compliant uninstall handler
 */
register_uninstall_hook( __FILE__, 'dpg_uninstall_handler' );
function dpg_uninstall_handler() {
    /* 1 ── Cron --------------------------------------------------------- */
    wp_clear_scheduled_hook( 'dpg_daily_backup' );

    /* 2 ── Drop DB tables ---------------------------------------------- */
    if ( class_exists( 'DPG_DB' ) && method_exists( 'DPG_DB', 'uninstall' ) ) {
        DPG_DB::uninstall();
    }

    /* 3 ── Options ------------------------------------------------------ */
    $options_to_delete = [
        'dpg_legacy_urls',
        'dpg_current_backup',
        'dpg_backup_file_path',
        'dpg_last_backup_time',
        'dpg_deactivated_timestamp',
        'dpg_pages_active',
        'dpg_deleted_templates',
        'dpg_global_score',
        'dpg_debug_mode',
        'dpg_auto_meta',
        'dpg_sitemap_enabled',
        'dpg_minify_html',
        'dpg_compress_output',
        'dpg_custom_scripts',
    ];
    foreach ( $options_to_delete as $option ) {
        delete_option( $option );
    }

    /* 4 ── Transients --------------------------------------------------- */
    global $wpdb;
    $wpdb->query(
        "DELETE FROM {$wpdb->options}
         WHERE option_name LIKE '_transient_dpg_%'
            OR option_name LIKE '_transient_timeout_dpg_%'"
    );

    /* 5 ── Local folders ------------------------------------------------ */
    dpg_remove_compliant_directories();
}

/* =========================================================================
 * CLASS INITIALIZATION
 * ========================================================================= */
add_action( 'init', 'dpg_initialize_classes', 10 );
function dpg_initialize_classes() {
    $classes_to_init = [
        'DPG_SEO_Protection' => 'SEO Protection',
        'DPG_Admin' => 'Admin Interface',
        'DPG_Frontend' => 'Frontend Handler',
        'DPG_Cron' => 'Cron Jobs',
    ];
    
    foreach ( $classes_to_init as $class_name => $description ) {
        if ( class_exists( $class_name ) ) {
            try {
                if ( method_exists( $class_name, 'init' ) ) {
                    call_user_func( [ $class_name, 'init' ] );
                } else {
                    new $class_name();
                }
                
                if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                    error_log( "DPG: {$description} ({$class_name}) initialized successfully" );
                }
            } catch ( Exception $e ) {
                if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                    error_log( "DPG: Failed to initialize {$class_name} - " . $e->getMessage() );
                }
            }
        } else {
            if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
                error_log( "DPG: Class {$class_name} not found" );
            }
        }
    }
}

/**
 * Add action links to plugin listing page
 */
add_filter( 'plugin_action_links_' . plugin_basename( __FILE__ ), 'dpg_add_plugin_action_links' );
function dpg_add_plugin_action_links( $links ) {
    $custom_links = [];
    
    $custom_links['settings'] = sprintf(
        '<a href="%s">%s</a>',
        esc_url( admin_url( 'admin.php?page=dpg-dashboard' ) ),
        esc_html__( 'Settings', 'dynamic-seo-pages')
    );
    
    return array_merge( $custom_links, $links );
}

/**
 * Add meta links below description (WordPress.org compliant)
 */
add_filter( 'plugin_row_meta', 'dpg_add_plugin_row_meta', 10, 2 );
function dpg_add_plugin_row_meta( $links, $file ) {
    if ( plugin_basename( __FILE__ ) === $file ) {
        $row_meta = [
            'support' => sprintf(
                '<a href="%s" target="_blank" rel="noopener">%s</a>',
                esc_url( 'https://wordpress.org/support/plugin/dynamic-seo-pages/' ),
                esc_html__( 'Support', 'dynamic-seo-pages')
            ),
        ];
        
        return array_merge( $links, $row_meta );
    }
    
    return $links;
}

/* =========================================================================
 * PLUGIN HEALTH CHECK
 * ========================================================================= */

/**
 * Add plugin health check to WordPress Site Health
 */
add_filter( 'site_status_tests', 'dpg_add_site_health_tests' );
function dpg_add_site_health_tests( $tests ) {
    $tests['direct']['dpg_plugin_health'] = [
        'label' => __( 'Dynamic SEO Pages Health', 'dynamic-seo-pages'),
        'test'  => 'dpg_site_health_test',
    ];
    return $tests;
}

function dpg_site_health_test() {
    $result = [
        'label'       => __( 'Dynamic SEO Pages is working correctly', 'dynamic-seo-pages'),
        'status'      => 'good',
        'badge'       => [
            'label' => __( 'Plugin', 'dynamic-seo-pages'),
            'color' => 'blue',
        ],
        'description' => sprintf(
            '<p>%s</p>',
            __( 'Dynamic SEO Pages is configured correctly and functioning properly.', 'dynamic-seo-pages')
        ),
        'test'        => 'dpg_plugin_health',
    ];

    // Check for basic issues only
    $issues = [];
    
    // Check directories
    if ( ! is_dir( DPG_DATA_DIR ) || ! is_writable( DPG_DATA_DIR ) ) {
        $issues[] = __( 'Data directory is not writable', 'dynamic-seo-pages');
    }
    
    // Check if pages are active
    if ( ! get_option( 'dpg_pages_active', false ) ) {
        $issues[] = __( 'Dynamic pages are not active', 'dynamic-seo-pages');
    }
    
    // Check for templates
    $template_count = count( glob( DPG_DATA_DIR . '*.json' ) );
    if ( $template_count === 0 ) {
        $issues[] = __( 'No templates have been created yet', 'dynamic-seo-pages');
    }

    // Update status based on issues found
    if ( ! empty( $issues ) ) {
        $result['status'] = 'recommended';
        $result['label'] = __( 'Dynamic SEO Pages has some recommendations', 'dynamic-seo-pages');
        $result['description'] = '<p>' . __( 'Dynamic SEO Pages is working but has some recommendations:', 'dynamic-seo-pages') . '</p>' .
                               '<ul><li>' . implode( '</li><li>', array_map( 'esc_html', $issues ) ) . '</li></ul>';
    }

    return $result;
}

// Final debug log
if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
    error_log( 'DPG: Main plugin file loaded successfully v' . DPG_VERSION );
}