<?php
// File: wp-content/plugins/dynamic-seo-pages/includes/class-dpg-db.php

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Handles database storage for Dynamic SEO Pages
 * ────────────────────────────────────────────────────────────────
 * 1. Per-context "items" tables     – one for every area / keyword
 * 2. Global  dpg_templates  table   – one row per template, incl. SEO score
 * 3. Global  dpg_scores     table   – optional historic scores (last run)
 */
class DPG_DB {

	/*--------------------------------------------------------------
	# 0.  One-time bootstrap
	--------------------------------------------------------------*/
	public static function init() {

		// Ensure the option that tracks contexts exists (even on brand-new installs)
		add_action( 'init', static function () {
			if ( false === get_option( 'dpg_contexts' ) ) {
				add_option( 'dpg_contexts', [] );
			}
		} );

		// Create / upgrade the global tables once on every pageload (cheap DB check)
		add_action( 'init', [ __CLASS__, 'maybe_create_templates_table' ], 4 );
		add_action( 'init', [ __CLASS__, 'maybe_create_score_table'     ], 5 );
	}

	/*--------------------------------------------------------------
	# 1.  PER-CONTEXT "ITEMS" TABLES
	--------------------------------------------------------------*/

	/** Create (or upgrade) one items table for a context */
	public static function create_table( $type, $id ) {
		global $wpdb;

		$type = ( $type === 'area' ) ? 'area' : 'keyword';
		$id   = absint( $id );

		// Validate inputs
		if ( ! $id || ! in_array( $type, [ 'area', 'keyword' ], true ) ) {
			if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
				error_log( 'DPG_DB: Invalid parameters for create_table - type: ' . $type . ', id: ' . $id );
			}
			return false;
		}

		require_once ABSPATH . 'wp-admin/includes/upgrade.php';
		$charset = $wpdb->get_charset_collate();

		$table    = self::get_table_name( $type, $id );
		$slug_col = ( $type === 'area' ) ? 'area_slug'    : 'keyword_slug';
		$name_col = ( $type === 'area' ) ? 'area_name'    : 'keyword_name';

		$sql = "
		CREATE TABLE {$table} (
			id          MEDIUMINT(9) NOT NULL AUTO_INCREMENT,
			{$slug_col} VARCHAR(200) NOT NULL,
			{$name_col} VARCHAR(200) NOT NULL,
			PRIMARY KEY  (id),
			UNIQUE KEY   {$slug_col} ({$slug_col})
		) {$charset};
		";

		$result = dbDelta( $sql );
		
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
			error_log( 'DPG_DB: Created/updated table ' . $table . ' - ' . print_r( $result, true ) );
		}
		
		return $result;
	}

	/** Overwrite every item in a context */
	public static function save_items( $type, $id, array $items ) {
		global $wpdb;

		$type = ( $type === 'area' ) ? 'area' : 'keyword';
		$id   = absint( $id );
		
		// Validate inputs
		if ( ! $id || ! in_array( $type, [ 'area', 'keyword' ], true ) ) {
			if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
				error_log( 'DPG_DB: Invalid parameters for save_items' );
			}
			return false;
		}

		$table    = self::get_table_name( $type, $id );
		$slug_col = ( $type === 'area' ) ? 'area_slug'    : 'keyword_slug';
		$name_col = ( $type === 'area' ) ? 'area_name'    : 'keyword_name';

		// Validate table name for security
		if ( ! self::validate_table_name( $table ) ) {
			if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
				error_log( 'DPG_DB: Invalid table name: ' . $table );
			}
			return false;
		}

		// Use DELETE instead of TRUNCATE for better security and transaction support
		$delete_result = $wpdb->query( "DELETE FROM `{$table}`" );
		if ( false === $delete_result ) {
			if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
				error_log( 'DPG_DB: Failed to clear table: ' . $table . ' - ' . $wpdb->last_error );
			}
			return false;
		}

		$success_count = 0;
		foreach ( $items as $txt ) {
			$insert_result = $wpdb->insert(
				$table,
				[
					$name_col => sanitize_text_field( $txt ),
					$slug_col => sanitize_title( $txt ),
				],
				[ '%s', '%s' ]
			);
			
			if ( false !== $insert_result ) {
				$success_count++;
			} elseif ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
				error_log( 'DPG_DB: Failed to insert item: ' . $txt . ' - ' . $wpdb->last_error );
			}
		}

		if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
			error_log( 'DPG_DB: Saved ' . $success_count . ' of ' . count( $items ) . ' items to ' . $table );
		}

		return $success_count;
	}

	/** Get table name helper */
	private static function get_table_name( $type, $id ) {
		global $wpdb;
		$type = sanitize_key( $type );
		$id = absint( $id );
		$prefix = ( $type === 'area' ) ? 'dpg_areas_' : 'dpg_keywords_';
		return $wpdb->prefix . $prefix . $id;
	}

	/** Validate table name for security */
	private static function validate_table_name( $table_name ) {
		global $wpdb;
		
		// Must start with WordPress prefix and follow our naming pattern
		$pattern = '/^' . preg_quote( $wpdb->prefix, '/' ) . 'dpg_(areas|keywords)_\d+$/';
		return preg_match( $pattern, $table_name ) === 1;
	}

	/*--------------------------------------------------------------
	# 2.  GLOBAL  dpg_templates  REGISTRY  (one row = one template)
	--------------------------------------------------------------*/

	private static function templates_table() {
		global $wpdb;
		return $wpdb->prefix . 'dpg_templates';
	}

	/** Create / upgrade the registry table (adds `seo_score` if missing) */
	public static function maybe_create_templates_table() {
		global $wpdb;

		$table   = self::templates_table();
		$charset = $wpdb->get_charset_collate();

		// Secure table existence check
		$table_exists = $wpdb->get_var( $wpdb->prepare( 
			"SELECT table_name FROM information_schema.tables 
			 WHERE table_schema = %s AND table_name = %s",
			DB_NAME,
			$table
		) );

		if ( ! $table_exists ) {
			require_once ABSPATH . 'wp-admin/includes/upgrade.php';
			$sql = "
			CREATE TABLE {$table} (
				id        BIGINT UNSIGNED NOT NULL AUTO_INCREMENT,
				ctx_id    VARCHAR(191)    NOT NULL,
				name      VARCHAR(191)    NOT NULL,
				seo_score TINYINT UNSIGNED NOT NULL DEFAULT 0,
				created   DATETIME        NOT NULL DEFAULT CURRENT_TIMESTAMP,
				PRIMARY KEY (id),
				UNIQUE KEY ctx_id (ctx_id)
			) {$charset};
			";
			
			$result = dbDelta( $sql );
			
			if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
				error_log( 'DPG_DB: Created templates table - ' . print_r( $result, true ) );
			}
			return;
		}

		// Table exists – make sure the new column is there (older installs)
		$has_col = $wpdb->get_var( $wpdb->prepare(
			"SELECT column_name FROM information_schema.columns 
			 WHERE table_schema = %s AND table_name = %s AND column_name = %s",
			DB_NAME,
			$table,
			'seo_score'
		) );

		if ( ! $has_col ) {
			$alter_result = $wpdb->query( 
				"ALTER TABLE {$table} ADD COLUMN seo_score TINYINT UNSIGNED NOT NULL DEFAULT 0" 
			);
			
			if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
				if ( false === $alter_result ) {
					error_log( 'DPG_DB: Failed to add seo_score column - ' . $wpdb->last_error );
				} else {
					error_log( 'DPG_DB: Added seo_score column to templates table' );
				}
			}
		}
	}

	/**
	 * Update *any* column on the templates table (currently used for `seo_score`)
	 *
	 * @param int    $id     Numeric template-ID (not ctx_id)
	 * @param string $column Column name (validated below)
	 * @param mixed  $value  New value
	 */
	public static function update_template_column( $id, $column, $value ) {
		global $wpdb;

		$allowed = [ 'name', 'seo_score' ];
		if ( ! in_array( $column, $allowed, true ) ) {
			if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
				error_log( 'DPG_DB: Invalid column name for update: ' . $column );
			}
			return false;
		}

		$id = absint( $id );
		if ( ! $id ) {
			if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
				error_log( 'DPG_DB: Invalid ID for template column update: ' . $id );
			}
			return false;
		}

		$result = $wpdb->update(
			self::templates_table(),
			[ $column => $value ],
			[ 'id'    => $id ],
			[ '%s' ],               // value format (string works for TINYINT too)
			[ '%d' ]                // where format
		);

		if ( false === $result && defined( 'WP_DEBUG' ) && WP_DEBUG ) {
			error_log( 'DPG_DB: Failed to update template column - ' . $wpdb->last_error );
		}

		return $result;
	}

	/*--------------------------------------------------------------
	# 3.  GLOBAL  dpg_scores  (optional "last run" score)
	--------------------------------------------------------------*/

	private static function score_table() {
		global $wpdb;
		return $wpdb->prefix . 'dpg_scores';
	}

	public static function maybe_create_score_table() {
		global $wpdb;

		$table   = self::score_table();
		$charset = $wpdb->get_charset_collate();

		// Secure table existence check
		$table_exists = $wpdb->get_var( $wpdb->prepare( 
			"SELECT table_name FROM information_schema.tables 
			 WHERE table_schema = %s AND table_name = %s",
			DB_NAME,
			$table
		) );

		if ( $table_exists ) {
			return; // already there
		}

		require_once ABSPATH . 'wp-admin/includes/upgrade.php';
		$sql = "
		CREATE TABLE {$table} (
			ctx_id     VARCHAR(191) NOT NULL,
			last_score TINYINT      NOT NULL,
			updated_at DATETIME     NOT NULL,
			PRIMARY KEY (ctx_id)
		) {$charset};
		";
		
		$result = dbDelta( $sql );
		
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
			error_log( 'DPG_DB: Created scores table - ' . print_r( $result, true ) );
		}
	}

	/** Save / replace the latest score (0-100) */
	public static function save_score( string $ctx_id, int $score ) {
		global $wpdb;

		$ctx_id = sanitize_key( $ctx_id );
		$score = max( 0, min( 100, intval( $score ) ) );

		if ( ! $ctx_id ) {
			if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
				error_log( 'DPG_DB: Invalid ctx_id for save_score' );
			}
			return false;
		}

		$result = $wpdb->replace(
			self::score_table(),
			[
				'ctx_id'     => $ctx_id,
				'last_score' => $score,
				'updated_at' => current_time( 'mysql' ),
			],
			[ '%s', '%d', '%s' ]
		);

		if ( false === $result && defined( 'WP_DEBUG' ) && WP_DEBUG ) {
			error_log( 'DPG_DB: Failed to save score - ' . $wpdb->last_error );
		}

		return $result;
	}

	/**
	 * Fetch one score (int) or many (OBJECT_K map) or **all** (OBJECT_K)
	 *
	 * @param string|array|null $ctx_id  null = all
	 */
	public static function get_score( $ctx_id = null ) {
		global $wpdb;
		$table = self::score_table();

		// single
		if ( is_string( $ctx_id ) ) {
			$ctx_id = sanitize_key( $ctx_id );
			if ( ! $ctx_id ) {
				return null;
			}
			
			return $wpdb->get_var(
				$wpdb->prepare( "SELECT last_score FROM {$table} WHERE ctx_id = %s", $ctx_id )
			);
		}

		// many (array)
		if ( is_array( $ctx_id ) ) {
			if ( ! $ctx_id ) {
				return [];
			}
			
			// Sanitize all context IDs
			$sanitized_ids = array_map( 'sanitize_key', $ctx_id );
			$sanitized_ids = array_filter( $sanitized_ids ); // Remove empty values
			
			if ( ! $sanitized_ids ) {
				return [];
			}
			
			$placeholders = implode( ',', array_fill( 0, count( $sanitized_ids ), '%s' ) );
			return $wpdb->get_results(
				$wpdb->prepare(
					"SELECT ctx_id, last_score FROM {$table}
					 WHERE ctx_id IN ( {$placeholders} )",
					$sanitized_ids
				),
				OBJECT_K
			);
		}

		// all
		return $wpdb->get_results( "SELECT ctx_id, last_score FROM {$table}", OBJECT_K );
	}

	/** Remove score row when a template is deleted */
	public static function delete_score( string $ctx_id ) {
		global $wpdb;
		
		$ctx_id = sanitize_key( $ctx_id );
		if ( ! $ctx_id ) {
			if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
				error_log( 'DPG_DB: Invalid ctx_id for delete_score' );
			}
			return false;
		}
		
		$result = $wpdb->delete( 
			self::score_table(), 
			[ 'ctx_id' => $ctx_id ], 
			[ '%s' ] 
		);
		
		if ( false === $result && defined( 'WP_DEBUG' ) && WP_DEBUG ) {
			error_log( 'DPG_DB: Failed to delete score - ' . $wpdb->last_error );
		}
		
		return $result;
	}

	/*--------------------------------------------------------------
	# 4.  CLEANUP / UNINSTALL METHODS
	--------------------------------------------------------------*/

	/**
	 * Clean up all DPG database tables and data
	 * Called during plugin uninstall
	 */
	public static function uninstall() {
		global $wpdb;

		// Drop all DPG tables
		$tables = $wpdb->get_results( 
			$wpdb->prepare(
				"SELECT table_name FROM information_schema.tables 
				 WHERE table_schema = %s AND table_name LIKE %s",
				DB_NAME,
				$wpdb->prefix . 'dpg_%'
			),
			ARRAY_A
		);

		foreach ( $tables as $table ) {
			$table_name = $table['table_name'];
			
			// Extra security: validate table name before dropping
			if ( strpos( $table_name, $wpdb->prefix . 'dpg_' ) === 0 ) {
				$wpdb->query( "DROP TABLE IF EXISTS `{$table_name}`" );
				
				if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
					error_log( 'DPG_DB: Dropped table: ' . $table_name );
				}
			}
		}

		// Clean up options
		delete_option( 'dpg_contexts' );
		
		if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
			error_log( 'DPG_DB: Uninstall cleanup completed' );
		}
	}

	/**
	 * Get database statistics for admin display
	 */
	public static function get_stats() {
		global $wpdb;

		$stats = [
			'templates_count' => 0,
			'total_items' => 0,
			'tables_count' => 0,
		];

		// Count templates
		$templates_table = self::templates_table();
		$stats['templates_count'] = $wpdb->get_var( "SELECT COUNT(*) FROM {$templates_table}" );

		// Count DPG tables
		$tables = $wpdb->get_results( 
			$wpdb->prepare(
				"SELECT table_name FROM information_schema.tables 
				 WHERE table_schema = %s AND table_name LIKE %s",
				DB_NAME,
				$wpdb->prefix . 'dpg_%'
			),
			ARRAY_A
		);

		$stats['tables_count'] = count( $tables );

		// Count total items across all item tables
		foreach ( $tables as $table ) {
			$table_name = $table['table_name'];
			
			// Only count item tables (not templates or scores)
			if ( strpos( $table_name, '_areas_' ) !== false || strpos( $table_name, '_keywords_' ) !== false ) {
				$count = $wpdb->get_var( "SELECT COUNT(*) FROM `{$table_name}`" );
				$stats['total_items'] += intval( $count );
			}
		}

		return $stats;
	}
}

/* Kick-start the option & table checks */
DPG_DB::init();