<?php
// File: includes/class-dpg-page-template.php
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class DPG_Page_Template {

    /** Holds our custom templates (file => human name) */
    private static $templates = [];

    /** Initialize all hooks */
    public static function init() {
        add_filter( 'theme_page_templates',  [ __CLASS__, 'add_templates' ],     10, 4 );
        add_filter( 'wp_insert_post_data',   [ __CLASS__, 'register_templates' ] );
        add_filter( 'template_include',      [ __CLASS__, 'load_template' ],     99    );
    }

    /**
     * Build our templates list from contexts, sanitizing every piece.
     *
     * @return array filename => human name
     */
    private static function get_templates() : array {
        if ( empty( self::$templates ) ) {
            $contexts = get_option( 'dpg_contexts', [] );
            if ( ! is_array( $contexts ) ) {
                return [];
            }

            foreach ( $contexts as $ctx ) {
                // sanitize each value
                $id        = sanitize_file_name( $ctx['id']        ?? '' );
                $type      = sanitize_key(       $ctx['type']      ?? '' );
                $name      = sanitize_text_field( $ctx['name']     ?? '' );
                $slug_base = sanitize_text_field( $ctx['slug_base'] ?? '' );

                // only allow our two types
                if ( ! $id || ! in_array( $type, [ 'area', 'keyword' ], true ) ) {
                    continue;
                }

                $file = "{$id}-{$type}-template.php";
                self::$templates[ $file ] = "{$name} ({$slug_base})";
            }
        }

        return self::$templates;
    }

    /**
     * Inject our templates into the Theme’s page‐template dropdown.
     */
    public static function add_templates( $post_templates, $wp_theme, $post, $post_type ) {
        if ( 'page' === $post_type ) {
            return array_merge( $post_templates, self::get_templates() );
        }
        return $post_templates;
    }

    /**
     * Ensure WP knows about them when saving a page (caches page templates).
     */
    public static function register_templates( $data ) {
        if ( isset( $data['post_type'] ) && 'page' === $data['post_type'] ) {
            $theme = wp_get_theme();
            $cache = $theme->get_page_templates();
            $cache = array_merge( $cache, self::get_templates() );
            // override theme’s internal cache
            $theme->page_templates = $cache;
        }
        return $data;
    }

    /**
     * When rendering a page, if it’s using one of ours, load it from plugin.
     */
    public static function load_template( $template ) {
        if ( is_page() ) {
            $post      = get_queried_object();
            $assigned  = get_page_template_slug( $post->ID );
            $templates = self::get_templates();

            if ( isset( $templates[ $assigned ] ) ) {
                $path = DPG_PLUGIN_DIR . 'templates/' . $assigned;
                // ensure it’s a real, readable file
                if ( is_file( $path ) && is_readable( $path ) ) {
                    return $path;
                }
            }
        }

        return $template;
    }
}

// Bootstrap
DPG_Page_Template::init();
