<?php
// File: wp-content/plugins/dynamic-seo-pages/includes/class-dpg-seo-protection.php

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class DPG_SEO_Protection {

    /**
     * Debug parameters that should trigger SEO protection
     */
    const DEBUG_PARAMS = [
        'dpg_debug',
        'check_rules',
        'force_refresh',
        'dpg_test_install',
    ];

    /**
     * Admin-only debug parameters that require elevated permissions
     */
    const ADMIN_ONLY_PARAMS = [
        'force_refresh',
        'dpg_test_install',
    ];

    /**
     * Hook into WordPress.
     */
    public static function init() {
        // Only apply when debug mode is enabled
        if ( ! self::is_debug_mode_enabled() ) {
            return;
        }

        // Protect as early as possible
        add_action( 'init',   [ __CLASS__, 'secure_debug_access' ], 1 );
        add_action( 'init',   [ __CLASS__, 'apply_seo_protection' ], 2 );
        add_action( 'init',   [ __CLASS__, 'log_bot_debug_access' ], 3 );

        // Append to robots.txt
        add_filter( 'robots_txt', [ __CLASS__, 'add_debug_disallow_to_robots_txt' ], 10, 2 );
    }

    /**
     * Restrict admin-only debug functions to users with manage_options.
     */
    public static function secure_debug_access() {
        foreach ( self::ADMIN_ONLY_PARAMS as $param ) {
            if ( isset( $_GET[ $param ] ) ) {
                if ( ! current_user_can( 'manage_options' ) ) {
                    status_header( 403 );
                    if ( ! headers_sent() ) {
                        header( 'X-Robots-Tag: noindex, nofollow, nosnippet, noarchive', false );
                    }
                    wp_die(
                        '<div style="text-align:center;padding:50px;font-family:sans-serif">'
                        . '<h1>' . esc_html__( 'Access Denied', 'dynamic-seo-pages') . '</h1>'
                        . '<p>' . esc_html__( 'You need administrator privileges to use this debug feature.', 'dynamic-seo-pages') . '</p>'
                        . '</div>',
                        esc_html__( 'Access Denied', 'dynamic-seo-pages'),
                        [ 'response' => 403 ]
                    );
                }
            }
        }
    }

    /**
     * Apply SEO protection (noindex) to any debug page.
     */
    public static function apply_seo_protection() {
        if ( ! self::is_debug_page() ) {
            return;
        }

        // Add meta-robots tags in <head>
        add_action( 'wp_head', [ __CLASS__, 'add_noindex_meta_tags' ], 1 );
        // Add HTTP header
        add_action( 'wp', [ __CLASS__, 'add_noindex_http_header' ] );
        // Filter the new robots API
        add_filter( 'wp_robots', [ __CLASS__, 'modify_robots_for_debug' ] );
    }

    /**
     * Determine if this request contains any debug parameter.
     */
    private static function is_debug_page() {
        foreach ( self::DEBUG_PARAMS as $param ) {
            if ( isset( $_GET[ $param ] ) ) {
                return true;
            }
        }
        return false;
    }

    /**
     * Print noindex meta tags.
     */
    public static function add_noindex_meta_tags() {
        ?>
        <!-- DPG Debug Page – SEO Protection -->
        <meta name="robots"       content="noindex, nofollow, nosnippet, noarchive, nocache, noimageindex">
        <meta name="googlebot"    content="noindex, nofollow, nosnippet, noarchive, nocache, noimageindex">
        <meta name="bingbot"      content="noindex, nofollow, nosnippet, noarchive, nocache, noimageindex">
        <?php
    }

    /**
     * Send X-Robots-Tag header.
     */
    public static function add_noindex_http_header() {
        if ( ! headers_sent() ) {
            header( 'X-Robots-Tag: noindex, nofollow, nosnippet, noarchive, nocache, noimageindex', false );
        }
    }

    /**
     * Force everything off in the new WP Robots API.
     *
     * @param array $robots
     * @return array
     */
    public static function modify_robots_for_debug( $robots ) {
        $robots['noindex']      = true;
        $robots['nofollow']     = true;
        $robots['nosnippet']    = true;
        $robots['noarchive']    = true;
        $robots['nocache']      = true;
        $robots['noimageindex'] = true;
        return $robots;
    }

    /**
     * Disallow debug URLs in robots.txt.
     *
     * @param string $output
     * @param bool   $public
     * @return string
     */
    public static function add_debug_disallow_to_robots_txt( $output, $public ) {
        if ( '1' !== $public ) {
            return $output;
        }

        $rules  = "\n# Dynamic SEO Pages – Debug Protection\n";
        foreach ( self::DEBUG_PARAMS as $param ) {
            $param_esc = esc_html( $param );
            $rules   .= "Disallow: /*?{$param_esc}=*\n";
            $rules   .= "Disallow: /*&{$param_esc}=*\n";
        }

        return $output . $rules;
    }

    /**
     * Log when a likely bot hits a debug page.
     */
    public static function log_bot_debug_access() {
        if ( ! self::is_debug_page() ) {
            return;
        }

        $ua        = isset( $_SERVER['HTTP_USER_AGENT'] ) ? sanitize_text_field( wp_unslash( $_SERVER['HTTP_USER_AGENT'] ) ) : '';
        $request_uri = isset( $_SERVER['REQUEST_URI'] ) ? esc_url_raw( wp_unslash( $_SERVER['REQUEST_URI'] ) ) : '';
        
        $indicators = [
            'googlebot','bingbot','slurp','duckduckbot','baiduspider',
            'yandexbot','facebookexternalhit','twitterbot','crawler',
        ];

        foreach ( $indicators as $bot ) {
            if ( false !== stripos( $ua, $bot ) ) {
                error_log( 'DPG-SEO-ALERT: Bot "' . $bot . '" accessed a debug page: ' . $request_uri );
                break;
            }
        }
    }

    /**
     * Is debug mode enabled in settings?
     *
     * @return bool
     */
    public static function is_debug_mode_enabled() {
        return (bool) get_option( 'dpg_debug_mode', false );
    }
}

// Fire it up
DPG_SEO_Protection::init();