/**
 * WordPress.org Compliant Form Validation - ENHANCED VERSION
 * File: wp-content/plugins/dynamic-seo-pages/includes/forms/assets/js/form-validation.js
 * 
 * FIXES:
 * - Enhanced WordPress readiness detection
 * - Timeout and fallback systems
 * - Better error handling and user feedback
 * - Fixed initialization timing issues
 * - SEO score field updates
 */
(function($) {
    'use strict';

    /**
     * Enhanced WordPress Debug Logging
     */
    function logDebug(message, isError = false) {
        if (window.dpgForm && window.dpgForm.debug) {
            console[isError ? 'error' : 'log']('DPG: ' + message);
        }
    }

    /**
     * FIXED: More robust WordPress readiness check
     */
    function isWordPressReady() {
        const hasConfig = window.dpgForm && 
                         window.dpgForm.ajaxUrl && 
                         window.dpgForm.nonce && 
                         window.dpgForm.strings;
        
        if (!hasConfig) {
            logDebug('WordPress config missing: ' + JSON.stringify({
                hasWindow: !!window.dpgForm,
                hasAjax: !!(window.dpgForm && window.dpgForm.ajaxUrl),
                hasNonce: !!(window.dpgForm && window.dpgForm.nonce),
                hasStrings: !!(window.dpgForm && window.dpgForm.strings)
            }), true);
        }
        
        return hasConfig;
    }

    /**
     * FIXED: Wait for WordPress configuration with timeout
     */
    function waitForWordPress(callback, maxWait = 5000) {
        let attempts = 0;
        const maxAttempts = maxWait / 100;
        
        function check() {
            attempts++;
            
            if (isWordPressReady()) {
                logDebug('WordPress config ready after ' + attempts + ' attempts');
                callback();
                return;
            }
            
            if (attempts < maxAttempts) {
                setTimeout(check, 100);
            } else {
                logDebug('WordPress config timeout after ' + maxWait + 'ms', true);
                // Try to initialize anyway with fallback
                initializeFallback();
            }
        }
        
        check();
    }

    /**
     * FIXED: Fallback initialization when WordPress config is missing
     */
    function initializeFallback() {
        logDebug('Initializing with fallback configuration');
        
        // Create minimal config
        window.dpgForm = window.dpgForm || {
            ajaxUrl: (window.ajaxurl || '/wp-admin/admin-ajax.php'),
            nonce: 'fallback_mode',
            homeUrl: (window.location.origin + '/'),
            debug: true,
            strings: {
                required: 'This field is required.',
                invalid: 'Please enter a valid value.',
                security_error: 'Security verification failed. Please refresh the page.',
                saving: 'Saving...',
                seo_tip: 'SEO Tip',
                seo_tip_message: 'For best SEO, ensure your template name and slug are related.'
            }
        };
        
        // Initialize with limited functionality
        new DPGFormValidator({ fallbackMode: true });
    }

    // Character count utility
    window.updateCharacterCount = function(inputId, counterId, maxLength) {
        const input = document.getElementById(inputId);
        const counter = document.getElementById(counterId);
        
        if (input && counter) {
            const remaining = maxLength - input.value.length;
            counter.textContent = remaining;
            counter.style.color = remaining < 10 ? '#d63638' : remaining < 20 ? '#dba617' : '#646970';
        }
    };

    // Template type toggle
    window.dpgToggleSlug = function(val) {
        const slugField = document.getElementById('dpg-slug-field');
        const previewField = document.getElementById('dpg-url-preview');
        const slugInput = document.getElementById('dpg_slug');
        
        if (val === 'keyword') {
            if (slugField) slugField.style.display = 'none';
            if (previewField) previewField.style.display = 'none';
            if (slugInput) slugInput.removeAttribute('required');
        } else {
            if (slugField) slugField.style.display = 'block';
            if (previewField) previewField.style.display = 'block';
            if (slugInput) slugInput.setAttribute('required', 'required');
            window.dpgUpdatePreview();
        }
    };

    // URL preview
    window.dpgUpdatePreview = function() {
        const slug = document.getElementById('dpg_slug')?.value || 'your-slug';
        const templateName = document.getElementById('dpg_name')?.value || '';
        const previewEl = document.getElementById('url-preview');
        const warningEl = document.getElementById('seo-warning');
        
        if (previewEl && window.dpgForm && window.dpgForm.homeUrl) {
            previewEl.innerHTML = '<code>' + window.dpgForm.homeUrl + slug + '-{location}/</code>';
        }
        
        // SEO warning
        if (templateName && slug && slug !== 'your-slug' && warningEl && window.dpgForm && window.dpgForm.strings) {
            const slugWords = slug.toLowerCase().replace(/[^a-z0-9]/g, ' ').trim().split(/\s+/);
            const nameWords = templateName.toLowerCase().replace(/[^a-z0-9]/g, ' ').trim().split(/\s+/);
            const significantWords = nameWords.filter(word => word.length > 3);
            const hasMatch = significantWords.some(word => slugWords.includes(word));
            
            if (!hasMatch) {
                const strings = window.dpgForm.strings;
                warningEl.style.display = 'block';
                warningEl.innerHTML = '<div class="dpg-seo-warning"><strong>⚠️ ' + (strings.seo_tip || 'SEO Tip') + ':</strong> ' + (strings.seo_tip_message || 'For best SEO, ensure your template name and slug are related.') + '<br>' + (strings.seo_example || 'Example: Template "Best Web Designer" → Slug "web-designer"') + '</div>';
            } else {
                warningEl.style.display = 'none';
            }
        }
    };

    // Template source toggle
    window.dpgToggleTemplateSource = function(source) {
        const pageSelection = document.getElementById('dpg-page-selection');
        const customTemplate = document.getElementById('dpg-custom-template');
        const htmlTextarea = document.getElementById('dpg_template_html');
        const pageSelect = document.getElementById('dpg_source_page_id');
        
        if (source === 'page') {
            if (pageSelection) pageSelection.style.display = 'block';
            if (customTemplate) customTemplate.style.display = 'none';
            if (htmlTextarea) htmlTextarea.removeAttribute('required');
            if (pageSelect) pageSelect.setAttribute('required', 'required');
        } else {
            if (pageSelection) pageSelection.style.display = 'none';
            if (customTemplate) customTemplate.style.display = 'block';
            if (htmlTextarea) htmlTextarea.setAttribute('required', 'required');
            if (pageSelect) pageSelect.removeAttribute('required');
        }
    };

    /**
     * FIXED: Enhanced SEO Analysis with better error handling
     */
    window.dpgUpdateSEOAnalysis = function() {
        if (!isWordPressReady()) {
            logDebug('Skipping SEO analysis - WordPress not ready');
            return;
        }

        let templateContent = '';
        const templateSource = document.getElementById('dpg_template_source')?.value || 'custom';
        
        if (templateSource === 'custom') {
            templateContent = document.getElementById('dpg_template_html')?.value || '';
        } else if (templateSource === 'page') {
            const pageSelect = document.getElementById('dpg_source_page_id');
            if (pageSelect && pageSelect.value) {
                templateContent = '[Page Content will be used as template with {item} replacements]';
            }
        }
        
        const formData = {
            template_html: templateContent,
            template_source: templateSource,
            source_page_id: document.getElementById('dpg_source_page_id')?.value || '',
            hard_meta: {
                title: document.getElementById('dpg_meta_title')?.value || '',
                description: document.getElementById('dpg_meta_description')?.value || '',
                keywords: document.getElementById('dpg_meta_keywords')?.value || '',
                og_title: document.getElementById('dpg_og_title')?.value || '',
                og_description: document.getElementById('dpg_og_description')?.value || '',
                og_image: document.getElementById('dpg_og_image')?.value || ''
            }
        };
        
        const focusKeyphrase = document.getElementById('dpg_focus_keyphrase')?.value || '';
        
        // Show loading state
        const scoreEl = document.getElementById('seo-score-number');
        if (scoreEl) {
            scoreEl.textContent = '...';
            scoreEl.className = 'dpg-seo-score-number dpg-seo-loading';
        }
        
        // WordPress AJAX request with enhanced error handling
        $.post(window.dpgForm.ajaxUrl, {
            action: 'dpg_seo_analysis',
            template_data: formData,
            focus_keyphrase: focusKeyphrase,
            security: window.dpgForm.nonce,
            sample_item: 'Example Item'
        })
        .done(function(response) {
            if (response && response.success && response.data) {
                window.dpgUpdateSEODisplay(response.data);
                logDebug('SEO analysis completed successfully');
                
                // Update hidden score field for form submission
                const hiddenScore = document.getElementById('dpg_seo_score');
                if (hiddenScore && response.data.percentage !== undefined) {
                    hiddenScore.value = response.data.percentage;
                }
                
            } else {
                const errorMsg = (response && response.data) ? response.data : 'Analysis failed';
                logDebug('SEO analysis failed: ' + errorMsg, true);
                
                const scoreEl = document.getElementById('seo-score-number');
                if (scoreEl) {
                    scoreEl.textContent = 'Error';
                    scoreEl.className = 'dpg-seo-score-number dpg-seo-error';
                }
                
                // Show help message
                const helpEl = document.getElementById('seo-help');
                if (helpEl) {
                    helpEl.style.display = 'block';
                    helpEl.innerHTML = '<p style="color: #d63638;">SEO analysis temporarily unavailable. Your template will still be saved.</p>';
                }
            }
        })
        .fail(function(xhr, status, error) {
            logDebug('SEO analysis AJAX failed: ' + status + ' - ' + error, true);
            
            const scoreEl = document.getElementById('seo-score-number');
            if (scoreEl) {
                if (xhr.status === 403) {
                    const strings = window.dpgForm.strings || {};
                    const errorMsg = strings.security_error || 'Security verification failed. Please refresh the page.';
                    
                    scoreEl.textContent = '403';
                    scoreEl.className = 'dpg-seo-score-number dpg-seo-error';
                    scoreEl.title = errorMsg;
                    
                    // Show security error notice
                    const helpEl = document.getElementById('seo-help');
                    if (helpEl) {
                        helpEl.style.display = 'block';
                        helpEl.innerHTML = '<p style="color: #d63638;">' + errorMsg + '</p>';
                    }
                } else {
                    scoreEl.textContent = 'N/A';
                    scoreEl.className = 'dpg-seo-score-number dpg-seo-neutral';
                    scoreEl.title = 'SEO analysis unavailable';
                }
            }
        });
    };

    /**
     * FIXED: Enhanced SEO display update with hidden score field
     */
    window.dpgUpdateSEODisplay = function(data) {
        if (!data) return;
        
        // Update score display
        const scoreEl = document.getElementById('seo-score-number');
        if (scoreEl && data.percentage !== undefined) {
            scoreEl.textContent = data.percentage + '%';
            scoreEl.className = 'dpg-seo-score-number dpg-seo-' + (data.status || 'poor');
        }
        
        // Update hidden score field for form submission
        const hiddenScore = document.getElementById('dpg_seo_score');
        if (hiddenScore && data.percentage !== undefined) {
            hiddenScore.value = data.percentage;
        }
        
        // Hide help message
        const helpEl = document.getElementById('seo-help');
        if (helpEl && (data.problems || data.improvements || data.good_results)) {
            helpEl.style.display = 'none';
        }
        
        // Update sections
        window.dpgUpdateSEOSection('seo-problems', 'seo-problems-list', data.problems || []);
        window.dpgUpdateSEOSection('seo-improvements', 'seo-improvements-list', data.improvements || []);
        window.dpgUpdateSEOSection('seo-good', 'seo-good-list', data.good_results || []);
    };

    // SEO section update
    window.dpgUpdateSEOSection = function(sectionId, listId, items) {
        const section = document.getElementById(sectionId);
        const list = document.getElementById(listId);
        
        if (!section || !list) return;
        
        if (!Array.isArray(items) || items.length === 0) {
            section.style.display = 'none';
            return;
        }
        
        section.style.display = 'block';
        list.innerHTML = items.map(function(item) {
            const itemType = item.type || 'default';
            const message = item.message || '';
            const howToFix = item.how_to_fix ? '<div class="dpg-seo-item-fix">' + item.how_to_fix + '</div>' : '';
            
            return '<div class="dpg-seo-item dpg-seo-item--' + itemType + '">' +
                   '<div class="dpg-seo-item-message">' + message + '</div>' +
                   howToFix +
                   '</div>';
        }).join('');
    };

    /**
     * FIXED: Enhanced Form Validator with better initialization
     */
    class DPGFormValidator {
        constructor(options = {}) {
            this.options = {
                formSelector: '.dpg-form',
                errorClass: 'dpg-field-error',
                successClass: 'dpg-field-success',
                errorMessageClass: 'dpg-error-message',
                debounceDelay: 300,
                scrollToError: true,
                showSuccessIcons: true,
                fallbackMode: false,
                ...options
            };

            this.validators = {};
            this.validationRules = {};
            this.isSubmitting = false;
            this.isInitialized = false;

            // Initialize immediately if WordPress is ready, or wait
            if (isWordPressReady() || this.options.fallbackMode) {
                this.init();
            } else {
                logDebug('WordPress not ready, waiting...');
                waitForWordPress(() => this.init());
            }
        }

        init() {
            logDebug('Initializing DPG Form Validator');
            
            this.setupValidationRules();
            this.bindEvents();
            this.initializeFields();
            
            // Mark as initialized after a delay to prevent initial validation
            setTimeout(() => {
                this.isInitialized = true;
                logDebug('Form validator fully initialized');
                
                // Remove any error notices about initialization
                $('.dpg-form .notice-error').each(function() {
                    const $notice = $(this);
                    if ($notice.text().includes('Form validation could not be initialized')) {
                        $notice.fadeOut(500, function() { $(this).remove(); });
                    }
                });
                
            }, 1000);
        }

        setupValidationRules() {
            const strings = (window.dpgForm && window.dpgForm.strings) || {};
            
            // Template name - always required
            this.addRule('dpg_name', {
                required: true,
                minLength: 3,
                maxLength: 100,
                pattern: /^[a-zA-Z0-9\s\-_]+$/,
                message: strings.name_validation || 'Template name must be 3-100 characters, letters, numbers, spaces, hyphens, and underscores only.'
            });

            // Slug - required only for service-area type
            this.addRule('dpg_slug', {
                required: (value, element) => {
                    const typeField = document.getElementById('dpg_type');
                    return typeField && typeField.value === 'service-area';
                },
                pattern: /^[a-z0-9\-]+$/,
                minLength: 3,
                maxLength: 50,
                message: strings.slug_validation || 'URL slug must be lowercase letters, numbers, and hyphens only.'
            });

            // Template HTML - required only when source is custom
            this.addRule('dpg_template_html', {
                required: (value, element) => {
                    const sourceField = document.getElementById('dpg_template_source');
                    return sourceField && sourceField.value === 'custom';
                },
                minLength: 10,
                custom: this.validateHTML.bind(this),
                message: strings.template_html_validation || 'Template HTML is required for custom templates.'
            });

            // Source page - required only when source is page
            this.addRule('dpg_source_page_id', {
                required: (value, element) => {
                    const sourceField = document.getElementById('dpg_template_source');
                    return sourceField && sourceField.value === 'page';
                },
                message: strings.source_page_validation || 'Please select a page when using existing page as template.'
            });

            // Items - always required
            this.addRule('dpg_items', {
                required: true,
                custom: this.validateItems.bind(this),
                message: strings.items_validation || 'At least one item is required.'
            });

            // Email validation
            this.addRule('email', {
                pattern: /^[^\s@]+@[^\s@]+\.[^\s@]+$/,
                message: strings.email_validation || 'Please enter a valid email address.'
            });

            // URL validation
            this.addRule('url', {
                pattern: /^https?:\/\/.+/,
                message: strings.url_validation || 'Please enter a valid URL starting with http:// or https://'
            });
            
            logDebug('Validation rules setup complete');
        }

        addRule(fieldName, rule) {
            this.validationRules[fieldName] = rule;
        }

        bindEvents() {
            const self = this;
            
            // Real-time validation - only after initialization
            $(document).on('input blur', 'input, textarea, select', function(e) {
                const $field = $(this);
                if (self.isInitialized && self.shouldValidateField($field)) {
                    // Only validate on blur or after user has started typing
                    if (e.type === 'blur' || ($field.val().length > 0 && $field.data('user-interacted'))) {
                        self.debounceValidation($field);
                    }
                    // Mark field as user-interacted
                    $field.data('user-interacted', true);
                }
            });

            // WordPress form submission with enhanced error handling
            $(document).on('submit', this.options.formSelector, function(e) {
                return self.handleFormSubmit(e);
            });

            // Template type change
            $(document).on('change', '#dpg_type', function(e) {
                const value = $(this).val();
                self.handleTemplateTypeChange(value);
                window.dpgToggleSlug(value);
                
                // Clear validation errors for slug field when type changes
                const $slugField = $('#dpg_slug');
                if ($slugField.length) {
                    self.clearFieldError($slugField);
                }
            });

            // Template source change
            $(document).on('change', '#dpg_template_source', function(e) {
                const value = $(this).val();
                self.handleTemplateSourceChange(value);
                window.dpgToggleTemplateSource(value);
                
                // Clear validation errors when source changes
                const $htmlField = $('#dpg_template_html');
                const $pageField = $('#dpg_source_page_id');
                self.clearFieldError($htmlField);
                self.clearFieldError($pageField);
            });

            // Name input for slug suggestion
            $(document).on('input', '#dpg_name', function(e) {
                const value = $(this).val();
                self.updateSlugSuggestion(value);
                window.dpgUpdatePreview();
            });

            // Character counters
            $(document).on('input', '#dpg_meta_title', function(e) {
                window.updateCharacterCount('dpg_meta_title', 'title-counter', 60);
            });

            $(document).on('input', '#dpg_meta_description', function(e) {
                window.updateCharacterCount('dpg_meta_description', 'description-counter', 160);
            });

            // FIXED: Enhanced SEO analysis with better debouncing
            if (isWordPressReady() || this.options.fallbackMode) {
                let seoTimeout;
                $(document).on('input change', '#dpg_focus_keyphrase, #dpg_template_html, #dpg_meta_title, #dpg_meta_description, #dpg_template_source, #dpg_source_page_id', function(e) {
                    clearTimeout(seoTimeout);
                    seoTimeout = setTimeout(function() {
                        if (typeof window.dpgUpdateSEOAnalysis === 'function') {
                            window.dpgUpdateSEOAnalysis();
                        }
                    }, 1500); // Increased delay to reduce server load
                });
            }
            
            logDebug('Event binding complete');
        }

        initializeFields() {
            // Character counters
            setTimeout(function() {
                window.updateCharacterCount('dpg_meta_title', 'title-counter', 60);
                window.updateCharacterCount('dpg_meta_description', 'description-counter', 160);
            }, 100);

            // Initialize conditional fields without triggering validation
            this.handleTemplateTypeChange($('#dpg_type').val());
            this.handleTemplateSourceChange($('#dpg_template_source').val());

            // WordPress CodeMirror with enhanced error handling
            if (typeof wp !== 'undefined' && wp.codeEditor) {
                try {
                    const editorSettings = wp.codeEditor.defaultSettings ? wp.codeEditor.defaultSettings : {};
                    editorSettings.codemirror = Object.assign({}, editorSettings.codemirror, {
                        mode: 'htmlmixed',
                        lineNumbers: true,
                        theme: 'default'
                    });
                    
                    const templateHtml = document.getElementById('dpg_template_html');
                    if (templateHtml) {
                        const editor = wp.codeEditor.initialize('dpg_template_html', editorSettings);
                        if (editor) {
                            editor.codemirror.on('change', function() {
                                editor.codemirror.save();
                                // Mark as user-interacted for validation
                                $(templateHtml).data('user-interacted', true);
                                setTimeout(function() {
                                    $(templateHtml).trigger('input');
                                }, 100);
                            });
                            logDebug('CodeMirror initialized successfully');
                        }
                    }
                } catch (error) {
                    logDebug('CodeMirror initialization failed: ' + error.message, true);
                    // Continue without CodeMirror
                }
            }
            
            logDebug('Field initialization complete');
        }

        shouldValidateField($field) {
            const name = $field.attr('name');
            return name && this.validationRules[name] && !$field.is(':disabled');
        }

        debounceValidation($field) {
            const fieldName = $field.attr('name');
            
            clearTimeout(this.validators[fieldName]);
            this.validators[fieldName] = setTimeout(() => {
                this.validateField($field);
            }, this.options.debounceDelay);
        }

        async validateField($field) {
            const fieldName = $field.attr('name');
            const value = $field.val();
            const rule = this.validationRules[fieldName];

            if (!rule) return true;

            const errors = [];
            const strings = (window.dpgForm && window.dpgForm.strings) || {};

            // Required validation with conditional logic
            if (rule.required) {
                const isRequired = typeof rule.required === 'function' 
                    ? rule.required(value, $field[0])
                    : rule.required;

                if (isRequired && !value.trim()) {
                    errors.push(strings.required || 'This field is required.');
                }
            }

            // Skip other validations if field is empty and not required
            if (!value.trim() && !errors.length) {
                this.clearFieldError($field);
                return true;
            }

            // Length validations
            if (rule.minLength && value.length < rule.minLength) {
                errors.push((strings.min_length || 'Minimum length is {0} characters.').replace('{0}', rule.minLength));
            }

            if (rule.maxLength && value.length > rule.maxLength) {
                errors.push((strings.max_length || 'Maximum length is {0} characters.').replace('{0}', rule.maxLength));
            }

            // Pattern validation
            if (rule.pattern && !rule.pattern.test(value)) {
                errors.push(rule.message || strings.invalid || 'Invalid format.');
            }

            // Custom validation
            if (rule.custom) {
                try {
                    const customResult = await rule.custom(value, $field[0]);
                    if (customResult !== true) {
                        errors.push(typeof customResult === 'string' ? customResult : rule.message);
                    }
                } catch (error) {
                    logDebug('Custom validation error for ' + fieldName + ': ' + error.message, true);
                    errors.push(strings.validation_error || 'Validation error occurred.');
                }
            }

            // Update field display
            if (errors.length > 0) {
                this.showFieldError($field, errors[0]);
                return false;
            } else {
                this.showFieldSuccess($field);
                return true;
            }
        }

        async handleFormSubmit(e) {
            if (this.isSubmitting) {
                e.preventDefault();
                return false;
            }

            const $form = $(e.target);
            
            // FIXED: Enhanced nonce checking
            const wpNonce = $form.find('input[name="_wpnonce"]').val();
            const dpgNonce = $form.find('input[name="dpg_nonce"]').val();
            const otherNonces = $form.find('input[name$="_nonce"]').length;
            
            if (!wpNonce && !dpgNonce && !otherNonces) {
                logDebug('No WordPress nonces found in form', true);
                e.preventDefault();
                
                const strings = (window.dpgForm && window.dpgForm.strings) || {};
                this.showFormError($form, strings.security_error || 'Security verification failed. Please refresh the page.');
                return false;
            }

            logDebug('Form submission validation starting');

            const isValid = await this.validateForm($form);

            if (!isValid) {
                e.preventDefault();
                this.scrollToFirstError();
                logDebug('Form validation failed', true);
                return false;
            }

            // Show loading state
            this.showLoadingState($form);
            this.isSubmitting = true;
            
            logDebug('Form validation passed, submitting');
            return true;
        }

        async validateForm($form) {
            const $fields = $form.find('input, textarea, select').filter((i, el) => {
                return this.shouldValidateField($(el));
            });

            const validationPromises = $fields.map((i, el) => {
                return this.validateField($(el));
            }).get();

            const results = await Promise.all(validationPromises);
            const isValid = results.every(result => result === true);
            
            logDebug('Form validation results: ' + results.length + ' fields, ' + (isValid ? 'VALID' : 'INVALID'));
            return isValid;
        }

        showFieldError($field, message) {
            this.clearFieldError($field);
            
            $field.addClass(this.options.errorClass);
            
            const $errorMessage = $('<div class="' + this.options.errorMessageClass + '" style="color: #d63638; font-size: 12px; margin-top: 4px;">' + message + '</div>');
            $field.closest('.dpg-field-wrapper, .dpg-field-container, .dpg-field').append($errorMessage);
        }

        showFieldSuccess($field) {
            this.clearFieldError($field);
            
            if (this.options.showSuccessIcons) {
                $field.addClass(this.options.successClass);
            }
        }

        clearFieldError($field) {
            $field.removeClass(this.options.errorClass + ' ' + this.options.successClass);
            $field.closest('.dpg-field-wrapper, .dpg-field-container, .dpg-field')
                .find('.' + this.options.errorMessageClass).remove();
        }

        showFormError($form, message) {
            $form.find('.dpg-form-error').remove();
            
            const $errorDiv = $('<div class="dpg-form-error notice notice-error" style="background: #fcf2f2; border-left: 4px solid #d63638; padding: 12px; margin: 16px 0;"><p>' + message + '</p></div>');
            $form.prepend($errorDiv);
            
            $('html, body').animate({
                scrollTop: $errorDiv.offset().top - 100
            }, 300);
        }

        scrollToFirstError() {
            if (!this.options.scrollToError) return;

            const $firstError = $('.' + this.options.errorClass).first();
            if ($firstError.length) {
                $('html, body').animate({
                    scrollTop: $firstError.offset().top - 100
                }, 300);
                $firstError.focus();
            }
        }

        showLoadingState($form) {
            const strings = (window.dpgForm && window.dpgForm.strings) || {};
            const $submitBtn = $form.find('input[type="submit"], button[type="submit"]');
            
            $submitBtn.prop('disabled', true)
                .addClass('dpg-loading')
                .data('original-text', $submitBtn.text())
                .text(strings.saving || 'Saving...');
        }

        hideLoadingState($form) {
            const $submitBtn = $form.find('input[type="submit"], button[type="submit"]');
            $submitBtn.prop('disabled', false)
                .removeClass('dpg-loading')
                .text($submitBtn.data('original-text') || 'Submit');
            
            this.isSubmitting = false;
        }

        // FIXED: Enhanced HTML validation with case-insensitive placeholder check
        validateHTML(value, element) {
            if (!value.trim()) {
                return true;
            }

            const strings = (window.dpgForm && window.dpgForm.strings) || {};

            // HTML syntax validation
            const parser = new DOMParser();
            const doc = parser.parseFromString(value, 'text/html');
            const parserErrors = doc.querySelector('parsererror');
            if (parserErrors) {
                return strings.html_syntax_error || 'Invalid HTML syntax detected.';
            }

            // FIXED: Case-insensitive {item} placeholder check
            if (!value.toLowerCase().includes('{item}')) {
                return strings.item_placeholder_required || 'Template must include {item} placeholder for dynamic content.';
            }

            return true;
        }

        validateItems(value, element) {
            const strings = (window.dpgForm && window.dpgForm.strings) || {};
            
            if (!value.trim()) {
                return strings.items_required || 'At least one item is required.';
            }

            const lines = value.split('\n').filter(line => line.trim());
            
            if (lines.length === 0) {
                return strings.items_required || 'At least one item is required.';
            }

            for (let line of lines) {
                const trimmed = line.trim();
                if (trimmed.includes('|')) {
                    const parts = trimmed.split('|');
                    if (parts.length !== 2) {
                        return strings.item_format_error || 'Item format should be "name|score" (e.g., "Item Name|0.8").';
                    }
                    
                    const [name, score] = parts;
                    if (!name.trim()) {
                        return strings.item_name_required || 'Item name cannot be empty.';
                    }
                    const scoreNum = parseFloat(score.trim());
                    if (isNaN(scoreNum) || scoreNum < 0.1 || scoreNum > 1.0) {
                        return strings.item_score_range || 'Item scores must be between 0.1 and 1.0.';
                    }
                }
            }

            return true;
        }

        // Helper methods
        handleTemplateTypeChange(type) {
            const $slugRow = $('#row-slug');
            const $urlPreviewRow = $('#row-url-preview');
            const $slugField = $('#dpg_slug');

            if (type === 'service-area') {
                $slugRow.show();
                $urlPreviewRow.show();
                $slugField.prop('required', true);
                this.updateUrlPreview();
            } else {
                $slugRow.hide();
                $urlPreviewRow.hide();
                $slugField.prop('required', false);
                this.clearFieldError($slugField);
            }
        }

        handleTemplateSourceChange(source) {
            const $htmlField = $('#dpg_template_html');
            const $pageField = $('#dpg_source_page_id');

            // Clear existing errors
            this.clearFieldError($htmlField);
            this.clearFieldError($pageField);

            // Don't re-validate immediately after source change
            // Let user interact first
        }

        updateSlugSuggestion(name) {
            if (!name.trim()) return;

            const suggestedSlug = name.toLowerCase()
                .replace(/[^a-z0-9\s\-]/g, '')
                .replace(/\s+/g, '-')
                .replace(/-+/g, '-')
                .replace(/^-|-$/g, '');

            const $slugField = $('#dpg_slug');
            if (!$slugField.val() || $slugField.val() === $slugField.data('auto-suggestion')) {
                $slugField.val(suggestedSlug).data('auto-suggestion', suggestedSlug);
                this.updateUrlPreview();
            }
        }

        updateUrlPreview() {
            const slug = $('#dpg_slug').val() || 'your-slug';
            const homeUrl = (window.dpgForm && window.dpgForm.homeUrl) || (window.location.origin + '/');
            
            $('#url-preview').html('<code>' + homeUrl + slug + '-{location}/</code>');
        }
    }

    /**
     * FIXED: Enhanced DOM ready initialization with multiple fallbacks
     */
    $(document).ready(function() {
        logDebug('DOM ready, checking WordPress status');
        
        // Remove any existing error notices first
        $('.dpg-form .notice-error').each(function() {
            const $notice = $(this);
            if ($notice.text().includes('Form validation could not be initialized')) {
                $notice.remove();
            }
        });
        
        // Try immediate initialization
        if (isWordPressReady()) {
            logDebug('WordPress ready immediately');
            new DPGFormValidator();
        } else {
            logDebug('WordPress not ready, waiting...');
            
            // Show a subtle loading message
            const $forms = $('.dpg-form');
            if ($forms.length > 0) {
                $forms.prepend('<div class="dpg-init-notice notice notice-info" style="margin: 10px 0;"><p>🔄 Initializing form validation...</p></div>');
            }
            
            // Wait for WordPress with fallback
            waitForWordPress(function() {
                // Remove loading message
                $('.dpg-init-notice').fadeOut(300, function() { $(this).remove(); });
                
                // Initialize validator
                new DPGFormValidator();
            });
        }
        
        // Additional initialization
        setTimeout(function() {
            if (document.getElementById('title-counter')) {
                window.updateCharacterCount('dpg_meta_title', 'title-counter', 60);
            }
            if (document.getElementById('description-counter')) {
                window.updateCharacterCount('dpg_meta_description', 'description-counter', 160);
            }
        }, 1000);
    });

    // WordPress form reset handling
    $(document).on('reset', '.dpg-form', function() {
        const $form = $(this);
        
        // Clear all validation states
        $form.find('.dpg-field-error, .dpg-field-success').removeClass('dpg-field-error dpg-field-success');
        $form.find('.dpg-error-message').remove();
        $form.find('.dpg-form-error').remove();
        
        // Reset character counters
        setTimeout(function() {
            window.updateCharacterCount('dpg_meta_title', 'title-counter', 60);
            window.updateCharacterCount('dpg_meta_description', 'description-counter', 160);
        }, 100);
        
        logDebug('Form reset completed');
    });

    // Expose for backward compatibility
    window.DPGFormValidator = DPGFormValidator;

})(jQuery);