<?php
// File: wp-content/plugins/dynamic-seo-pages/includes/forms/class-dpg-form-base.php

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * WordPress.org Compliant Base Form Class
 * Provides common functionality for all DPG forms
 */
abstract class DPG_Form_Base {

    /** @var string HTML id attribute for the <form> */
    protected $form_id;

    /** @var string WP action name to handle this form via admin-post.php */
    protected $action;

    /** @var array Collected validation or processing errors */
    public $errors = [];

    /** @var array Form configuration passed from admin */
    protected $config = [];

    /**
     * WordPress.org compliant constructor.
     *
     * @param string|array $form_id_or_config Form ID or configuration array
     * @param string       $unused           Deprecated parameter for compatibility
     * @param string       $form_action      admin-post.php?action= value
     */
    public function __construct( $form_id_or_config = '', $unused = '', $form_action = '' ) {
        // Handle new configuration-based initialization
        if ( is_array( $form_id_or_config ) ) {
            $this->config = $form_id_or_config;
            $this->form_id = sanitize_key( get_class( $this ) );
            $this->action = sanitize_key( $form_action );
        } else {
            // Legacy initialization for backward compatibility
            $this->form_id = $form_id_or_config ?: sanitize_key( get_class( $this ) );
            $this->action = sanitize_key( $form_action );
            $this->config = [];
        }
        
        $this->init();
    }

    /**
     * Override in child classes to hook in.
     */
    protected function init() {
        // Child classes can add hooks here
    }

   
  protected function persist_items_to_db( $ctx_id, array $items ) {
        // Abort early if the helper class is not available (should never happen).
        if ( ! class_exists( 'DPG_DB' ) ) {
            return;
        }

        // Decide whether this is an "area" or "keyword" context from its suffix.
        $type_key = ( substr( $ctx_id, -5 ) === '-area' ) ? 'area' : 'keyword';

        /* --------------------------------------------------------------------
         * 1. Obtain (or assign) the numeric ID stored in the dpg_contexts option
         * ------------------------------------------------------------------ */
        $contexts = get_option( 'dpg_contexts', [] );

        // Look for an existing registry row.
        $row = wp_list_filter( $contexts, [ 'ctx_id' => $ctx_id ] );
        if ( $row ) {
            // The context already exists – reuse its numeric ID.
            $numeric_id = absint( array_shift( $row )['id'] );
        } else {
            // New context.  Pick the next integer ID and store it.
            $numeric_id = empty( $contexts ) ? 1 : ( max( array_column( $contexts, 'id' ) ) + 1 );

            $contexts[] = [
                'id'      => $numeric_id,
                'ctx_id'  => $ctx_id,
                'created' => current_time( 'mysql' ),
            ];

            // Do NOT autoload this potentially large option.
            update_option( 'dpg_contexts', $contexts, 'no' );
        }

        /* --------------------------------------------------------------------
         * 2. Create / ensure the per-context table and insert the items
         * ------------------------------------------------------------------ */
        DPG_DB::create_table( $type_key, $numeric_id );

        // DPG_DB::save_items() expects simple strings, so flatten any scored arrays.
        $clean_items = array_map(
            static function ( $item ) {
                return is_array( $item ) ? ( $item['name'] ?? '' ) : $item;
            },
            $items
        );

        DPG_DB::save_items( $type_key, $numeric_id, $clean_items );
    }

    
    /**
     * WordPress.org compliant asset enqueuing with comprehensive localization.
     */
public function enqueue_assets() {
    $base = plugin_dir_url( DPG_PLUGIN_FILE ) . 'includes/forms/assets/';

    /* ----------  styles ---------- */
    wp_enqueue_style(
        'dpg-admin-forms',
        $base . 'css/admin-forms.css',
        [],
        DPG_VERSION
    );

    wp_enqueue_style(
        'dpg-components-css',
        $base . 'css/components.css',
        [ 'dpg-admin-forms' ],
        DPG_VERSION
    );

    /* ----------  scripts ---------- */

    // (1) validation helper – **always enqueue first**
    wp_enqueue_script(
        'dpg-form-validation',
        $base . 'js/form-validation.js',
        [ 'jquery' ],
        DPG_VERSION,
        true
    );

    // (2) shared UI components – *register* first because we need to add inline JS
    wp_register_script(
        'dpg-components-js',
        $base . 'js/components.js',
        [ 'jquery', 'dpg-form-validation' ], // depends on validation
        DPG_VERSION,
        true
    );

    // make sure a global namespace exists before the bundle runs
    wp_add_inline_script(
        'dpg-components-js',
        'window.dpgForm = window.dpgForm || {};',
        'before'
    );

    // enqueue the components bundle
    wp_enqueue_script( 'dpg-components-js' );

    /* ----------  localise once ---------- */
    $this->localize_form_script();
}

 
protected function localize_form_script() {
    $form_config = [
        'ajaxUrl'     => admin_url( 'admin-ajax.php' ),
        'homeUrl'     => trailingslashit( home_url() ),
        'wpDebug'     => defined( 'WP_DEBUG' ) && WP_DEBUG, // Critical: Add wpDebug
        'debug'       => defined( 'WP_DEBUG' ) && WP_DEBUG,
        'nonce'       => wp_create_nonce( 'dpg_form_nonce' ),
        'nonces'      => [
            'form'      => wp_create_nonce( 'dpg_form_nonce' ),
            'page_data' => wp_create_nonce( 'dpg_form_nonce' ),
            'ajax'      => wp_create_nonce( 'dpg_form_nonce' ),
        ],
        'strings'     => $this->get_localized_strings(),
        'seoAnalyzer' => $this->get_seo_analyzer_config(),
    ];

    // Primary localization to validation script (loads first)
    if ( wp_script_is( 'dpg-form-validation', 'enqueued' ) || wp_script_is( 'dpg-form-validation', 'registered' ) ) {
        wp_localize_script( 'dpg-form-validation', 'dpgForm', $form_config );
        wp_localize_script( 'dpg-form-validation', 'dpgCreateForm', $form_config );
        
        // Ensure early availability with inline script
        wp_add_inline_script( 'dpg-form-validation', 
            'window.dpgForm = window.dpgForm || ' . wp_json_encode( $form_config ) . ';', 
            'before' 
        );
    }

    // Backup localizations for other scripts
    $script_handles = [ 'dpg-components-js', 'dpg-unified-seo-analyzer' ];
    foreach ( $script_handles as $handle ) {
        if ( wp_script_is( $handle, 'enqueued' ) || wp_script_is( $handle, 'registered' ) ) {
            wp_localize_script( $handle, 'dpgForm', $form_config );
            wp_localize_script( $handle, 'dpgCreateForm', $form_config );
        }
    }
}


    /**
     * WordPress.org compliant localized strings for JavaScript validation.
     * Child classes can override to add specific strings.
     */
protected function get_localized_strings() {
    return [
        // Core validation messages
        'required'                => __( 'This field is required.', 'dpg' ),
        'invalid'                 => __( 'Please enter a valid value.', 'dpg' ),
        'min_length'              => __( 'Minimum length is {0} characters.', 'dpg' ),
        'max_length'              => __( 'Maximum length is {0} characters.', 'dpg' ),
        'validation_error'        => __( 'Validation error occurred.', 'dpg' ),
        'security_error'          => __( 'Security verification failed. Please refresh the page and try again.', 'dpg' ),
        
        // Field-specific validation  
        'name_validation'         => __( 'Template name must be 3-100 characters and contain only letters, numbers, spaces, hyphens, and underscores.', 'dpg' ),
        'slug_validation'         => __( 'URL slug must be lowercase letters, numbers, and hyphens only.', 'dpg' ),
        'template_html_validation' => __( 'Template HTML is required when using custom template.', 'dpg' ),
        'source_page_validation'  => __( 'Please select a page when using existing page as template.', 'dpg' ),
        'items_validation'        => __( 'At least one item is required.', 'dpg' ),
        'email_validation'        => __( 'Please enter a valid email address.', 'dpg' ),
        'url_validation'          => __( 'Please enter a valid URL starting with http:// or https://', 'dpg' ),
        'html_syntax_error'       => __( 'Invalid HTML syntax detected.', 'dpg' ),
        'item_placeholder_required' => __( 'Template must include {item} placeholder for dynamic content.', 'dpg' ),
        'items_required'          => __( 'At least one item is required.', 'dpg' ),
        'item_format_error'       => __( 'Item format should be "name|score" (e.g., "Item Name|0.8").', 'dpg' ),
        'item_name_required'      => __( 'Item name cannot be empty.', 'dpg' ),
        'item_score_range'        => __( 'Item scores must be between 0.1 and 1.0.', 'dpg' ),
        
        // UI messages
        'saving'                  => __( 'Saving...', 'dpg' ),
        'saved'                   => __( 'Saved!', 'dpg' ),
        'error'                   => __( 'An error occurred. Please try again.', 'dpg' ),
        'loading'                 => __( 'Loading...', 'dpg' ),
        'confirm_delete'          => __( 'Are you sure you want to delete this?', 'dpg' ),
        
        // SEO messages
        'seo_tip'                 => __( 'SEO Tip', 'dpg' ),
        'seo_tip_message'         => __( 'For best SEO, ensure your template name and slug are related.', 'dpg' ),
        'seo_example'             => __( 'Example: Template "Best Web Designer" → Slug "web-designer"', 'dpg' ),
    ];
}

    /**
     * WordPress.org compliant SEO analyzer configuration.
     * Child classes can override to provide specific SEO settings.
     */
protected function get_seo_analyzer_config() {
    if ( ! $this->has_seo_features() ) {
        return [
            'options' => [
                'enabled' => false,
                'autoRun' => false,
                'debounceDelay' => 1500
            ]
        ];
    }

    return [
        'options' => [
            'enabled' => true,
            'autoRun' => true,
            'debounceDelay' => 1500
        ],
        'settings' => [
            'minContentLength' => 10,
            'maxContentLength' => 10000,
            'titleLength' => [ 'min' => 30, 'max' => 60 ],
            'descriptionLength' => [ 'min' => 120, 'max' => 160 ]
        ],
        'restApi' => [
            'available' => $this->check_rest_api_availability(),
            'endpoints' => [
                'pages' => rest_url( 'wp/v2/pages/' ),
                'posts' => rest_url( 'wp/v2/posts/' ),
            ],
        ],
        'ajaxUrl' => admin_url( 'admin-ajax.php' ), // Add AJAX URL for fallback
        'fallbacks' => [
            'ajax' => true,
            'serverSide' => true,
            'enhanced' => true
        ]
    ];
}


public function embed_page_data_for_seo( $page_id ) {
        if ( ! $page_id ) {
            return;
        }
        
        $post = get_post( $page_id );
        if ( ! $post ) {
            return;
        }
        
        // Allow draft/private pages for logged-in users with edit capability
        $allowed_statuses = [ 'publish' ];
        if ( current_user_can( 'edit_post', $post->ID ) ) {
            $allowed_statuses = [ 'publish', 'draft', 'private' ];
        }
        
        if ( ! in_array( $post->post_status, $allowed_statuses, true ) ) {
            return;
        }
        
        // Apply content filters to render shortcodes, blocks, etc. - THIS IS KEY!
        $content = apply_filters( 'the_content', $post->post_content );
        
        // Ensure we have the {item} placeholder for template analysis
        if ( false === strpos( $content, '{item}' ) ) {
            // Add {item} placeholder as a comment so analysis works
            $content = "<!-- Dynamic content: {item} -->\n" . $content;
        }
        
        // Create comprehensive page data for JavaScript fallback
        $page_data = array(
            'content' => $content,
            'title' => apply_filters( 'the_title', $post->post_title, $post->ID ),
            'excerpt' => $post->post_excerpt ?: wp_trim_words( wp_strip_all_tags( $content ), 25 ),
            'id' => $post->ID,
            'status' => $post->post_status,
            'content_length' => strlen( $content ),
            'raw_content_length' => strlen( $post->post_content ),
            'has_item_placeholder' => ( false !== strpos( $content, '{item}' ) ),
            'method' => 'server_embedded'
        );
        
        // Embed as JSON in a hidden script tag
        printf(
            '<script type="application/json" id="dpg-page-data-%d">%s</script>',
            $page_id,
            wp_json_encode( $page_data, JSON_UNESCAPED_UNICODE )
        );
        
        // DEBUG: Output for troubleshooting
        if ( defined( 'WP_DEBUG' ) && WP_DEBUG ) {
            printf(
                '<!-- DPG DEBUG: Embedded page %d, content: %d chars, has {item}: %s -->',
                $page_id,
                strlen( $content ),
                $page_data['has_item_placeholder'] ? 'Yes' : 'No'
            );
        }
    }

private function check_rest_api_availability() {
    // Check if REST API is disabled
    if ( ! function_exists( 'rest_get_url_prefix' ) ) {
        return false;
    }

    // Check if REST API endpoints are accessible
    $rest_url = rest_url( 'wp/v2/' );
    if ( ! $rest_url ) {
        return false;
    }

    // Additional checks for common REST API blocking plugins
    if ( defined( 'DISABLE_WP_JSON' ) && DISABLE_WP_JSON ) {
        return false;
    }

    return true;
}

    /**
     * Check if this form has SEO features.
     * Child classes should override this.
     */
    protected function has_seo_features() {
        return false;
    }

    /**
     * WordPress.org compliant form start rendering.
     *
     * @param string|array $attributes Extra attributes for the <form> tag.
     */
protected function render_form_start( $attributes = [] ) {
    // Handle both string and array inputs for backward compatibility
    if ( is_string( $attributes ) ) {
        $attributes = [ 'action' => $attributes ];
    }

    $defaults = [
        'id' => esc_attr( $this->form_id ),
        'class' => 'dpg-form',
        'method' => 'post',
        'action' => esc_url( admin_url( 'admin-post.php' ) ),
        'novalidate' => 'novalidate', // Use our custom validation
    ];
    $attrs = wp_parse_args( $attributes, $defaults );

    echo '<form';
    foreach ( $attrs as $key => $val ) {
        if ( null !== $val && '' !== $val ) {
            printf( ' %s="%s"', esc_attr( $key ), esc_attr( $val ) );
        }
    }
    echo '>';

    // WordPress action handler
    if ( $this->action ) {
        printf(
            '<input type="hidden" name="action" value="%s">',
            esc_attr( $this->action )
        );
    }

    // Enhanced nonce fields - create multiple for compatibility
    wp_nonce_field( 'dpg_form_nonce', '_wpnonce' );
    if ( $this->action ) {
        wp_nonce_field( $this->action, 'dpg_nonce' );
    }
    if ( $this->form_id && $this->form_id !== $this->action ) {
        wp_nonce_field( $this->form_id, $this->form_id . '_nonce' );
    }
}

    /**
     * WordPress.org compliant form end rendering.
     */
    protected function render_form_end() {
        echo '</form>';
    }

    /**
     * WordPress.org compliant nonce verification.
     *
     * @return void Will wp_die() on failure.
     */
protected function verify_nonce() {
    // Check both nonce fields for compatibility
    $nonce_verified = false;
    
    if ( isset( $_POST['_wpnonce'] ) && wp_verify_nonce( $_POST['_wpnonce'], 'dpg_form_nonce' ) ) {
        $nonce_verified = true;
    }
    
    if ( isset( $_POST[ $this->form_id . '_nonce' ] ) && wp_verify_nonce( $_POST[ $this->form_id . '_nonce' ], $this->form_id ) ) {
        $nonce_verified = true;
    }
    
    // Additional check for action-specific nonces
    if ( $this->action && isset( $_POST['dpg_nonce'] ) && wp_verify_nonce( $_POST['dpg_nonce'], $this->action ) ) {
        $nonce_verified = true;
    }
    
    if ( ! $nonce_verified ) {
        wp_die( 
            __( 'Security verification failed. Please refresh the page and try again.', 'dpg' ), 
            __( 'Security Error', 'dpg' ),
            [ 'response' => 403 ] 
        );
    }
}

    /**
     * WordPress.org compliant section rendering.
     *
     * @param string   $title
     * @param callable $callback  Print the fields inside.
     * @param string   $desc
     * @param bool     $collapsible
     */
    protected function render_section( $title, $callback, $desc = '', $collapsible = false ) {
        $section_id = sanitize_title( $title );
        $collapsed_class = $collapsible ? ' dpg-section-collapsible' : '';
        $collapsed_state = $collapsible ? ' dpg-section-collapsed' : '';

        printf(
            '<div class="dpg-form-section%s%s" data-section="%s">',
            esc_attr( $collapsed_class ),
            esc_attr( $collapsed_state ),
            esc_attr( $section_id )
        );

        // header
        printf(
            '<div class="dpg-section-header"><h3 class="dpg-section-title">%s</h3>',
            esc_html( $title )
        );
        if ( $collapsible ) {
            echo '<button type="button" class="dpg-section-toggle" aria-expanded="false">'
               . '<span class="dashicons dashicons-arrow-down-alt2"></span>'
               . '</button>';
        }
        echo '</div>'; // .dpg-section-header

        echo '<div class="dpg-section-content"' . ( $collapsible ? ' style="display:none;"' : '' ) . '>';
        if ( $desc ) {
            printf(
                '<p class="dpg-section-description">%s</p>',
                wp_kses_post( $desc )
            );
        }
        if ( is_callable( $callback ) ) {
            call_user_func( $callback );
        }
        echo '</div>'; // .dpg-section-content

        echo '</div>'; // .dpg-form-section
    }

    /**
     * WordPress.org compliant field rendering.
     *
     * @param array $args {
     *   @type string $type        text|textarea|select|checkbox|radio|editor|number|email|url|hidden
     *   @type string $id
     *   @type string $name
     *   @type string $label
     *   @type mixed  $value
     *   @type string $placeholder
     *   @type string $description
     *   @type bool   $required
     *   @type string $class
     *   @type array  $attributes  Additional html attrs.
     *   @type array  $options     For select/radio.
     *   @type int    $rows        textarea rows.
     *   @type int    $cols        textarea cols.
     * }
     */
    protected function render_field( $args ) {
        $defaults = [
            'type' => 'text',
            'id' => '',
            'name' => '',
            'label' => '',
            'value' => '',
            'placeholder' => '',
            'description' => '',
            'required' => false,
            'class' => '',
            'attributes' => [],
            'options' => [],
            'rows' => 4,
            'cols' => 50,
        ];
        $field = wp_parse_args( $args, $defaults );
        $fid = $field['id'] ?: $field['name'];
        $req = $field['required'] ? ' required' : '';
        $mark = $field['required'] ? ' <span class="dpg-required">*</span>' : '';

        // Don't wrap hidden fields
        if ( 'hidden' === $field['type'] ) {
            $this->render_input_field( $fid, $field, $req );
            return;
        }

        echo '<div class="dpg-field-wrapper dpg-field-' . esc_attr( $field['type'] ) . '">';

        if ( $field['label'] ) {
            printf(
                '<label for="%s" class="dpg-field-label">%s%s</label>',
                esc_attr( $fid ),
                esc_html( $field['label'] ),
                $mark
            );
        }

        echo '<div class="dpg-field-container">';

        switch ( $field['type'] ) {
            case 'textarea':
                $this->render_textarea_field( $fid, $field, $req );
                break;
            case 'select':
                $this->render_select_field( $fid, $field, $req );
                break;
            case 'checkbox':
                $this->render_checkbox_field( $fid, $field );
                break;
            case 'radio':
                $this->render_radio_field( $fid, $field, $req );
                break;
            case 'editor':
                $this->render_editor_field( $fid, $field );
                break;
            default:
                $this->render_input_field( $fid, $field, $req );
                break;
        }

        if ( $field['description'] ) {
            printf(
                '<p class="dpg-field-description">%s</p>',
                wp_kses_post( $field['description'] )
            );
        }

        echo '</div>'; // .dpg-field-container
        echo '</div>'; // .dpg-field-wrapper
    }

    /*------------------------------------------------------------
     *  WORDPRESS.ORG COMPLIANT FIELD RENDERERS
     *------------------------------------------------------------*/

    private function render_input_field( $fid, $f, $req ) {
        $attrs = wp_parse_args( $f['attributes'], [
            'type' => $f['type'],
            'id' => $fid,
            'name' => $f['name'],
            'value' => $f['value'],
            'class' => 'dpg-input ' . $f['class'],
            'placeholder' => $f['placeholder'],
        ]);

        // Remove empty attributes except for value (which can be empty string)
        $attrs = array_filter( $attrs, function( $v, $k ) {
            return '' !== $v || 'value' === $k;
        }, ARRAY_FILTER_USE_BOTH );

        echo '<input';
        foreach ( $attrs as $k => $v ) {
            printf( ' %s="%s"', esc_attr( $k ), esc_attr( $v ) );
        }
        echo $req . '>';
    }

    private function render_textarea_field( $fid, $f, $req ) {
        $attrs = wp_parse_args( $f['attributes'], [
            'id' => $fid,
            'name' => $f['name'],
            'class' => 'dpg-textarea ' . $f['class'],
            'rows' => intval( $f['rows'] ),
            'cols' => intval( $f['cols'] ),
            'placeholder' => $f['placeholder'],
        ]);

        // Remove empty attributes
        $attrs = array_filter( $attrs, function( $v ) {
            return '' !== $v && 0 !== $v;
        });

        echo '<textarea';
        foreach ( $attrs as $k => $v ) {
            if ( 'value' !== $k ) {
                printf( ' %s="%s"', esc_attr( $k ), esc_attr( $v ) );
            }
        }
        echo $req . '>';
        echo esc_textarea( $f['value'] );
        echo '</textarea>';
    }

    private function render_select_field( $fid, $f, $req ) {
        $attrs = wp_parse_args( $f['attributes'], [
            'id' => $fid,
            'name' => $f['name'],
            'class' => 'dpg-select ' . $f['class'],
        ]);

        // Remove empty attributes
        $attrs = array_filter( $attrs, function( $v ) {
            return '' !== $v;
        });

        echo '<select';
        foreach ( $attrs as $k => $v ) {
            printf( ' %s="%s"', esc_attr( $k ), esc_attr( $v ) );
        }
        echo $req . '>';

        foreach ( $f['options'] as $value => $label ) {
            printf(
                '<option value="%s"%s>%s</option>',
                esc_attr( $value ),
                selected( $f['value'], $value, false ),
                esc_html( $label )
            );
        }

        echo '</select>';
    }

    private function render_checkbox_field( $fid, $f ) {
        $checked = checked( $f['value'], 1, false );
        $attrs = wp_parse_args( $f['attributes'], [
            'class' => 'dpg-checkbox ' . $f['class'],
        ]);

        // Build attributes string
        $attr_str = '';
        foreach ( $attrs as $k => $v ) {
            if ( '' !== $v ) {
                $attr_str .= sprintf( ' %s="%s"', esc_attr( $k ), esc_attr( $v ) );
            }
        }

        printf(
            '<label class="dpg-checkbox-label"><input type="checkbox" id="%1$s" name="%2$s" value="1"%3$s%4$s><span class="dpg-checkbox-text">%5$s</span></label>',
            esc_attr( $fid ),
            esc_attr( $f['name'] ),
            $checked,
            $attr_str,
            esc_html( $f['label'] )
        );
    }

    private function render_radio_field( $fid, $f, $req ) {
        echo '<div class="dpg-radio-group">';
        foreach ( $f['options'] as $value => $label ) {
            $id = esc_attr( "{$f['name']}_{$value}" );
            $sel = checked( $f['value'], $value, false );
            $attrs = wp_parse_args( $f['attributes'], [
                'class' => 'dpg-radio ' . $f['class'],
            ]);

            // Build attributes string
            $attr_str = '';
            foreach ( $attrs as $k => $v ) {
                if ( '' !== $v ) {
                    $attr_str .= sprintf( ' %s="%s"', esc_attr( $k ), esc_attr( $v ) );
                }
            }

            printf(
                '<label class="dpg-radio-label"><input type="radio" id="%1$s" name="%2$s" value="%3$s"%4$s%5$s%6$s><span class="dpg-radio-text">%7$s</span></label>',
                $id,
                esc_attr( $f['name'] ),
                esc_attr( $value ),
                $sel,
                $req,
                $attr_str,
                esc_html( $label )
            );
        }
        echo '</div>';
    }

    private function render_editor_field( $fid, $f ) {
        $attrs = wp_parse_args( $f['attributes'], [
            'class' => 'dpg-editor ' . $f['class'],
        ]);

        printf(
            '<div class="dpg-editor-wrapper"><textarea id="%1$s" name="%2$s" class="%3$s" rows="%4$d">%5$s</textarea></div>',
            esc_attr( $fid ),
            esc_attr( $f['name'] ),
            esc_attr( $attrs['class'] ),
            intval( $f['rows'] ),
            esc_textarea( $f['value'] )
        );
    }

    /**
     * WordPress.org compliant submit button.
     */
    protected function render_submit_button( $text = '', $type = 'primary', $atts = [] ) {
        $text = $text ?: __( 'Save Changes', 'dpg' );
        $defaults = [
            'type' => 'submit',
            'class' => 'dpg-button dpg-button-' . sanitize_key( $type ),
        ];
        $attrs = wp_parse_args( $atts, $defaults );

        echo '<button';
        foreach ( $attrs as $k => $v ) {
            if ( '' !== $v ) {
                printf( ' %s="%s"', esc_attr( $k ), esc_attr( $v ) );
            }
        }
        echo '>' . esc_html( $text ) . '</button>';
    }

    /**
     * WordPress.org compliant notice rendering.
     *
     * @param string $message
     * @param string $type     info|success|warning|error
     * @param bool   $dismissible
     */
protected function render_notice( $message, $type = 'info', $dismissible = true ) {
    $type = sanitize_key( $type );
    $classes = [ 'dpg-notice', 'dpg-notice-' . $type ];
    if ( $dismissible ) {
        $classes[] = 'dpg-notice-dismissible';
    }
    
    $icons = [
        'info' => 'ℹ️',
        'success' => '✅',
        'warning' => '⚠️', 
        'error' => '❌'
    ];
    
    $icon = isset( $icons[ $type ] ) ? $icons[ $type ] : $icons['info'];
    
    printf( '<div class="%s">', esc_attr( implode( ' ', $classes ) ) );
    printf( '<div class="dpg-notice-content">' );
    printf( '<div class="dpg-notice-icon">%s</div>', $icon );
    printf( '<div class="dpg-notice-message">%s</div>', wp_kses_post( $message ) );
    if ( $dismissible ) {
        echo '<button type="button" class="dpg-notice-dismiss">&times;</button>';
    }
    printf( '</div>' );
    echo '</div>';
}

    /**
     * WordPress.org compliant validation - override in children.
     *
     * @param array $data
     * @return true|array
     */
public function validate( $data ) {
    $errors = [];
    return empty( $errors ) ? true : $errors;
}

    /**
     * WordPress.org compliant processing - override in children.
     *
     * @param array $data
     * @return bool
     */
    public function process( $data = null ) {
        return false;
    }

    protected function validate_template_content( $data ) {
    $errors = [];
    $template_source = $data['dpg_template_source'] ?? 'custom';

    /* ── Custom-HTML branch ───────────────────────────── */
    if ( 'custom' === $template_source ) {
        $html = trim( $data['dpg_template_html'] ?? '' );

        if ( $html === '' ) {
            $errors[] = __( 'Template HTML is required when using custom template.', 'dpg' );
        } elseif ( stripos( $html, '{item}' ) === false ) {
            $errors[] = __( 'Template HTML must contain {item} placeholder.', 'dpg' );
        }
        return $errors;
    }

    /* ── Existing-page branch ─────────────────────────── */
    $page_id = absint( $data['dpg_source_page_id'] ?? 0 );
    if ( ! $page_id ) {
        $errors[] = __( 'Please select a page when using an existing page as template.', 'dpg' );
        return $errors;
    }

    $page = get_post( $page_id );
    if ( ! $page || $page->post_status !== 'publish' ) {
        $errors[] = __( 'Selected page is not published.', 'dpg' );
        return $errors;
    }

    // Check if page is already assigned to another template
    $skip_ctx = $data['dpg_ctx_id'] ?? null;
    if ( $other = $this->get_page_assigned_template( $page_id, $skip_ctx ) ) {
        $errors[] = sprintf(
            __( 'This page is already used by the "%s" template. Delete or re-assign that template first.', 'dpg' ),
            esc_html( $other )
        );
    }

    return $errors;
}


protected function get_page_assigned_template( $page_id, $skip_ctx_id = null ) {
    global $wp_filesystem;
    if ( empty( $wp_filesystem ) ) {
        require_once ABSPATH . 'wp-admin/includes/file.php';
        WP_Filesystem();
    }

    $dir = trailingslashit( DPG_DATA_DIR );
    $files = glob( $dir . '*.json' );
    
    if ( ! $files ) {
        return false;
    }

    foreach ( $files as $file ) {
        $ctx = basename( $file, '.json' );
        if ( $skip_ctx_id && $ctx === $skip_ctx_id ) {
            continue;
        }
        
        $raw = $wp_filesystem->get_contents( $file );
        if ( false === $raw ) {
            continue;
        }
        
        $data = json_decode( $raw, true );
        if ( isset( $data['source_page_id'] ) && absint( $data['source_page_id'] ) === $page_id ) {
            return $ctx;
        }
    }

    return false;
}

protected function get_processed_template_content( $data ) {
    $template_source = $data['dpg_template_source'] ?? 'custom';
    
    if ( 'custom' === $template_source ) {
        // Process custom HTML template
        $html = wp_unslash( $data['dpg_template_html'] ?? '' );
        
        if ( empty( trim( $html ) ) ) {
            return '';
        }
        
        $processed_html = wp_kses_post( $html );
        
        // Ensure {item} placeholder exists for custom templates
        if ( false === strpos( $processed_html, '{item}' ) ) {
            $processed_html = "<!-- Dynamic content: {item} -->\n" . $processed_html;
        }
        
        return $processed_html;
        
    } elseif ( 'page' === $template_source ) {
        $page_id = absint( $data['dpg_source_page_id'] ?? 0 );
        
        if ( ! $page_id ) {
            return '';
        }
        
        $post = get_post( $page_id );
        if ( ! $post || 'publish' !== $post->post_status ) {
            return '';
        }
        
        // Get page content with filters applied
        return apply_filters( 'the_content', $post->post_content );
    }
    
    return '';
}

    /**
     * WordPress.org compliant form data collection.
     *
     * @param array $defaults
     * @return array
     */
protected function get_form_data( $defaults = [] ) {
    $data = [];
    foreach ( $defaults as $key => $def ) {
        if ( isset( $_POST[ $key ] ) ) {
            $value = wp_unslash( $_POST[ $key ] );
            
            // Handle different data types properly
            if ( is_numeric( $def ) ) {
                $data[ $key ] = is_numeric( $value ) ? ( is_float( $def ) ? floatval( $value ) : intval( $value ) ) : $def;
            } elseif ( is_bool( $def ) ) {
                $data[ $key ] = ! empty( $value );
            } elseif ( is_array( $def ) ) {
                $data[ $key ] = is_array( $value ) ? $value : ( empty( $value ) ? [] : [ $value ] );
            } else {
                $data[ $key ] = $value;
            }
        } else {
            $data[ $key ] = $def;
        }
    }
    
    // SPECIAL HANDLING: Ensure template_source and related fields work together
    if ( isset( $data['dpg_template_source'] ) ) {
        if ( 'page' === $data['dpg_template_source'] ) {
            // When using page source, clear template_html if it's just default content
            if ( isset( $data['dpg_template_html'] ) && 
                 ( empty( $data['dpg_template_html'] ) || 
                   strpos( $data['dpg_template_html'], '<div class="template-container">' ) !== false ) ) {
                $data['dpg_template_html'] = '';
            }
        } elseif ( 'custom' === $data['dpg_template_source'] ) {
            // When using custom source, clear page ID
            $data['dpg_source_page_id'] = '';
        }
    }
    
    return $data;
}

    /**
     * WordPress.org compliant data sanitization.
     *
     * @param array $data
     * @param array $rules [ 'field_name' => 'text'|'textarea'|... ]
     * @return array
     */
protected function sanitize_data( $data, $rules = [] ) {
    $out = [];

    foreach ( $data as $key => $val ) {
        // Determine which rule to apply for this field
        $rule = isset( $rules[ $key ] ) ? $rules[ $key ] : 'text';

        // Handle explicit nulls
        if ( is_null( $val ) ) {
            $out[ $key ] = '';
            continue;
        }

        switch ( $rule ) {
            case 'textarea':
                $out[ $key ] = sanitize_textarea_field( $val );
                break;

            case 'email':
                $out[ $key ] = sanitize_email( $val );
                break;

            case 'url':
                $out[ $key ] = esc_url_raw( $val );
                break;

            case 'html':
            case 'template_html':
                // Use wp_kses_post but allow basic HTML for templates
                if ( empty( trim( $val ) ) ) {
                    $out[ $key ] = '';
                } else {
                    $out[ $key ] = wp_kses_post( $val );
                    
                    // Ensure {item} placeholder is preserved during sanitization
                    if ( false === strpos( $out[ $key ], '{item}' ) && false !== strpos( $val, '{item}' ) ) {
                        // If sanitization removed {item}, add it back as a comment
                        $out[ $key ] = "<!-- Dynamic content: {item} -->\n" . $out[ $key ];
                    }
                }
                break;

            case 'number':
                $out[ $key ] = is_numeric( $val ) ? floatval( $val ) : 0;
                break;

            case 'integer':
                $out[ $key ] = is_numeric( $val ) ? intval( $val ) : 0;
                break;

            case 'boolean':
                $out[ $key ] = ! empty( $val );
                break;

            case 'array':
                $out[ $key ] = is_array( $val )
                    ? array_map( 'sanitize_text_field', $val )
                    : [];
                break;

            case 'key':
                $out[ $key ] = sanitize_key( $val );
                break;

            case 'title':
                $out[ $key ] = sanitize_title( $val );
                break;

            default:
                // text, etc.
                $out[ $key ] = sanitize_text_field( $val );
                break;
        }
    }

    return $out;
}

    /**
     * WordPress.org compliant form header rendering.
     */
    protected function render_form_header( $title, $desc = '' ) {
        printf( '<div class="dpg-form-header"><h2 class="dpg-form-title">%s</h2>',
            esc_html( $title )
        );
        if ( $desc ) {
            printf( '<p class="dpg-form-description">%s</p>',
                wp_kses_post( $desc )
            );
        }
        echo '</div>';
    }

    /**
     * WordPress.org compliant shorthand to open form and include nonces.
     */
    protected function start_form( $override_id = '' ) {
        $attrs = [ 'class' => 'dpg-form' ]; // Ensure dpg-form class is always present
        if ( $override_id ) {
            // Handle both "id class" and just "id" formats
            if ( strpos( $override_id, ' ' ) !== false ) {
                $parts = explode( ' ', $override_id, 2 );
                $attrs['id'] = sanitize_key( $parts[0] );
                $attrs['class'] = esc_attr( $parts[1] );
            } else {
                $attrs['id'] = sanitize_key( $override_id );
            }
        }
        $this->render_form_start( $attrs );
    }

    /**
     * WordPress.org compliant shorthand to close form.
     */
    protected function end_form( $submit_text = '' ) {
        echo '<div class="dpg-form-actions">';
        $this->render_submit_button( $submit_text );
        echo '</div>';
        $this->render_form_end();
    }
}