<?php
// File: wp-content/plugins/dynamic-seo-pages/includes/forms/class-dpg-form-create.php

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

require_once DPG_PLUGIN_DIR . 'includes/forms/class-dpg-form-base.php';

/**
 * WordPress.org Compliant Create Template Form Class
 * Handles the creation of new templates with SEO analysis
 */
class DPG_Form_Create extends DPG_Form_Base {

    /** @var DPG_SEO_Analyzer */
    private $seo_analyzer;

    /**
     * Constructor.
     *
     * @param array $config Configuration passed from DPG_Admin.
     */
public function __construct( $config = [] ) {
    $this->config = $config;
    parent::__construct( 'create-template', '', 'dpg_create_template' );
    
    // CRITICAL: Register AJAX handlers early
    $this->register_ajax_handlers();
    
    // Register WordPress action handlers
    add_action( 'admin_post_dpg_create_template', [ $this, 'handle_submission' ] );
    
    // Ensure scripts are enqueued when form is rendered
    add_action( 'admin_enqueue_scripts', [ $this, 'enqueue_assets' ] );
}

public function register_ajax_handlers() {
    // Register the actual SEO analysis handler (missing from your code)
    add_action( 'wp_ajax_dpg_seo_analysis', [ $this, 'ajax_seo_analysis' ] );
    add_action( 'wp_ajax_nopriv_dpg_seo_analysis', [ $this, 'ajax_seo_analysis' ] );
    
    // Keep existing page data handler
    add_action( 'wp_ajax_dpg_get_page_data', [ $this, 'ajax_get_page_data' ] );
    add_action( 'wp_ajax_nopriv_dpg_get_page_data', [ $this, 'ajax_get_page_data' ] );
}

    /**
     * WordPress asset enqueuing.
     */
public function enqueue_assets() {
    $base = plugin_dir_url( DPG_PLUGIN_FILE ) . 'includes/forms/assets/';

    /* 1. Base styles first */
    wp_enqueue_style(
        'dpg-admin-forms',
        $base . 'css/admin-forms.css',
        [],
        DPG_VERSION
    );

    wp_enqueue_style(
        'dpg-form-components',
        $base . 'css/components.css',
        [ 'dpg-admin-forms' ],
        DPG_VERSION
    );

    /* 2. CodeMirror for HTML editor */
    wp_enqueue_code_editor( [ 'type' => 'text/html' ] );
    wp_enqueue_script( 'wp-theme-plugin-editor' );
    wp_enqueue_style( 'wp-codemirror' );

    /* 3. CRITICAL: Enqueue scripts in proper dependency order */
    
    // Validation script (foundation)
    wp_enqueue_script(
        'dpg-form-validation',
        $base . 'js/form-validation.js',
        [ 'jquery' ],
        DPG_VERSION,
        true
    );

    // Components script (depends on validation)
    wp_enqueue_script(
        'dpg-components-js',
        $base . 'js/components.js',
        [ 'jquery', 'dpg-form-validation' ],
        DPG_VERSION,
        true
    );

    // SEO analyzer (depends on components)
    wp_enqueue_script(
        'dpg-unified-seo-analyzer',
        $base . 'js/seo-analyzer.js',
        [ 'jquery', 'wp-i18n', 'dpg-components-js', 'dpg-form-validation' ],
        DPG_VERSION,
        true
    );

    /* 4. FIXED: Single, comprehensive localization */
    $this->localize_scripts_properly();
}

/**
 * FIXED: Proper script localization that ensures dpgForm is available
 */
private function localize_scripts_properly() {
    $config = [
        'ajaxUrl'  => admin_url( 'admin-ajax.php' ),
        'homeUrl'  => trailingslashit( home_url() ),
        'wpDebug'  => defined( 'WP_DEBUG' ) && WP_DEBUG, // Add wpDebug flag
        'debug'    => defined( 'WP_DEBUG' ) && WP_DEBUG,
        'nonce'    => wp_create_nonce( 'dpg_form_nonce' ),
        'nonces'   => [
            'form'      => wp_create_nonce( 'dpg_form_nonce' ),
            'page_data' => wp_create_nonce( 'dpg_form_nonce' ),
            'ajax'      => wp_create_nonce( 'dpg_form_nonce' ),
        ],
        'strings'  => [
            'loading'             => __( 'Loading...', 'dpg' ),
            'saving'              => __( 'Saving...', 'dpg' ),
            'saved'               => __( 'Saved!', 'dpg' ),
            'error'               => __( 'An error occurred. Please try again.', 'dpg' ),
            'security_error'      => __( 'Security verification failed. Please refresh the page and try again.', 'dpg' ),
            'required'            => __( 'This field is required.', 'dpg' ),
            'invalid'             => __( 'Please enter a valid value.', 'dpg' ),
            'min_length'          => __( 'Minimum length is {0} characters.', 'dpg' ),
            'max_length'          => __( 'Maximum length is {0} characters.', 'dpg' ),
            'name_validation'     => __( 'Template name must be 3-100 characters and contain only letters, numbers, spaces, hyphens, and underscores.', 'dpg' ),
            'slug_validation'     => __( 'URL slug must be lowercase letters, numbers, and hyphens only.', 'dpg' ),
            'template_html_validation' => __( 'Template HTML is required when using custom template.', 'dpg' ),
            'source_page_validation' => __( 'Please select a page when using existing page as template.', 'dpg' ),
            'items_validation'    => __( 'At least one item is required.', 'dpg' ),
            'html_syntax_error'   => __( 'Invalid HTML syntax detected.', 'dpg' ),
            'item_placeholder_required' => __( 'Template must include {item} placeholder for dynamic content.', 'dpg' ),
            'seo_tip'            => __( 'SEO Tip', 'dpg' ),
            'seo_tip_message'    => __( 'For best SEO, ensure your template name and slug are related.', 'dpg' ),
            'seo_example'        => __( 'Example: Template "Best Web Designer" → Slug "web-designer"', 'dpg' ),
        ],
        'seoAnalyzer' => [
            'options' => [
                'enabled' => true,
                'autoRun' => true,
                'debounceDelay' => 1500
            ],
            'settings' => [
                'minContentLength' => 10,
                'maxContentLength' => 10000,
                'titleLength' => [ 'min' => 30, 'max' => 60 ],
                'descriptionLength' => [ 'min' => 120, 'max' => 160 ]
            ]
        ]
    ];

    // Localize to validation script first (it loads first)
    wp_localize_script( 'dpg-form-validation', 'dpgForm', $config );
    
    // Also provide for compatibility
    wp_localize_script( 'dpg-form-validation', 'dpgCreateForm', $config );
    
    // Add inline script to ensure global availability
    wp_add_inline_script( 'dpg-form-validation', 
        'window.dpgForm = window.dpgForm || ' . wp_json_encode( $config ) . ';', 
        'before' 
    );
}

 private function get_current_source_page_id() {
        // For create form, check POST data
        if ( isset( $_POST['dpg_source_page_id'] ) && is_numeric( $_POST['dpg_source_page_id'] ) ) {
            return intval( $_POST['dpg_source_page_id'] );
        }
        
        // Check if we have a default or pre-selected page
        if ( isset( $_GET['source_page'] ) && is_numeric( $_GET['source_page'] ) ) {
            return intval( $_GET['source_page'] );
        }
        
        return 0;
    }

    /**
     * Enable SEO features for this form.
     *
     * @return bool
     */
    protected function has_seo_features() {
        return true;
    }


        public function render() {
        $this->enqueue_assets();

        echo '<div class="dpg-create-template-wrapper">';
        $this->render_form_header(
            __( 'Create New Template', 'dpg' ),
            __( 'Generate multiple SEO-optimized pages automatically.', 'dpg' )
        );

        // WordPress form with proper nonces
        $this->start_form( 'dpg-create-template-form' );
        
        // WordPress nonce fields (both for compatibility)
        wp_nonce_field( $this->action, 'dpg_nonce' );
        wp_nonce_field( 'dpg_form_nonce', '_wpnonce' );

        // WordPress security fields
        printf(
            '<input type="hidden" name="dpg_user_id" value="%d">',
            esc_attr( get_current_user_id() )
        );
        printf(
            '<input type="hidden" name="dpg_timestamp" value="%d">',
            esc_attr( time() )
        );

        // Display WordPress notices
        if ( ! empty( $this->errors ) ) {
            $this->render_notice(
                wp_kses_post( implode( '<br>', array_map( 'esc_html', $this->errors ) ) ),
                'error'
            );
        }

        // ROW 1: Basic Configuration + SEO Settings (Two Columns)
        $this->render_section(
            __( 'Template Configuration & SEO', 'dpg' ),
            [ $this, 'render_row_one_fields' ],
        );

        // ROW 2: Template Content (Full Width)
        $this->render_section(
            __( 'Template Content', 'dpg' ),
            [ $this, 'render_content_fields' ],
            __( 'Write your HTML here. Use <code>{item}</code> for dynamic content.', 'dpg' )
        );

        // ROW 3: SEO Analysis (Full Width)
        $this->render_section(
            __( 'SEO Analysis', 'dpg' ),
            [ $this, 'render_seo_analysis' ],
        );

        // ROW 4: Social Media (Open by default)
        $this->render_section(
            __( 'Social Media', 'dpg' ),
            [ $this, 'render_social_fields' ],
            __( 'Configure sharing metadata.', 'dpg' ),
            false // Not collapsible - always open
        );

        // ROW 5: Template Options (Open by default)
        $this->render_section(
            __( 'Template Options', 'dpg' ),
            [ $this, 'render_options_fields' ],
            __( 'Additional generation settings.', 'dpg' ),
            false // Not collapsible - always open
        );

        // ENHANCED: Embed page data for SEO analysis fallback with debugging
        $source_page_id = $this->get_current_source_page_id();
        if ( $source_page_id ) {
            $this->debug_page_content( $source_page_id ); // Debug output
            $this->embed_page_data_for_seo( $source_page_id ); // Embed page data
        }

        // WordPress form close
        $this->end_form( __( 'Create Template', 'dpg' ) );
        echo '</div>';
    }
    /**
     * WordPress.org compliant form rendering.
     */
/**
 * WordPress-compliant AJAX handler for page data
 */
public function ajax_get_page_data() {
    // Nonce verification
    if ( ! check_ajax_referer( 'dpg_form_nonce', 'nonce', false ) ) {
        wp_send_json_error( __( 'Security verification failed.', 'dpg' ), 403 );
    }

    // Capability check
    if ( ! current_user_can( 'manage_options' ) ) {
        wp_send_json_error( __( 'Permission denied.', 'dpg' ), 403 );
    }
    
    // Validate and sanitize input
    $page_id = isset( $_GET['id'] ) ? absint( $_GET['id'] ) : 0;
    if ( ! $page_id ) {
        wp_send_json_error( __( 'Invalid page ID.', 'dpg' ), 400 );
    }
    
    // Use the comprehensive function from your main plugin file
    if ( function_exists( 'dpg_ajax_get_comprehensive_page_data' ) ) {
        dpg_ajax_get_comprehensive_page_data();
    } else {
        wp_send_json_error( __( 'AJAX handler not available.', 'dpg' ), 500 );
    }
}

public function embed_page_data_for_seo( $page_id ) {
    $post = get_post( $page_id );
    if ( ! $post || 'publish' !== $post->post_status ) {
        return;
    }
    
    // Get page content for SEO analysis
    $page_content = apply_filters( 'the_content', $post->post_content );
    
    // Embed page data as JSON for JavaScript SEO analyzer
    $page_data = [
        'id' => $page_id,
        'title' => get_the_title( $page_id ),
        'content' => wp_strip_all_tags( $page_content ),
        'excerpt' => get_the_excerpt( $page_id ),
        'url' => get_permalink( $page_id ),
    ];
    
    printf(
        '<script type="application/json" id="dpg-page-data">%s</script>',
        wp_json_encode( $page_data )
    );
}

      
protected function debug_page_content( $page_id ) {
    // Only show debug info if WordPress debug is enabled AND user has proper capabilities
    if ( ! defined( 'WP_DEBUG' ) || ! WP_DEBUG || ! current_user_can( 'manage_options' ) ) {
        return;
    }
    
    $post = get_post( $page_id );
    if ( ! $post ) {
        printf( '<!-- DEBUG: Page %d not found -->', absint( $page_id ) );
        return;
    }
    
    $raw_content = $post->post_content;
    $processed_content = apply_filters( 'the_content', $raw_content );
    
    printf( 
        '<!-- DEBUG PAGE ANALYSIS:
        Page ID: %d
        Title: %s
        Status: %s
        Raw content length: %d
        Processed content length: %d
        Raw has {item}: %s
        Processed has {item}: %s
        First 200 chars of processed: %s
        -->',
        absint( $page_id ),
        esc_html( $post->post_title ),
        esc_html( $post->post_status ),
        strlen( $raw_content ),
        strlen( $processed_content ),
        ( false !== strpos( $raw_content, '{item}' ) ) ? 'Yes' : 'No',
        ( false !== strpos( $processed_content, '{item}' ) ) ? 'Yes' : 'No',
        esc_html( substr( wp_strip_all_tags( $processed_content ), 0, 200 ) )
    );
}

    /**
     * Render Row 1: Basic Configuration + SEO Settings (Two Columns)
     */
    public function render_row_one_fields() {
        echo '<div class="dpg-row-layout dpg-row-layout--two-col">';
        
        // LEFT COLUMN: Basic Configuration + Items
        echo '<div class="dpg-column">';
            echo '<h4 class="dpg-column-title">' . esc_html__( 'Configure basic settings for your template.', 'dpg' ) . '</h4>';
            
            // Template Name - FIRST as requested (80% width)
            $this->render_field( [
                'type'        => 'text',
                'id'          => 'dpg_name',
                'name'        => 'dpg_name',
                'label'       => __( 'Template Name', 'dpg' ),
                'required'    => true,
                'placeholder' => __( 'e.g., Best Web Designers', 'dpg' ),
                'description' => __( 'A descriptive name for this template.', 'dpg' ),
                'attributes'  => [ 
                    'class' => 'dpg-input--80',
                    'data-validate-field' => 'dpg_name'
                ],
                'value'       => '', // Explicitly set empty to prevent validation on load
            ] );

            // Template Type - SECOND as requested (80% width)
            $this->render_field( [
                'type'        => 'select',
                'id'          => 'dpg_type',
                'name'        => 'dpg_type',
                'label'       => __( 'Template Type', 'dpg' ),
                'required'    => true,
                'options'     => [
                    'service-area' => __( 'Service or Product Area', 'dpg' ),
                    'keyword'      => __( 'Keyword Based',       'dpg' ),
                ],
                'value'       => 'service-area',
                'description' => __( 'URLs like "plumbing-chicago" or "best-plumber".', 'dpg' ),
                'attributes'  => [ 'class' => 'dpg-input--80' ],
            ] );

            // URL Slug - THIRD as requested (80% width)
            echo '<div id="dpg-slug-field">';
                $this->render_field( [
                    'type'        => 'text',
                    'id'          => 'dpg_slug',
                    'name'        => 'dpg_slug',
                    'label'       => __( 'URL Slug', 'dpg' ),
                    'required'    => true,
                    'placeholder' => __( 'e.g., web-designer', 'dpg' ),
                    'description' => __( 'Used in URLs like "web-designer-chicago".', 'dpg' ),
                    'attributes'  => [ 
                        'class' => 'dpg-input--80',
                        'data-validate-field' => 'dpg_slug'
                    ],
                    'value'       => '', // Explicitly set empty
                ] );
                echo '<div id="dpg-url-preview" style="display:none;">';
                    echo '<div class="dpg-url-preview">';
                        echo '<label class="dpg-url-preview-label">' . esc_html__( 'URL Example', 'dpg' ) . '</label>';
                        echo '<div id="url-preview" class="dpg-url-example"></div>';
                        echo '<div id="seo-warning" class="dpg-seo-warning" style="display:none;"></div>';
                    echo '</div>';
                echo '</div>';
            echo '</div>';

            // Items List
            echo '<h5 class="dpg-subsection-title">' . esc_html__( 'Items List', 'dpg' ) . '</h5>';
            echo '<p class="dpg-subsection-description">' . esc_html__( 'One item per line; optional scores via |.', 'dpg' ) . '</p>';
            $this->render_field( [
                'type'        => 'textarea',
                'id'          => 'dpg_items',
                'name'        => 'dpg_items',
                'label'       => __( 'Items (one per line)', 'dpg' ),
                'required'    => true,
                'placeholder' => __( "New York City\nLos Angeles\nChicago\nHouston", 'dpg' ),
                'description' => __( 'Each line generates a page; add scores via "|".', 'dpg' ),
                'rows'        => 8,
                'attributes'  => [ 'data-validate-field' => 'dpg_items' ],
                'value'       => '', // Explicitly set empty
            ] );
        echo '</div>';
        
        // RIGHT COLUMN: SEO Settings
        echo '<div class="dpg-column">';
            echo '<h4 class="dpg-column-title">' . esc_html__( 'Optimize your template for search engines.', 'dpg' ) . '</h4>';
            
            // Focus Keyphrase (80% width)
            $this->render_field( [
                'type'        => 'text',
                'id'          => 'dpg_focus_keyphrase',
                'name'        => 'dpg_focus_keyphrase',
                'label'       => __( 'Focus Keyphrase', 'dpg' ),
                'placeholder' => __( 'e.g., web designer', 'dpg' ),
                'description' => __( 'Main phrase for SEO analysis.', 'dpg' ),
                'attributes'  => [ 'class' => 'dpg-input--80' ],
                'value'       => '',
            ] );

            // SEO Title (80% width)
            $this->render_field( [
                'type'        => 'text',
                'id'          => 'dpg_meta_title',
                'name'        => 'dpg_meta_title',
                'label'       => __( 'SEO Title', 'dpg' ),
                'placeholder' => __( 'Best {item} – Professional Services', 'dpg' ),
                'description' => __( 'Appears in search results.', 'dpg' ),
                'attributes'  => [
                    'maxlength' => '60',
                    'class' => 'dpg-input--80'
                ],
                'value'       => '',
            ] );
            echo '<div class="dpg-field-help">';
                echo '<span id="title-counter">60</span> ' . esc_html__( 'characters remaining', 'dpg' );
            echo '</div>';

            // Meta Description
            $this->render_field( [
                'type'        => 'textarea',
                'id'          => 'dpg_meta_description',
                'name'        => 'dpg_meta_description',
                'label'       => __( 'Meta Description', 'dpg' ),
                'placeholder' => __( 'Brief summary with {item}.', 'dpg' ),
                'description' => __( 'Appears in search results.', 'dpg' ),
                'rows'        => 3,
                'attributes'  => [
                    'maxlength' => '160'
                ],
                'value'       => '',
            ] );
            echo '<div class="dpg-field-help">';
                echo '<span id="description-counter">160</span> ' . esc_html__( 'characters remaining', 'dpg' );
            echo '</div>';

            // Meta Keywords
            $this->render_field( [
                'type'        => 'textarea',
                'id'          => 'dpg_meta_keywords',
                'name'        => 'dpg_meta_keywords',
                'label'       => __( 'Meta Keywords', 'dpg' ),
                'placeholder' => __( '{item}, services, local', 'dpg' ),
                'description' => __( 'Comma-separated keywords.', 'dpg' ),
                'rows'        => 2,
                'value'       => '',
            ] );
        echo '</div>';
        
        echo '</div>'; // End dpg-row-layout
    }

    /**
     * Render content fields
     */

public function render_content_fields() {
    /* Current state (after a failed submit) or sensible defaults */
    $template_source = isset( $_POST['dpg_template_source'] )
        ? sanitize_key( $_POST['dpg_template_source'] )
        : 'custom';

    $source_page_id = isset( $_POST['dpg_source_page_id'] )
        ? absint( $_POST['dpg_source_page_id'] )
        : 0;

    /* Template-source selector */
    echo '<div class="dpg-field-grid dpg-field-grid--mixed">';
        echo '<div>';
            $this->render_field( [
                'type'       => 'select',
                'id'         => 'dpg_template_source',
                'name'       => 'dpg_template_source',
                'label'      => __( 'Template Source', 'dpg' ),
                'required'   => true,
                'options'    => [
                    'custom' => __( 'Custom HTML', 'dpg' ),
                    'page'   => __( 'Existing Page', 'dpg' ),
                ],
                'value'      => $template_source,
                'attributes' => [ 'class' => 'dpg-input--medium' ],
                'description'=> __( 'Choose where the HTML should come from.', 'dpg' ),
            ] );
        echo '</div><div></div>';
    echo '</div>';

    /* Page dropdown – custom HTML so we can inject data-in-use */
    $show_page_sel = ( $template_source === 'page' ) ? 'block' : 'none';

    echo '<div id="dpg-page-selection"
               class="dpg-field-grid dpg-field-grid--mixed"
               style="display:' . esc_attr( $show_page_sel ) . ';">';

        echo '<div>';

            echo '<label for="dpg_source_page_id" class="dpg-field-label">'
                   . esc_html__( 'Select Page', 'dpg' ) .
                 '</label>';

            echo '<select id="dpg_source_page_id"
                           name="dpg_source_page_id"
                           class="dpg-select dpg-input--medium"
                           data-validate-field="dpg_source_page_id">';

                /* first blank/placeholder row */
                printf(
                    '<option value="">%s</option>',
                    esc_html__( 'Select a page…', 'dpg' )
                );

                /* every real page carries the extra attribute */
                $pages = get_pages( [
                    'post_status' => 'publish',
                    'number'      => 200,
                    'sort_column' => 'post_title',
                ] );

                foreach ( $pages as $p ) {
                    // FIXED: Add the second parameter (null for create form)
                    $in_use = $this->get_page_assigned_template( $p->ID, null ) ? 1 : 0;

                    printf(
                        '<option value="%1$d" data-in-use="%3$d"%4$s>%2$s</option>',
                        $p->ID,
                        esc_html( $p->post_title . " (ID: {$p->ID})" ),
                        $in_use,
                        selected( $p->ID, $source_page_id, false )
                    );
                }

            echo '</select>';

            /* NEW inline banner */
            echo '<div id="dpg-page-in-use-alert"
                    class="dpg-notice dpg-notice-error dpg-page-alert"
                    style="display:none;">'
            . esc_html__(
                'You can\'t use this page – it is already assigned to another template. ' .
                'Delete that template, or re-assign it to a different page, before you can use this one.',
                'dpg'
            )
            . '</div>';

            /* help text, now re-worded */
            echo '<p class="dpg-field-description">'
            . esc_html__(
                'Select a different page, or free this page up by editing/deleting the template that currently uses it.',
                'dpg'
            )
            . '</p>';

        echo '</div><div></div>';
    echo '</div>'; // #dpg-page-selection

    /* Custom-HTML textarea (only when source = custom) */
    $show_html = ( $template_source === 'custom' ) ? 'block' : 'none';

    echo '<div id="dpg-custom-template"
               class="dpg-field dpg-field-textarea dpg-field-required"
               style="display:' . esc_attr( $show_html ) . ';">';

        echo '<label for="dpg_template_html" class="dpg-field-label">'
             . esc_html__( 'Template HTML', 'dpg' ) .
             ' <span class="dpg-required">*</span></label>';

        echo '<div class="dpg-field-input">';
            $starter = isset( $_POST['dpg_template_html'] )
                ? wp_unslash( $_POST['dpg_template_html'] )
                : '<h1>{item}</h1>';   // very small default

            echo '<textarea name="dpg_template_html" id="dpg_template_html"
                            rows="15"
                            class="dpg-input dpg-textarea dpg-code-editor"
                            data-validate-field="dpg_template_html">'
                 . esc_textarea( $starter ) .
                 '</textarea>';

            echo '<p class="dpg-field-description">'
                   . wp_kses_post( __( 'Use <code>{item}</code> where the item name should be inserted.', 'dpg' ) ) .
                 '</p>';
        echo '</div>';

    echo '</div>'; // #dpg-custom-template
}




protected function get_page_assigned_template( $page_id, $skip_ctx_id = null ) {
    // Get all existing templates to check for conflicts
    $data_dir = DPG_DATA_DIR;
    if ( ! is_dir( $data_dir ) ) {
        return false;
    }
    
    $json_files = glob( $data_dir . '*.json' );
    if ( ! $json_files ) {
        return false;
    }
    
    global $wp_filesystem;
    if ( ! function_exists( 'WP_Filesystem' ) ) {
        require_once ABSPATH . 'wp-admin/includes/file.php';
    }
    WP_Filesystem();
    
    foreach ( $json_files as $file ) {
        if ( ! $wp_filesystem->exists( $file ) ) {
            continue;
        }
        
        $content = $wp_filesystem->get_contents( $file );
        if ( false === $content ) {
            continue;
        }
        
        $template_data = json_decode( $content, true );
        if ( ! is_array( $template_data ) ) {
            continue;
        }
        
        // Check if this template uses the same page
        if ( isset( $template_data['source_page_id'] ) && 
             absint( $template_data['source_page_id'] ) === absint( $page_id ) ) {
            
            // NEW: Check if we should skip this context (for edit form)
            $ctx = basename( $file, '.json' );
            if ( $skip_ctx_id && $ctx === $skip_ctx_id ) {
                continue;
            }
            
            return $template_data['name'] ?? basename( $file, '.json' );
        }
    }
    
    return false;
}

// ADD: Method to get processed template content (missing from free version)
protected function get_processed_template_content( $data ) {
    if ( ! empty( $data['dpg_template_source'] )
      && $data['dpg_template_source'] === 'page'
      && ! empty( $data['dpg_source_page_id'] )
    ) {
        $post = get_post( absint( $data['dpg_source_page_id'] ) );
        if ( $post ) {
            return apply_filters( 'the_content', $post->post_content );
        }
    }

    return $data['dpg_template_html'] ?? '';
}

// ADD: Enhanced debug method for template content
protected function debug_template_content( $data, $html ) {
    if ( ! defined( 'WP_DEBUG' ) || ! WP_DEBUG || ! current_user_can( 'manage_options' ) ) {
        return;
    }
    
    printf( 
        '<!-- DEBUG TEMPLATE CONTENT:
        Template Source: %s
        Source Page ID: %s
        HTML Length: %d
        Has {item} placeholder: %s
        First 200 chars: %s
        -->',
        esc_html( $data['dpg_template_source'] ?? 'unknown' ),
        esc_html( $data['dpg_source_page_id'] ?? 'none' ),
        strlen( $html ),
        ( false !== strpos( $html, '{item}' ) ) ? 'Yes' : 'No',
        esc_html( substr( wp_strip_all_tags( $html ), 0, 200 ) )
    );
}


/**
 * Render the SEO-analysis panel shown in the Create-Template screen.
 * – Shows Problems / Improvements / Good-results
 * – Injects a hidden field (#dpg_seo_score) that the JS keeps updated
 *   so the PHP handler can read the final score on submit.
 */

 /**
 * NEW: Missing SEO Analysis AJAX Handler
 * This is what the JavaScript is trying to call
 */
public function ajax_seo_analysis() {
    // Nonce verification (multiple methods for compatibility)
    $nonce_verified = false;
    
    if ( isset( $_POST['security'] ) && wp_verify_nonce( $_POST['security'], 'dpg_form_nonce' ) ) {
        $nonce_verified = true;
    }
    
    if ( isset( $_POST['nonce'] ) && wp_verify_nonce( $_POST['nonce'], 'dpg_form_nonce' ) ) {
        $nonce_verified = true;
    }
    
    if ( ! $nonce_verified ) {
        wp_send_json_error( __( 'Security verification failed.', 'dpg' ), 403 );
    }

    // Capability check
    if ( ! current_user_can( 'manage_options' ) ) {
        wp_send_json_error( __( 'Permission denied.', 'dpg' ), 403 );
    }
    
    // Get and sanitize input
    $template_data = wp_unslash( $_POST['template_data'] ?? [] );
    $focus_keyphrase = sanitize_text_field( wp_unslash( $_POST['focus_keyphrase'] ?? '' ) );
    $sample_item = sanitize_text_field( wp_unslash( $_POST['sample_item'] ?? 'Sample Item' ) );
    
    // Basic validation
    if ( empty( $template_data ) || ! is_array( $template_data ) ) {
        wp_send_json_error( __( 'Invalid template data.', 'dpg' ), 400 );
    }
    
    // Simple SEO analysis (you can enhance this)
    $analysis_result = $this->perform_seo_analysis( $template_data, $focus_keyphrase, $sample_item );
    
    wp_send_json_success( $analysis_result );
}

/**
 * Basic SEO Analysis Implementation
 * Replace with your actual SEO analyzer logic
 */
private function perform_seo_analysis( $template_data, $focus_keyphrase, $sample_item ) {
    $problems = [];
    $improvements = [];
    $good_results = [];
    $score = 50; // Default score
    
    // Get template content
    $content = '';
    if ( 'custom' === ( $template_data['template_source'] ?? 'custom' ) ) {
        $content = $template_data['template_html'] ?? '';
    } else {
        $content = '[Page content will be analyzed when page is selected]';
    }
    
    // Replace {item} with sample item for analysis
    $analyzed_content = str_replace( '{item}', $sample_item, $content );
    
    // Basic checks
    if ( empty( trim( $content ) ) ) {
        $problems[] = [
            'type' => 'content',
            'message' => 'Template content is empty',
            'how_to_fix' => 'Add content to your template'
        ];
    } else {
        $good_results[] = [
            'type' => 'content',
            'message' => 'Template has content'
        ];
        $score += 20;
    }
    
    if ( strpos( $content, '{item}' ) !== false ) {
        $good_results[] = [
            'type' => 'placeholder',
            'message' => 'Template includes {item} placeholder'
        ];
        $score += 15;
    } else {
        $problems[] = [
            'type' => 'placeholder',
            'message' => 'Template missing {item} placeholder',
            'how_to_fix' => 'Add {item} where dynamic content should appear'
        ];
    }
    
    // Meta title check
    $meta_title = $template_data['hard_meta']['title'] ?? '';
    if ( ! empty( $meta_title ) ) {
        $title_length = strlen( $meta_title );
        if ( $title_length >= 30 && $title_length <= 60 ) {
            $good_results[] = [
                'type' => 'title',
                'message' => 'Meta title length is optimal (' . $title_length . ' characters)'
            ];
            $score += 10;
        } else {
            $improvements[] = [
                'type' => 'title',
                'message' => 'Meta title length should be 30-60 characters (currently ' . $title_length . ')',
                'how_to_fix' => 'Adjust meta title length'
            ];
        }
    } else {
        $improvements[] = [
            'type' => 'title',
            'message' => 'Add a meta title for better SEO',
            'how_to_fix' => 'Fill in the SEO Title field'
        ];
    }
    
    // Focus keyphrase check
    if ( ! empty( $focus_keyphrase ) ) {
        $good_results[] = [
            'type' => 'keyphrase',
            'message' => 'Focus keyphrase is set: ' . $focus_keyphrase
        ];
        $score += 10;
        
        // Check if keyphrase appears in content
        if ( stripos( $analyzed_content, $focus_keyphrase ) !== false ) {
            $good_results[] = [
                'type' => 'keyphrase_content',
                'message' => 'Focus keyphrase appears in content'
            ];
            $score += 5;
        } else {
            $improvements[] = [
                'type' => 'keyphrase_content',
                'message' => 'Focus keyphrase should appear in template content',
                'how_to_fix' => 'Include "' . $focus_keyphrase . '" in your template HTML'
            ];
        }
    } else {
        $improvements[] = [
            'type' => 'keyphrase',
            'message' => 'Set a focus keyphrase for SEO optimization',
            'how_to_fix' => 'Add a focus keyphrase in the SEO section'
        ];
    }
    
    // Determine status based on score
    $status = 'poor';
    if ( $score >= 80 ) {
        $status = 'good';
    } elseif ( $score >= 60 ) {
        $status = 'ok';
    }
    
    return [
        'percentage' => min( 100, max( 0, $score ) ),
        'status' => $status,
        'problems' => $problems,
        'improvements' => $improvements,
        'good_results' => $good_results,
        'analysis_date' => current_time( 'mysql' ),
        'method' => 'basic_php_analysis'
    ];
}


public function render_seo_analysis() {
	echo '<div class="dpg-seo-analysis">';

		/* ── header with round badge ───────────────────────────── */
		echo '<div class="dpg-seo-header">';
			echo '<h3>' . esc_html__( 'SEO Analysis', 'dpg' ) . '</h3>';
			echo '<div class="dpg-seo-score">';
				echo '<div class="dpg-seo-score-circle">';
					echo '<div id="seo-score-number" class="dpg-seo-score-number">0</div>';
					echo '<div class="dpg-seo-score-label">' . esc_html__( 'SEO Score', 'dpg' ) . '</div>';
				echo '</div>';
			echo '</div>';
		echo '</div>';

		/* ── results: problems (full width) + 2-col grid ──────── */
		echo '<div class="dpg-seo-results">';

			/* Problems */
			echo '<div id="seo-problems" class="dpg-seo-section dpg-seo-problems" style="display:none;">';
				echo '<h4 class="dpg-seo-section-title">🔴 ' . esc_html__( 'Problems', 'dpg' ) . '</h4>';
				echo '<div id="seo-problems-list" class="dpg-seo-items"></div>';
			echo '</div>';

			/* two-column wrapper */
			echo '<div class="dpg-seo-two-column">';

				/* Improvements (left) */
				echo '<div class="dpg-seo-column">';
					echo '<div id="seo-improvements" class="dpg-seo-section dpg-seo-improvements" style="display:none;">';
						echo '<h4 class="dpg-seo-section-title">🟡 ' . esc_html__( 'Improvements', 'dpg' ) . '</h4>';
						echo '<div id="seo-improvements-list" class="dpg-seo-items"></div>';
					echo '</div>';
				echo '</div>';

				/* Good results (right) */
				echo '<div class="dpg-seo-column">';
					echo '<div id="seo-good" class="dpg-seo-section dpg-seo-good" style="display:none;">';
						echo '<h4 class="dpg-seo-section-title">🟢 ' . esc_html__( 'Good Results', 'dpg' ) . '</h4>';
						echo '<div id="seo-good-list" class="dpg-seo-items"></div>';
					echo '</div>';
				echo '</div>';

			echo '</div>'; // .dpg-seo-two-column

			/* helper banner shown when no analysis yet */
			echo '<div id="seo-help" class="dpg-seo-help">';
				echo '<p>' . esc_html__( 'Fill in the template content above to see SEO analysis. The analysis will update as you type.', 'dpg' ) . '</p>';
			echo '</div>';

		echo '</div>'; // .dpg-seo-results

		/* ── NEW: keep the latest score for PHP ───────────────── */
		echo '<input type="hidden" id="dpg_seo_score" name="dpg_seo_score" value="0">';

	echo '</div>'; // .dpg-seo-analysis
}

    /**
     * Render social media fields - Social Title, Social Image URL, Social Description in same row
     */
    public function render_social_fields() {
        // First row: Social Title, Social Image URL, Social Description
        echo '<div class="dpg-field-grid dpg-field-grid--three-col">';
            echo '<div>';
                $this->render_field( [
                    'type'        => 'text',
                    'id'          => 'dpg_og_title',
                    'name'        => 'dpg_og_title',
                    'label'       => __( 'Social Title', 'dpg' ),
                    'description' => __( 'Use {item} placeholder.', 'dpg' ),
                    'value'       => '',
                ] );
            echo '</div>';
            
            echo '<div>';
                $this->render_field( [
                    'type'        => 'url',
                    'id'          => 'dpg_og_image',
                    'name'        => 'dpg_og_image',
                    'label'       => __( 'Social Image URL', 'dpg' ),
                    'description' => __( '1200×630px recommended.', 'dpg' ),
                    'value'       => '',
                ] );
            echo '</div>';
            
            echo '<div>';
                $this->render_field( [
                    'type'        => 'textarea',
                    'id'          => 'dpg_og_description',
                    'name'        => 'dpg_og_description',
                    'label'       => __( 'Social Description', 'dpg' ),
                    'rows'        => 3,
                    'description' => __( 'Use {item} placeholder.', 'dpg' ),
                    'value'       => '',
                ] );
            echo '</div>';
        echo '</div>';

        // Second row: Twitter fields
        echo '<div class="dpg-field-grid dpg-field-grid--three-col">';
            echo '<div>';
                $this->render_field( [
                    'type'        => 'text',
                    'id'          => 'dpg_tw_title',
                    'name'        => 'dpg_tw_title',
                    'label'       => __( 'Twitter Title', 'dpg' ),
                    'description' => __( 'Optional; falls back to Social Title.', 'dpg' ),
                    'value'       => '',
                ] );
            echo '</div>';
            
            echo '<div>';
                $this->render_field( [
                    'type'        => 'url',
                    'id'          => 'dpg_tw_image',
                    'name'        => 'dpg_tw_image',
                    'label'       => __( 'Twitter Image URL', 'dpg' ),
                    'description' => __( 'Optional; falls back to Social Image.', 'dpg' ),
                    'value'       => '',
                ] );
            echo '</div>';
            
            echo '<div>';
                $this->render_field( [
                    'type'        => 'textarea',
                    'id'          => 'dpg_tw_description',
                    'name'        => 'dpg_tw_description',
                    'label'       => __( 'Twitter Description', 'dpg' ),
                    'rows'        => 2,
                    'description' => __( 'Optional; falls back to Social Description.', 'dpg' ),
                    'value'       => '',
                ] );
            echo '</div>';
        echo '</div>';
    }

    /**
     * Render template options
     */
    public function render_options_fields() {
        echo '<div class="dpg-field-grid dpg-field-grid--two-col">';
            // Header
            echo '<div>';
                echo '<label class="dpg-checkbox-label">';
                    echo '<input type="checkbox" id="dpg_include_header" name="dpg_include_header" value="1" class="dpg-checkbox" checked>';
                    echo '<span class="dpg-checkbox-text">' . esc_html__( 'Include Website Header', 'dpg' ) . '</span>';
                echo '</label>';
                echo '<p class="dpg-field-description">' . esc_html__( 'Include theme header on generated pages.', 'dpg' ) . '</p>';
            echo '</div>';
            // Footer
            echo '<div>';
                echo '<label class="dpg-checkbox-label">';
                    echo '<input type="checkbox" id="dpg_include_footer" name="dpg_include_footer" value="1" class="dpg-checkbox" checked>';
                    echo '<span class="dpg-checkbox-text">' . esc_html__( 'Include Website Footer', 'dpg' ) . '</span>';
                echo '</label>';
                echo '<p class="dpg-field-description">' . esc_html__( 'Include theme footer on generated pages.', 'dpg' ) . '</p>';
            echo '</div>';
        echo '</div>';
    }

/**
 * Handle “Create Template” form submission.
 * – Security + capability checks
 * – Saves template JSON and stub PHP file
 * – ★ NEW ★  stores the live SEO-score in DB + JSON
 */
public function handle_submission() {

	/* ──────────────────────────────
	 * 0. Security / capability
	 * ──────────────────────────── */
	$nonce_ok = (
		isset( $_POST['dpg_nonce'] )   && wp_verify_nonce( $_POST['dpg_nonce'], $this->action )
	) || (
		isset( $_POST['_wpnonce'] )    && wp_verify_nonce( $_POST['_wpnonce'], 'dpg_form_nonce' )
	);

	if ( ! $nonce_ok ) {
		wp_die( __( 'Security verification failed.', 'dpg' ), 403 );
	}
	if ( ! current_user_can( 'manage_options' ) ) {
		wp_die( __( 'Permission denied.', 'dpg' ) );
	}

	/* ──────────────────────────────
	 * 1. Collect + sanitise data
	 * ──────────────────────────── */
	$data = $this->get_form_data( [
		'dpg_user_id'          => 0,
		'dpg_timestamp'        => 0,
		'dpg_type'             => '',
		'dpg_name'             => '',
		'dpg_slug'             => '',
		'dpg_template_source'  => 'custom',
		'dpg_source_page_id'   => '',
		'dpg_template_html'    => '',
		'dpg_items'            => '',
		'dpg_focus_keyphrase'  => '',
		'dpg_meta_title'       => '',
		'dpg_meta_description' => '',
		'dpg_meta_keywords'    => '',
		'dpg_og_title'         => '',
		'dpg_og_description'   => '',
		'dpg_og_image'         => '',
		'dpg_tw_title'         => '',
		'dpg_tw_description'   => '',
		'dpg_tw_image'         => '',
		'dpg_include_header'   => '',
		'dpg_include_footer'   => '',
		'dpg_seo_score'        => 0,          // ★ NEW ★  0-100 from JS
	] );

	$score = max( 0, min( 100, intval( $data['dpg_seo_score'] ) ) );   // normalise

	/* basic user / timestamp check */
	if ( absint( $data['dpg_user_id'] ) !== get_current_user_id()
	     || time() - absint( $data['dpg_timestamp'] ) > 300 ) {
		wp_die( __( 'Form expired or invalid user.', 'dpg' ) );
	}

	/* ──────────────────────────────
	 * 2. Validation – early bail
	 * ──────────────────────────── */
	$this->errors = $this->validate( $data );
	if ( ! empty( $this->errors ) ) {
		return $this->render();
	}

	/* ──────────────────────────────
	 * 3. Normalised variables
	 * ──────────────────────────── */
	$type  = sanitize_key( $data['dpg_type'] );          // service-area | keyword
	$name  = sanitize_text_field( $data['dpg_name'] );
	$slug  = sanitize_title( $data['dpg_slug'] );

	$items = array_filter(
		array_map( 'trim', explode( "\n", wp_unslash( $data['dpg_items'] ) ) )
	);

	/* fetch HTML (custom vs. page) */
	if ( 'custom' === $data['dpg_template_source'] ) {
		$html = wp_kses_post( $data['dpg_template_html'] );
	} else {
		$page = get_post( absint( $data['dpg_source_page_id'] ) );
		if ( ! $page || 'publish' !== $page->post_status ) {
			wp_die( __( 'Invalid template source page.', 'dpg' ) );
		}
		$html = $page->post_content;
	}

	/* build context-ID */
	$ctx_id = ( 'service-area' === $type )
		? $slug . '-area'
		: 'keyword-' . sanitize_title( $name );

	/* ──────────────────────────────
	 * 4. Database bookkeeping
	 * ──────────────────────────── */
	$type_key = ( 'service-area' === $type ) ? 'area' : 'keyword';

	/* 4-a register / fetch numeric ID in option dpg_contexts */
	$contexts = get_option( 'dpg_contexts', [] );
	$match    = wp_list_filter( $contexts, [ 'ctx_id' => $ctx_id ] );

	if ( $match ) {
		$numeric_id = absint( array_shift( $match )['id'] );
	} else {
		$numeric_id = empty( $contexts ) ? 1 : ( max( array_column( $contexts, 'id' ) ) + 1 );
		$contexts[] = [
			'id'      => $numeric_id,
			'ctx_id'  => $ctx_id,
			'name'    => $name,
			'created' => current_time( 'mysql' ),
		];
		update_option( 'dpg_contexts', $contexts, false );
	}

	/* 4-b create table & store items */
	if ( class_exists( 'DPG_DB' ) ) {
		DPG_DB::create_table( $type_key, $numeric_id );
		DPG_DB::save_items ( $type_key, $numeric_id, $items );

		/* 4-c ★ NEW ★: persist score in templates table */
		if ( method_exists( 'DPG_DB', 'update_template_column' ) ) {
			DPG_DB::update_template_column( $numeric_id, 'seo_score', $score );
		}

		/* ★ ALSO mirror into the global score table */
		if ( method_exists( 'DPG_DB', 'save_score' ) ) {
			DPG_DB::save_score( $ctx_id, $score );
		}
	}

	/* ──────────────────────────────
	 * 5. Build JSON payload
	 * ──────────────────────────── */
	$parsed_items = [];
	foreach ( $items as $line ) {
		if ( false !== strpos( $line, '|' ) ) {
			list( $label, $prio ) = array_map( 'trim', explode( '|', $line, 2 ) );
			$parsed_items[] = [
				'name'  => sanitize_text_field( $label ),
				'score' => is_numeric( $prio )
					? max( 0.1, min( 1.0, floatval( $prio ) ) )
					: null,
			];
		} else {
			$parsed_items[] = sanitize_text_field( $line );
		}
	}

	$data_arr = [
		'name'            => $name,
		'type'            => $type,
		'template_source' => $data['dpg_template_source'],
		'source_page_id'  => absint( $data['dpg_source_page_id'] ),
		'focus_keyphrase' => sanitize_text_field( $data['dpg_focus_keyphrase'] ),
		'created_date'    => current_time( 'mysql' ),
		'template_html'   => $html,
		'items'           => $parsed_items,
		'include_header'  => ! empty( $data['dpg_include_header'] ),
		'include_footer'  => ! empty( $data['dpg_include_footer'] ),
		'seo_score'       => $score,   // ★ NEW ★
		'hard_meta'       => [
			'title'               => sanitize_text_field( $data['dpg_meta_title'] ),
			'description'         => sanitize_textarea_field( $data['dpg_meta_description'] ),
			'keywords'            => sanitize_textarea_field( $data['dpg_meta_keywords'] ),
			'og_title'            => sanitize_text_field( $data['dpg_og_title'] ),
			'og_description'      => sanitize_textarea_field( $data['dpg_og_description'] ),
			'og_image'            => esc_url_raw( $data['dpg_og_image'] ),
			'twitter_title'       => sanitize_text_field( $data['dpg_tw_title'] ),
			'twitter_description' => sanitize_textarea_field( $data['dpg_tw_description'] ),
			'twitter_image'       => esc_url_raw( $data['dpg_tw_image'] ),
		],
	];

	/* ──────────────────────────────
	 * 6. Save JSON + stub PHP file
	 * ──────────────────────────── */
	global $wp_filesystem;
	if ( empty( $wp_filesystem ) ) {
		require_once ABSPATH . 'wp-admin/includes/file.php';
		WP_Filesystem();
	}
	$json_path = trailingslashit( DPG_DATA_DIR ) . $ctx_id . '.json';
	$php_path  = trailingslashit( DPG_TMPL_DIR ) . $ctx_id . '.php';

	if ( $wp_filesystem->exists( $json_path ) ) {
		wp_die( __( 'Template with this ID already exists.', 'dpg' ) );
	}

	$json_ok = $wp_filesystem->put_contents(
		$json_path,
		wp_json_encode( $data_arr, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE ),
		FS_CHMOD_FILE
	);

	$stub  = "<?php\n";
	$stub .= "/**\n * Template: " . esc_html( $name ) . "\n * Context: " . esc_html( $ctx_id ) . "\n */\n\n";
	$stub .= "if ( ! defined( 'ABSPATH' ) || ! isset( \$dpg_current_item ) ) { status_header( 404 ); exit; }\n";
	$stub .= "global \$dpg_current_item, \$dpg_current_slug;\n";

	$php_ok = $wp_filesystem->put_contents( $php_path, $stub, FS_CHMOD_FILE );

	if ( ! $json_ok || ! $php_ok ) {
		wp_die( __( 'Failed to save template files.', 'dpg' ) );
	}

	/* ──────────────────────────────
	 * 7. Redirect & flush rules
	 * ──────────────────────────── */
	flush_rewrite_rules();
	wp_safe_redirect(
		add_query_arg( 'created', '1', admin_url( 'admin.php?page=dpg-dashboard' ) )
	);
	exit;
}




    /**
     * WordPress form validation.
     */
    public function validate( $data ) {
        $errors = [];

        if ( empty( $data['dpg_type'] ) ) {
            $errors[] = __( 'Template type is required.', 'dpg' );
        }
        if ( empty( $data['dpg_name'] ) ) {
            $errors[] = __( 'Template name is required.', 'dpg' );
        }
        if ( 'service-area' === ( $data['dpg_type'] ?? '' ) && empty( $data['dpg_slug'] ) ) {
            $errors[] = __( 'URL slug is required for Service Area templates.', 'dpg' );
        }

        $src = $data['dpg_template_source'] ?? 'custom';
        if ( 'custom' === $src && empty( $data['dpg_template_html'] ) ) {
            $errors[] = __( 'Template HTML is required for custom templates.', 'dpg' );
        } elseif ( 'page' === $src ) {
            if ( empty( $data['dpg_source_page_id'] ) ) {
                $errors[] = __( 'Please select a page template.', 'dpg' );
            } else {
                $page = get_post( absint( $data['dpg_source_page_id'] ) );
                if ( ! $page || 'publish' !== $page->post_status ) {
                    $errors[] = __( 'Selected page is not valid or not published.', 'dpg' );
                }
                // Check if page is already used
                if ( function_exists( 'dpg_is_page_template_source' )
                    && dpg_is_page_template_source( absint( $data['dpg_source_page_id'] ) )
                ) {
                    $errors[] = __( 'That page is already used by another template.', 'dpg' );
                }
            }
        }

        if ( empty( $data['dpg_items'] ) ) {
            $errors[] = __( 'Items list is required.', 'dpg' );
        }

        return $errors;
    }

}