<?php
// File: wp-content/plugins/dynamic-seo-pages/includes/forms/class-dpg-form-settings.php

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

require_once DPG_PLUGIN_DIR . 'includes/forms/class-dpg-form-base.php';

/**
 * Settings Form Class
 * Handles plugin settings and configuration
 */
class DPG_Form_Settings extends DPG_Form_Base {

    /**
     * Settings Form Constructor
     *
     * @param array $config Form configuration from admin
     */
    public function __construct( $config = [] ) {
        $this->config = $config;
        parent::__construct( 'settings', '', 'dpg_save_settings' );

        // AJAX hook for clearing cache
        add_action( 'wp_ajax_dpg_clear_cache', [ $this, 'handle_clear_cache_ajax' ] );

        // Admin‐post hook for saving settings
        add_action( 'admin_post_dpg_save_settings', [ $this, 'process' ] );
    }

    /**
     * Handle cache clearing via AJAX
     */
    public function handle_clear_cache_ajax() {
        if ( ! check_ajax_referer( 'dpg_clear_cache', 'nonce', false ) ) {
            wp_send_json_error( __( 'Security verification failed.', 'dynamic-seo-pages'), 403 );
        }
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error( __( 'Permission denied.', 'dynamic-seo-pages'), 403 );
        }

        try {
            $success = $this->clear_plugin_cache();
            if ( $success ) {
                wp_send_json_success( __( 'Cache cleared successfully.', 'dynamic-seo-pages') );
            }
            wp_send_json_error( __( 'Failed to clear cache.', 'dynamic-seo-pages'), 500 );
        } catch ( Exception $e ) {
            error_log( 'DPG Cache Clear Error: ' . $e->getMessage() );
            wp_send_json_error( __( 'An error occurred while clearing cache.', 'dynamic-seo-pages'), 500 );
        }
    }

    /**
     * Clears all plugin transients and cache files
     *
     * @return bool
     */
    private function clear_plugin_cache() {
        global $wpdb;

        // Delete plugin transients
        $wpdb->query( $wpdb->prepare(
            "DELETE FROM {$wpdb->options} WHERE option_name LIKE %s",
            '_transient_dpg_%'
        ) );
        $wpdb->query( $wpdb->prepare(
            "DELETE FROM {$wpdb->options} WHERE option_name LIKE %s",
            '_transient_timeout_dpg_%'
        ) );

        // Delete any cache files
        if ( defined( 'DPG_CACHE_DIR' ) && is_dir( DPG_CACHE_DIR ) ) {
            global $wp_filesystem;
            if ( empty( $wp_filesystem ) ) {
                require_once ABSPATH . 'wp-admin/includes/file.php';
                WP_Filesystem();
            }
            foreach ( glob( DPG_CACHE_DIR . '*.cache' ) as $file ) {
                if ( $wp_filesystem->is_file( $file ) ) {
                    $wp_filesystem->delete( $file );
                }
            }
        }

        return true;
    }

    /**
     * Process settings form submission
     *
     * @param array|null $data Defaults to $_POST if null
     */
public function process( $data = null ) {
    if ( ! is_array( $data ) ) {
        $data = $_POST;
    }

    if ( ! current_user_can( DPG_REQUIRED_CAPABILITY ) ) {
        wp_die( esc_html__( 'Permission denied', 'dynamic-seo-pages') );
    }
    check_admin_referer( 'dpg_save_settings_action', 'dpg_settings_nonce' );

    $submitted_user = absint( $data['dpg_user_id'] ?? 0 );
    $submitted_time = absint( $data['dpg_timestamp'] ?? 0 );
    $current_user   = get_current_user_id();

    if ( $submitted_user !== $current_user ) {
        wp_die( esc_html__( 'Security check failed: User mismatch', 'dynamic-seo-pages') );
    }
    if ( time() - $submitted_time > 300 ) {
        wp_die( esc_html__( 'Security check failed: Form expired', 'dynamic-seo-pages') );
    }

    // Always-on
    update_option( 'dpg_pages_active', true );

    // Cache settings
    $enabled  = ! empty( $data['dpg_cache_enabled'] );
    $duration = isset( $data['dpg_cache_duration'] )
                ? absint( $data['dpg_cache_duration'] )
                : get_option( 'dpg_cache_duration', 3600 );

    update_option( 'dpg_cache_enabled',  $enabled );
    update_option( 'dpg_cache_duration', $duration );

    // ADD THIS: Custom scripts settings
    $custom_scripts = isset( $data['dpg_custom_scripts'] ) 
                    ? wp_kses( $data['dpg_custom_scripts'], [
                        'script' => [
                            'type' => [],
                            'src' => [],
                            'async' => [],
                            'defer' => [],
                            'id' => [],
                            'class' => [],
                        ],
                        'noscript' => [],
                        'meta' => [
                            'name' => [],
                            'content' => [],
                            'property' => [],
                        ],
                        'link' => [
                            'rel' => [],
                            'href' => [],
                            'type' => [],
                        ],
                    ] )
                    : '';

    update_option( 'dpg_custom_scripts', $custom_scripts );

    flush_rewrite_rules();

    wp_safe_redirect( add_query_arg(
        [
            'page'           => 'dpg-dashboard',
            'tab'            => 'settings',
            'settings_saved' => '1',
        ],
        admin_url( 'admin.php' )
    ) );
    exit;
}

    /**
     * Enqueue any JS/CSS needed by this form
     */
public function enqueue_assets() {
        parent::enqueue_assets();

        wp_localize_script( 'dpg-form-validation', 'dpgForm', [
            'ajaxUrl' => admin_url( 'admin-ajax.php' ),
            'nonce'   => wp_create_nonce( 'dpg_form_nonce' ),
            'strings' => [
                'required'            => __( 'This field is required.', 'dynamic-seo-pages'),
                'invalid'             => __( 'Invalid format.', 'dynamic-seo-pages'),
                'saving'              => __( 'Saving...', 'dynamic-seo-pages'),
                'security_error'      => __( 'Security verification failed. Please refresh the page and try again.', 'dynamic-seo-pages'),
                'cache_clear_confirm' => __( 'Are you sure you want to clear all cached pages? This cannot be undone.', 'dynamic-seo-pages'),
                'cache_clearing'      => __( 'Clearing...', 'dynamic-seo-pages'),
                'cache_cleared'       => __( 'Cache Cleared!', 'dynamic-seo-pages'),
                'cache_error'         => __( 'Error – Try Again', 'dynamic-seo-pages'),
                'clear_cache'         => __( 'Clear All Cache', 'dynamic-seo-pages'),
                'exporting'           => __( 'Exporting...', 'dynamic-seo-pages'),
            ],
            'nonces' => [
                'clear_cache' => wp_create_nonce( 'dpg_clear_cache' ),
                'export_migration' => wp_create_nonce( 'dpg_export_migration' ),
            ],
        ] );
    }

    /**
     * Render the settings form
     */
 public function render() {
        $this->enqueue_assets();

        echo '<div class="dpg-form-container">';

        $this->render_form_header(
            __( 'Plugin Settings', 'dynamic-seo-pages'),
            __( 'Configure your Dynamic SEO Pages Generator settings.', 'dynamic-seo-pages')
        );

        if ( isset( $_GET['settings_saved'] ) && '1' === $_GET['settings_saved'] ) {
            echo '<div class="notice notice-success is-dismissible"><p>'
               . esc_html__( 'Settings saved successfully!', 'dynamic-seo-pages')
               . '</p></div>';
        }

        $this->start_form( 'dpg-settings-form dpg-form' );
        wp_nonce_field( 'dpg_save_settings_action', 'dpg_settings_nonce' );

        printf( '<input type="hidden" name="action"        value="%s">', esc_attr( $this->action ) );
        printf( '<input type="hidden" name="dpg_user_id"   value="%d">', esc_attr( get_current_user_id() ) );
        printf( '<input type="hidden" name="dpg_timestamp" value="%d">', esc_attr( time() ) );

        $this->render_section(
            __( 'Plugin Status', 'dynamic-seo-pages'),
            [ $this, 'render_status_fields' ],
            __( 'Current status and configuration of the plugin.', 'dynamic-seo-pages'),
            false
        );

        $this->render_section(
            __( 'Cache & Scripts Settings', 'dynamic-seo-pages'),
            [ $this, 'render_cache_fields' ],
            __( 'Configure caching and custom scripts for standalone templates (without header/footer).', 'dynamic-seo-pages'),
            false
        );

        $this->end_form( __( 'Save Settings', 'dynamic-seo-pages') );

        ?>
        <script>
        jQuery(document).ready(function($) {
            $('#dpg-export-migration').on('click', function() {
                var button = $(this);
                var originalText = button.text();
                
                button.text(dpgForm.strings.exporting).prop('disabled', true);
                
                var form = $('<form>', {
                    method: 'POST',
                    action: dpgForm.ajaxUrl
                });
                
                form.append($('<input>', {
                    type: 'hidden',
                    name: 'action',
                    value: 'dpg_export_for_migration'
                }));
                
                form.append($('<input>', {
                    type: 'hidden',
                    name: 'nonce',
                    value: dpgForm.nonces.export_migration
                }));
                
                $('body').append(form);
                form.submit();
                
                setTimeout(function() {
                    button.text(originalText).prop('disabled', false);
                }, 2000);
            });
        });
        </script>
        <?php

        echo '</div>';
    }

    /**
     * Render status fields
     */
  public function render_status_fields() {
        $pages_active   = get_option( 'dpg_pages_active', false );
        $template_count = dpg_get_user_template_count();
        $total_pages    = dpg_get_generated_pages_count();

        echo '<div class="dpg-status-grid">';

        echo '<div class="dpg-status-item">'
           . '<div class="dpg-status-label">' . esc_html__( 'Dynamic Pages', 'dynamic-seo-pages') . '</div>'
           . '<div class="dpg-status-value">'
             . ( $pages_active
                 ? '<span class="dpg-status-active">✅ ' . esc_html__( 'Active', 'dynamic-seo-pages') . '</span>'
                 : '<span class="dpg-status-inactive">❌ ' . esc_html__( 'Inactive', 'dynamic-seo-pages') . '</span>'
               )
           . '</div>'
           . '<div class="dpg-status-description">'
             . esc_html__( 'Always on—cannot be disabled.', 'dynamic-seo-pages')
           . '</div>'
           . '</div>';

        echo '<div class="dpg-status-item">'
           . '<div class="dpg-status-label">' . esc_html__( 'Templates Created', 'dynamic-seo-pages') . '</div>'
           . '<div class="dpg-status-value"><span class="dpg-status-count">' . esc_html( $template_count ) . '</span></div>'
           . '</div>';

        echo '<div class="dpg-status-item">'
           . '<div class="dpg-status-label">' . esc_html__( 'Generated Pages', 'dynamic-seo-pages') . '</div>'
           . '<div class="dpg-status-value"><span class="dpg-status-count">' . esc_html( $total_pages ) . '</span></div>'
           . '</div>';

        echo '<div class="dpg-status-item">'
           . '<div class="dpg-status-label">' . esc_html__( 'Export for Migration', 'dynamic-seo-pages') . '</div>'
           . '<div class="dpg-status-value">'
           . '<button type="button" id="dpg-export-migration" class="dpg-button dpg-button-small dpg-button-primary">'
           . esc_html__( '📤 Export Data', 'dynamic-seo-pages')
           . '</button>'
           . '</div>'
           . '<div class="dpg-status-description">'
           . esc_html__( 'Export templates and settings for Pro migration.', 'dynamic-seo-pages')
           . '</div>'
           . '</div>';

        echo '</div>';
    }

    /**
     * Render cache settings fields
     */
/**
 * Render cache settings fields
 */
public function render_cache_fields() {
    $cache_enabled  = get_option( 'dpg_cache_enabled', false );
    $cache_duration = get_option( 'dpg_cache_duration', 3600 );
    $custom_scripts = get_option( 'dpg_custom_scripts', '' ); // ADD THIS LINE

    echo '<div class="dpg-alert warning" style="margin-bottom: 20px;">';
    echo '<div class="dpg-alert-content">';
    echo '<div class="dpg-alert-icon">⚠️</div>';
    echo '<div class="dpg-alert-body">';
    echo '<h4 class="dpg-alert-title">' . esc_html__( 'Cache Usage Recommendation', 'dynamic-seo-pages') . '</h4>';
    echo '<p class="dpg-alert-message">' . esc_html__( 'Only enable caching for templates that do not include website header and footer. For templates with header/footer, use a dedicated caching plugin instead.', 'dynamic-seo-pages') . '</p>';
    echo '</div>';
    echo '</div>';
    echo '</div>';

    echo '<div class="dpg-field-grid dpg-field-grid--two-col">';
      echo '<div>';
        echo '<label class="dpg-checkbox-label">';
        echo '<input type="checkbox" class="dpg-checkbox" id="dpg_cache_enabled" name="dpg_cache_enabled" value="1"' . ( $cache_enabled ? ' checked' : '' ) . '>';
        echo '<span class="dpg-checkbox-text">' . esc_html__( 'Enable Page Caching', 'dynamic-seo-pages') . '</span>';
        echo '</label>';
        echo '<p class="dpg-field-description">' . esc_html__( 'Cache standalone template pages (without header/footer).', 'dynamic-seo-pages') . '</p>';
      echo '</div>';

      echo '<div>';
        $this->render_field([
            'type'        => 'number',
            'id'          => 'dpg_cache_duration',
            'name'        => 'dpg_cache_duration',
            'label'       => __( 'Cache Duration (seconds)', 'dynamic-seo-pages'),
            'value'       => $cache_duration,
            'placeholder' => '3600',
            'description' => __( 'How long to cache pages (default: 1 hour).', 'dynamic-seo-pages'),
            'attributes'  => [ 'min' => '300', 'max' => '86400', 'step' => '300' ],
        ]);
      echo '</div>';
    echo '</div>';

    // ADD THIS SECTION FOR CUSTOM SCRIPTS
    echo '<div class="dpg-field-grid dpg-field-grid--single">';
      echo '<div>';
        $this->render_field([
            'type'        => 'textarea',
            'id'          => 'dpg_custom_scripts',
            'name'        => 'dpg_custom_scripts',
            'label'       => __( 'Custom Scripts', 'dynamic-seo-pages'),
            'value'       => $custom_scripts,
            'placeholder' => __( 'Add Google Analytics, Facebook Pixel, or other tracking scripts here...', 'dynamic-seo-pages'),
            'description' => __( 'These scripts will be added to the <head> section of standalone pages (pages without header/footer). Use for tracking codes, analytics, etc.', 'dynamic-seo-pages'),
            'rows'        => 8,
        ]);
      echo '</div>';
    echo '</div>';

    echo '<div class="dpg-field-grid dpg-field-grid--mixed">';
      echo '<div>';
        echo '<button type="button" id="dpg-clear-cache" class="dpg-button dpg-button-secondary">';
        echo esc_html__( 'Clear All Cache', 'dynamic-seo-pages');
        echo '</button>';
        echo '<p class="dpg-field-description">' . esc_html__( 'Immediately clear all cached pages.', 'dynamic-seo-pages') . '</p>';
      echo '</div>';
      echo '<div></div>';
    echo '</div>';
}

    /**
     * No validation needed here
     */
    public function validate_data( $data ) {
        return [];
    }

    /**
     * Disable SEO features for this form
     */
    protected function has_seo_features() {
        return false;
    }
}